﻿using System;
using System.Data;
using System.Data.Common;
using System.Collections.Generic;
using System.Text;

namespace fb2pg
{
    /// <summary>
    /// Esta clase gestiona los generadores asociados a las tablas de la base de datos Firebird
    /// </summary>
    class TablasGeneradores
    {
        #region · Campos ·

        /// <summary>
        /// Nombre del generador
        /// </summary>
        string generador;
        /// <summary>
        /// Nombre del generador
        /// </summary>
        public string Generador
        {
            get { return generador; }
            set { generador = value; }
        }

        /// <summary>
        /// Nombre de la tabla
        /// </summary>
        string tabla;
        /// <summary>
        /// Nombre de la tabla
        /// </summary>
        public string Tabla
        {
            get { return tabla; }
            set { tabla = value; }
        }

        /// <summary>
        /// Nombre del campo que utiliza el generador
        /// </summary>
        string campo;
        /// <summary>
        /// Nombre del campo que utiliza el generador
        /// </summary>
        public string Campo
        {
            get { return campo; }
            set { campo = value; }
        }

        /// <summary>
        /// Valor que tiene el contador actualmente
        /// </summary>
        int valorActual;
        /// <summary>
        /// Valor que tiene el contador actualmente
        /// </summary>
        public int ValorActual
        {
            get { return valorActual; }
            set { valorActual = value; }
        }


        #endregion

        #region · Constructor ·

        /// <summary>
        /// Crea una instancia de la clase <see cref="TablasGeneradores"/>
        /// </summary>
        /// <param name="g">Nombre del generador</param>
        /// <param name="t">Nombre de la tabla</param>
        /// <param name="c">Nombre del campo asociado al generador</param>
        /// <param name="v">Valor actual del generador</param>
        public TablasGeneradores(string g, string t, string c, int v)
        {
            this.tabla = t.Trim().ToLowerInvariant();
            this.generador = g.Trim().ToLowerInvariant();
            this.campo = c.Trim().ToLowerInvariant();
            this.valorActual = v;
            if (this.valorActual == 0)
                this.valorActual++;
        }

        #endregion

        #region · Métodos estáticos ·

        /// <summary>
        /// Rellena una lista de objetos <see cref="TablasGeneradores"/> con los generadores que tiene una base de datos
        /// </summary>
        /// <remarks>
        /// Para asociar unn generador con una tabla, este método se basa en la suposición de que, en las tablas que 
        /// utilizan los campos autoincrementales, existe un trigger de tipo "before insert" y que contiene una sentencia 
        /// del tipo "new.xxx = gen_id(yyy, 1)" donde 'xxx' es el nombre del campo e 'yyy' es el nombre del generador.
        /// </remarks>
        /// <param name="conn">Conexión con el servidor Firebird</param>
        /// <param name="lista">Lista de objetos <see cref="TablasGeneradores"/> que vamos a rellenar</param>
        public static void RellenaListaTablasGeneradores(DbConnection conn, List<TablasGeneradores> lista)
        {
            if (conn == null || lista == null)
                return;

            using (DbCommand cmd = conn.CreateCommand())
            {
                DataTable dtTriggers = conn.GetSchema("Triggers");

                int numFilas = dtTriggers.Rows.Count;
                for (int nLin = 0; nLin < numFilas; nLin++)
                {
                    DataRow row = dtTriggers.Rows[nLin];

                    if (Convert.ToInt32(row["IS_SYSTEM_TRIGGER"].ToString()) == 0 && Convert.ToInt32(row["TRIGGER_TYPE"].ToString()) == 1)
                    {
                        string tabla = row["TABLE_NAME"].ToString();
                        string source = row["SOURCE"].ToString().ToUpper();
                        int pos = source.IndexOf("GEN_ID(");
                        if (pos != -1)
                        {
                            //
                            // Nombre del campo
                            //
                            int posPunto = source.LastIndexOf('.', pos - 1);
                            StringBuilder campo = new StringBuilder();
                            if (posPunto != -1)
                            {
                                for (int i = posPunto + 1; source[i] != ' ' && source[i] != '='; i++)
                                    campo.Append(source[i]);
                            }
                            //
                            // Nombre del generador asociado
                            //
                            StringBuilder generador = new StringBuilder();
                            for (int i = pos + 7; Char.IsLetterOrDigit(source[i]) || source[i] == '_'; i++)
                                generador.Append(source[i]);

                            int valor = 0;
                            cmd.CommandText = String.Format("SELECT GEN_ID({0},0) FROM RDB$DATABASE", generador.ToString());
                            object o = cmd.ExecuteScalar();
                            if (o != null)
                                valor = Convert.ToInt32(o);

                            lista.Add(new TablasGeneradores(generador.ToString(), tabla, campo.ToString(), valor));
                        }
                    }
                }
            }
        }

        #endregion
    }
}
