/* $PostgresPy: if/src/utils.c,v 1.22 2004/07/27 16:13:06 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *    Postgres[Py] utilities
 */
#include <setjmp.h>
#include <pputils.h>

#include <postgres.h>
#include <access/tupdesc.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_cast.h>
#include <catalog/catversion.h>
#include <executor/tuptable.h>
#include <nodes/params.h>
#include <parser/parse_type.h>
#include <utils/array.h>
#include <utils/rel.h>
#include <utils/builtins.h>
#include <utils/palloc.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/elog.h>
#include <tcop/tcopprot.h>
#include <pg.h>
#include <PGExcept.h>

#include <Python.h>
#include <structmember.h>
#include <py.h>

#include "datum.h"
#include "tup.h"
#include "tupd.h"
#include "obj.h"
#include "type.h"
#include "utils.h"
#include "module.h"

/*
 * PySpec - Inspect a PyObj and print some info about it.
 * This is here to provide a function that the debugger can call.
 */
void
PySpec(PyObj ob)
{
	PySPEC(ob);
}

PyObj
PyObj_FromDatumAndTypeTuple(Datum d, HeapTuple tt)
{
	PyObj rob = NULL;
	Form_pg_type ts = TYPESTRUCT(tt);

	if (ts->typrelid)
	{
		TupleDesc rtd;
		PyObj tdo;

		rtd = RelationOidGetDescr(ts->typrelid);
		tdo = PyPgTupleDesc_New(rtd);
		rob = PyPgTuple_New((HeapTuple)d, tdo);
		DECREF(tdo);
	}
	else
		rob = PgObj_FromDatumAndTypeOid(d, HeapTupleGetOid(tt));

	return(rob);
}

PyObj
PyObj_FromDatumAndTypeOid(Datum d, Oid to)
{
	PyObj rob;
	HeapTuple tt;

	tt = SearchSysCache(TYPEOID, to, 0, 0, 0);	
	rob = PyObj_FromDatumAndTypeTuple(d, tt);
	ReleaseSysCache(tt);

	return(rob);
}

Oid
TypeOidMod_FromPyObject(PyObj ob, int32 *typmod)
{
	volatile Oid typoid = InvalidOid;
	*typmod = -1;

	if (PyObject_TypeCheck(ob, &PyString_Type))
	{
		PgErr_TRAP( parseTypeString(PYASSTR(ob), &typoid, typmod); );
	}
	else if (PyObject_TypeCheck(ob, &PyInt_Type))
	{
		typoid = PYASINT(ob);
	}
	else if (PgObj_TypeCheck(ob) && PgObj_FetchTypeOid(ob) == OIDOID)
	{
		typoid = (Oid)(PgObj(ob)->ob_datum);
	}
	else if (PgType_TypeCheck(ob))
	{
		typoid = PgType_FetchOid(ob);
	}
	else
	{
		typoid = InvalidOid;
		PyErr_Format(PyExc_TypeError,
			"Type(1) must be a string(TypeName), Integer(TypeOid), "
			"Type HeapTuple, not a %s.", ob->ob_type->tp_name);
	}

	return(typoid);
}

Oid
TypeOid_FromPyObject(PyObj ob)
{
	int32 mod;
	return(TypeOidMod_FromPyObject(ob, &mod));
}

TupleDesc
RelationOidGetDescr(Oid reloid)
{
	Relation rel;
	TupleDesc rtupd;

	rel = RelationIdGetRelation(reloid);
	rtupd = CreateTupleDescCopyConstr(RelationGetDescr(rel));
	RelationClose(rel);

	return(rtupd);
}

TupleDesc
TupleDesc_FromTypeOid(Oid typoid)
{
	HeapTuple tt;
	Form_pg_type ts;
	TupleDesc rtd;

	tt = SearchSysCache(TYPEOID, typoid, 0, 0, 0);
	ts = TYPESTRUCT(tt);

	if (ts->typrelid)
	{
		rtd = RelationOidGetDescr(ts->typrelid);
	}
	else
	{
		rtd = CreateTemplateTupleDesc(1, false);
		TupleDescInitEntry(rtd, 1, "column", typoid, -1, 0, false);
	}
	ReleaseSysCache(tt);

	return(rtd);
}

ParamListInfo
ParamListInfo_FromDatumsAndNulls
(
	unsigned long nargs,
	Datum *values,
	char *nulls
)
{
	unsigned int i = 0;
	ParamListInfo pli = palloc0((nargs + 1) * sizeof(ParamListInfoData));

	for (i = 0; i < nargs; ++i)
	{
		pli[i].kind = PARAM_NUM;
		pli[i].id = i + 1;
		pli[i].isnull = (nulls[i] == 'n');
		pli[i].value = values[i];
	}
	
	pli[i].kind = PARAM_INVALID;
	return(pli);
}

char *
PgTypeName_FromOid(Oid typoid)
{
	char *pstr;
	HeapTuple ttup;

	ttup = SearchSysCache(TYPEOID, typoid, 0, 0, 0);
	pstr = pstrdup(NameStr(TYPESTRUCT(ttup)->typname));
	ReleaseSysCache(ttup);

	return(pstr);
}

Datum
Datum_Cast(Datum d, Oid from, Oid to)
{
	HeapTuple tup;
	Oid castor;
	Datum rd;
	
	tup = SearchSysCache(CASTSOURCETARGET, from, to, 0, 0);
	if (tup == NULL)
	{
		char *fromTypeStr, *toTypeStr;
		fromTypeStr = PgTypeName_FromOid(from);
		toTypeStr = PgTypeName_FromOid(to);

		ereport(ERROR,(
			errmsg("no function for casting a '%s' datum to a '%s' datum",
				fromTypeStr, toTypeStr)
		));
	}

	castor = CASTSTRUCT(tup)->castfunc;
	ReleaseSysCache(tup);

	rd = OidFunctionCall1(castor, d);
	return(rd);
}

ArrayType *
Array_FromTypeTupleAndPySequence(HeapTuple ttup, PyObj seq)
{
	Form_pg_type typs = TYPESTRUCT(ttup), etyps;
	Oid elmtoid = typs->typelem;

	HeapTuple ett;
	int i, nelems;
	Datum *des;
	bool *frees;
	ArrayType *ra;

	ett = SearchSysCache(TYPEOID, ObjectIdGetDatum(elmtoid), 0, 0, 0);
	if (!HeapTupleIsValid(ett))
	{
		PyErr_Format(PyExc_TypeError,
			"array's element type(%d) does not exist", elmtoid);
		return(NULL);
	}
	etyps = TYPESTRUCT(ett);

	nelems = PySequence_Length(seq);
	des = palloc(sizeof(Datum) * nelems);
	frees = palloc(sizeof(bool) * nelems);
	for (i = 0; i < nelems; ++i)
	{
		PyObj ob;
		ob = PySequence_GetItem(seq, i);
		frees[i] = true;

		if (PgObj_TypeCheck(ob))
		{
			Oid otoid;
			if ((otoid = PgObj_FetchTypeOid(ob)) == elmtoid)
			{
				des[i] = PgDat_FetchDatum(ob);
				frees[i] = false;
			}
			else
				des[i] = Datum_Cast(PgDat_FetchDatum(ob), otoid, elmtoid);
		}
		else
			des[i] = Datum_FromPyObjectAndTypeTuple(ob, ett);

		DECREF(ob);
	}
	ra = construct_array(des, nelems, elmtoid,
			etyps->typlen, etyps->typbyval, etyps->typalign);

	if (!etyps->typbyval)
	{
		for (i = 0; i < nelems; ++i)
		{
			if (frees[i])
				pfree(DatumGetPointer(des[i]));
		}
	}
	pfree(des);
	pfree(frees);

	ReleaseSysCache(ett);

	return(ra);
}

PyObj
PyErr_SetPgError(void)
{
#ifdef HAS_ERRORDATA
	ErrorData *edata = &errordata[errordata_stack_depth];

	switch (edata->code)
	{
		case ERRCODE_OUT_OF_MEMORY:
			PyErr_NoMemory();
		break;
		case ERRCODE_DISK_FULL:
			PyErr_SetString(PyExc_IOError, "disk full");
		break;
		case ERRCODE_IO_ERROR:
			PyErr_SetString(PyExc_IOError, edata->message);
		break;
		case ERRCODE_SYNTAX_ERROR:
			PyErr_SetString(PyExc_SyntaxError, edata->message);
		break;

		default:
			PyErr_SetString(PyExc_PgErr, edata->message);
		break;
	}
#else
	PyErr_SetNone(PyExc_PgErr);
#endif

	return(PyErr_Occurred());
}
