/*
 * Copyright (c) 2012, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package org.glassfish.hk2.api;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Type;
import java.util.Set;

/**
 * An Injectee represents the point of injection.  It can be used
 * by injection resolvers to discover all of the information available about
 * the entity being injected into.
 * 
 * @author jwells
 */
public interface Injectee {
    /**
     * This is the required type of the injectee.  The object
     * that is injected into this point must be type-safe
     * with regards to this type
     * 
     * @return The type that this injectee is expecting.  Any object
     * injected into this injection point must be type-safe with
     * regards to this type
     */
    public Type getRequiredType();
    
    /**
     * This is the set of required qualifiers for this injectee.  All
     * of these qualifiers must be present on the implementation class
     * of the object that is injected into this injectee.  Note that the
     * fields of the annotation must also match
     * 
     * @return Will not return null, but may return an empty set.  The
     * set of all qualifiers that must match.
     */
    public Set<Annotation> getRequiredQualifiers();
    
    /**
     * If this Injectee is a constructor or method parameter, this will
     * return the index of the parameter.  If this Injectee is a field,
     * this will return -1
     * 
     * @return the position of the parameter, or -1 if this is a field
     */
    public int getPosition();
    
    /**
     * Returns the parent class for this injectee.  This is the
     * class of the object that will be injected into.  This
     * field may return null if this is from a service lookup
     * 
     * @return The class of the object that will be injected into
     */
    public Class<?> getInjecteeClass();
    
    /**
     * If this Injectee is in a constructor this will return the 
     * constructor being injected into.  If this Injectee is in a
     * method this will return the method being injected into.  If this
     * injectee represents a field, this will return the field being
     * injected into.  This injectee may be neither in which case this
     * will return null
     * 
     * @return The parent of the injectee, which may be null
     */
    public AnnotatedElement getParent();
    
    /**
     * This method returns true if this injection point is annotated
     * with &#86;Optional.  In this case if there is no definition for
     * the injection point in the system it is allowable for the system
     * to merely return null
     * 
     * @return true if the injection point is annotated with &#86;Optional,
     * false otherwise
     */
    public boolean isOptional();
    
    /**
     * This method returns true if this injection point is annotated
     * with &#86;Self.  In this case the required type must have
     * a raw class of {@link ActiveDescriptor}, isOptional must be
     * false and the set of required qualifiers must be empty
     * 
     * @return true if the injection point is annotated with &#86;Self,
     * false otherwise
     */
    public boolean isSelf();
    
    /**
     * This method returns the {@link Unqualified} annotation if it is present
     * on the injection point.  This can be used to determine what qualifiers
     * should NOT be given to the injection point.
     * 
     * @return The {@link Unqualified} annotation on the injection point, or null
     * if there is no {@link Unqualified} annotation on the injection point
     */
    public Unqualified getUnqualified();
    
    /**
     * This returns the ActiveDescriptor of the injectee
     * if it is known
     * 
     * @return The ActiveDescriptor of the injectee,
     * or null if the parent is not known
     */
    public ActiveDescriptor<?> getInjecteeDescriptor();
}
