package org.mandas.docker.client.messages.mount;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link VolumeOptions}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableVolumeOptions.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableVolumeOptions implements VolumeOptions {
  private final @Nullable Boolean noCopy;
  private final @Nullable Map<String, String> labels;
  private final @Nullable Driver driverConfig;

  private ImmutableVolumeOptions(
      @Nullable Boolean noCopy,
      @Nullable Map<String, String> labels,
      @Nullable Driver driverConfig) {
    this.noCopy = noCopy;
    this.labels = labels;
    this.driverConfig = driverConfig;
  }

  /**
   * @return The value of the {@code noCopy} attribute
   */
  @JsonProperty("NoCopy")
  @Override
  public @Nullable Boolean noCopy() {
    return noCopy;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code driverConfig} attribute
   */
  @JsonProperty("DriverConfig")
  @Override
  public @Nullable Driver driverConfig() {
    return driverConfig;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link VolumeOptions#noCopy() noCopy} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for noCopy (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableVolumeOptions withNoCopy(@Nullable Boolean value) {
    if (Objects.equals(this.noCopy, value)) return this;
    return new ImmutableVolumeOptions(value, this.labels, this.driverConfig);
  }

  /**
   * Copy the current immutable object by replacing the {@link VolumeOptions#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableVolumeOptions withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableVolumeOptions(this.noCopy, newValue, this.driverConfig);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link VolumeOptions#driverConfig() driverConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for driverConfig (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableVolumeOptions withDriverConfig(@Nullable Driver value) {
    if (this.driverConfig == value) return this;
    return new ImmutableVolumeOptions(this.noCopy, this.labels, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableVolumeOptions} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableVolumeOptions
        && equalTo(0, (ImmutableVolumeOptions) another);
  }

  private boolean equalTo(int synthetic, ImmutableVolumeOptions another) {
    return Objects.equals(noCopy, another.noCopy)
        && Objects.equals(labels, another.labels)
        && Objects.equals(driverConfig, another.driverConfig);
  }

  /**
   * Computes a hash code from attributes: {@code noCopy}, {@code labels}, {@code driverConfig}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(noCopy);
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + Objects.hashCode(driverConfig);
    return h;
  }

  /**
   * Prints the immutable value {@code VolumeOptions} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "VolumeOptions{"
        + "noCopy=" + noCopy
        + ", labels=" + labels
        + ", driverConfig=" + driverConfig
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link VolumeOptions} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable VolumeOptions instance
   */
  public static ImmutableVolumeOptions copyOf(VolumeOptions instance) {
    if (instance instanceof ImmutableVolumeOptions) {
      return (ImmutableVolumeOptions) instance;
    }
    return ImmutableVolumeOptions.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableVolumeOptions ImmutableVolumeOptions}.
   * <pre>
   * ImmutableVolumeOptions.builder()
   *    .noCopy(Boolean | null) // nullable {@link VolumeOptions#noCopy() noCopy}
   *    .labels(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link VolumeOptions#labels() labels}
   *    .driverConfig(org.mandas.docker.client.messages.mount.Driver | null) // nullable {@link VolumeOptions#driverConfig() driverConfig}
   *    .build();
   * </pre>
   * @return A new ImmutableVolumeOptions builder
   */
  public static ImmutableVolumeOptions.Builder builder() {
    return new ImmutableVolumeOptions.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableVolumeOptions ImmutableVolumeOptions}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements VolumeOptions.Builder {
    private Boolean noCopy;
    private Map<String, String> labels = null;
    private Driver driverConfig;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code VolumeOptions} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(VolumeOptions instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Boolean noCopyValue = instance.noCopy();
      if (noCopyValue != null) {
        noCopy(noCopyValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      @Nullable Driver driverConfigValue = instance.driverConfig();
      if (driverConfigValue != null) {
        driverConfig(driverConfigValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link VolumeOptions#noCopy() noCopy} attribute.
     * @param noCopy The value for noCopy (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NoCopy")
    public final Builder noCopy(@Nullable Boolean noCopy) {
      this.noCopy = noCopy;
      return this;
    }

    /**
     * Put one entry to the {@link VolumeOptions#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link VolumeOptions#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link VolumeOptions#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link VolumeOptions#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link VolumeOptions#driverConfig() driverConfig} attribute.
     * @param driverConfig The value for driverConfig (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("DriverConfig")
    public final Builder driverConfig(@Nullable Driver driverConfig) {
      this.driverConfig = driverConfig;
      return this;
    }

    /**
     * Builds a new {@link ImmutableVolumeOptions ImmutableVolumeOptions}.
     * @return An immutable instance of VolumeOptions
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableVolumeOptions build() {
      return new ImmutableVolumeOptions(noCopy, labels == null ? null : createUnmodifiableMap(false, false, labels), driverConfig);
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
