/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_s_ImageColorRGB2XYZ.c	9.3	07/11/05 SMI"

/*
 * FUNCTION
 *      mlib_ImageColorRGB2XYZ - color conversion from RGB to XYZ space
 *
 * SYNOPSIS
 *      mlib_status mlib_ImageColorRGB2XYZ(mlib_image       *dst,
 *                                         const mlib_image *src);
 *
 * ARGUMENT
 *      dst     pointer to output image
 *      src     pointer to input image
 *
 * RESTRICTION
 *      src and dst must be the same type, the same size and the same
 *      number of channels.
 *      They must be 3-channel images.
 *      They can be in MLIB_BYTE, MLIB_SHORT, MLIB_USHORT or MLIB_INT data type.
 *
 * DESCRIPTION
 *
 *        |X|   |fmat[0] fmat[1] fmat[2]|   |R|
 *        |Y| = |fmat[3] fmat[4] fmat[5]| * |G|
 *        |Z|   |fmat[6] fmat[7] fmat[8]|   |B|
 *
 *   where
 *
 *      fmat[] = { 0.412453, 0.357580, 0.180423,
 *                 0.212671, 0.715160, 0.072169,
 *                 0.019334, 0.119193, 0.950227 };
 */

#include <mlib_image.h>
#include <mlib_ImageCheck.h>

/* *********************************************************** */

#if ! defined(__MEDIALIB_OLD_NAMES)
#if defined(__SUNPRO_C)

#pragma weak mlib_ImageColorRGB2XYZ = __mlib_ImageColorRGB2XYZ

#elif defined(__GNUC__)	/* defined(__SUNPRO_C) */
__typeof__(__mlib_ImageColorRGB2XYZ) mlib_ImageColorRGB2XYZ
    __attribute__((weak, alias("__mlib_ImageColorRGB2XYZ")));

#else /* defined(__SUNPRO_C) */

#error  "unknown platform"

#endif /* defined(__SUNPRO_C) */
#endif /* ! defined(__MEDIALIB_OLD_NAMES) */

/* *********************************************************** */

static const mlib_u32 t_r2xyz[256] = {
	0, 1024, 2049, 3073, 3074, 4098, 5122, 2103299,
	2104323, 2105348, 2106372, 2106373, 2107397, 2108421, 2109446, 2110470,
	2111495, 2111495, 2112519, 2113544, 4211720, 4212745, 4213769, 4214793,
	4214794, 4215818, 4216843, 4217867, 4218892, 4219916, 4220940, 4220941,
	4221965, 6320142, 6321166, 6322190, 6323215, 6323215, 6324240, 6325264,
	6326288, 6327313, 6328337, 6329362, 6329362, 6330387, 8428563, 8429587,
	8430612, 8431636, 8432661, 8432661, 8433685, 8434710, 8435734, 8436759,
	8437783, 8437784, 8438808, 10536984, 10538009, 10539033, 10540058,
	    10541082,
	10541082, 10542107, 10543131, 10544156, 10545180, 10546204, 10547229,
	    10547229,
	12645406, 12646430, 12647455, 12648479, 12649503, 12650528, 12650528,
	    12651553,
	12652577, 12653601, 12654626, 12655650, 12655651, 14753827, 14754851,
	    14755876,
	14756900, 14757925, 14758949, 14758950, 14759974, 14760998, 14762023,
	    14763047,
	14764072, 16862248, 16862248, 16863273, 16864297, 16865322, 16866346,
	    16867370,
	16867371, 16868395, 16869420, 16870444, 16871469, 16872493, 18970669,
	    18970670,
	18971694, 18972719, 18973743, 18974767, 18975792, 18976816, 18976817,
	    18977841,
	18978865, 18979890, 18980914, 21079091, 21079091, 21080116, 21081140,
	    21082164,
	21083189, 21084213, 21085238, 21085238, 21086262, 21087287, 21088311,
	    21089336,
	23187512, 23188537, 23188537, 23189561, 23190586, 23191610, 23192635,
	    23193659,
	23193659, 23194684, 23195708, 23196733, 23197757, 25295933, 25296958,
	    25296958,
	25297983, 25299007, 25300032, 25301056, 25302080, 25303105, 25303105,
	    25304130,
	25305154, 25306178, 27404355, 27405379, 27406404, 27406404, 27407428,
	    27408453,
	27409477, 27410502, 27411526, 27411527, 27412551, 27413575, 27414600,
	    29512776,
	29513801, 29514825, 29514825, 29515850, 29516874, 29517899, 29518923,
	    29519947,
	29520972, 29520972, 29521997, 29523021, 31621198, 31622222, 31623246,
	    31623247,
	31624271, 31625296, 31626320, 31627344, 31628369, 31629393, 31629394,
	    31630418,
	31631442, 33729619, 33730643, 33731668, 33732692, 33732693, 33733717,
	    33734741,
	33735766, 33736790, 33737815, 33737815, 33738839, 33739864, 35838040,
	    35839065,
	35840089, 35841114, 35841114, 35842138, 35843163, 35844187, 35845212,
	    35846236,
	35847260, 35847261, 35848285, 37946462, 37947486, 37948510, 37949535,
	    37950559,
	37950560, 37951584, 37952609, 37953633, 37954657, 37955682, 37955682,
	    37956707,
	40054883, 40055907, 40056932, 40057956, 40058981, 40058981, 40060005,
	    40061030,
	40062054, 40063079, 40064103, 40065128, 40065128, 42163304, 42164329,
	    42165353
};

/* *********************************************************** */

static const mlib_u32 t_g2xyz[256] = {
	0, 3072, 2103297, 2106369, 4205569, 4208642, 6308866, 6311939,
	8412163, 8415235, 10515460, 10517508, 12617732, 12620805, 14721029,
	    14724101,
	16824326, 16827398, 18926598, 18929671, 21029895, 21032968, 21036040,
	    23136264,
	23139337, 25239561, 25241609, 27341834, 27344906, 29445130, 29448203,
	    31548427,
	31551499, 33650700, 33653772, 35753997, 35757069, 37857293, 37860366,
	    39960590,
	39962638, 42062863, 42065935, 44166159, 44169232, 44172304, 46272528,
	    46274577,
	48374801, 48377874, 50478098, 50481170, 52581395, 52584467, 54683667,
	    54686740,
	56786964, 56790036, 58890261, 58893333, 60993557, 60995606, 63095830,
	    63098903,
	65199127, 65202199, 65205272, 67305496, 67308568, 69407769, 69410841,
	    71511065,
	71514138, 73614362, 73617434, 75717659, 75719707, 77819932, 77823004,
	    79923228,
	79926301, 82026525, 82029597, 84128798, 84131870, 86232094, 86235167,
	    86238239,
	88338463, 88341536, 90440736, 90443809, 92544033, 92547105, 94647330,
	    94650402,
	96750626, 96752675, 98852899, 98855971, 100956196, 100959268, 103059492,
	    103062565,
	105162789, 105164838, 107265062, 107268134, 107271207, 109371431,
	    109374503, 111474728,
	111476776, 113577000, 113580073, 115680297, 115683369, 117783594,
	    117786666, 119885867,
	119888939, 121989163, 121992236, 124092460, 124095532, 126195757,
	    126197805, 128298029,
	128301102, 130401326, 130404398, 130407471, 132507695, 132509744,
	    134609968, 134613040,
	136713265, 136716337, 138816561, 138819634, 140918834, 140921906,
	    143022131, 143025203,
	145125427, 145128500, 147228724, 147231797, 149330997, 149334069,
	    151434294, 151437366,
	151440438, 153540663, 153543735, 155642935, 155646008, 157746232,
	    157749304, 159849529,
	159852601, 161952826, 161954874, 164055098, 164058171, 166158395,
	    166161467, 168261692,
	168264764, 170363964, 170367037, 172467261, 172470334, 172473406,
	    174573630, 174576703,
	176675903, 176678975, 178779200, 178782272, 180882496, 180885569,
	    182985793, 182987841,
	185088066, 185091138, 187191363, 187194435, 189294659, 189297732,
	    191397956, 191400004,
	193500229, 193503301, 193506373, 195606598, 195609670, 197709894,
	    197711943, 199812167,
	199815240, 201915464, 201918536, 204018761, 204021833, 206121033,
	    206124106, 208224330,
	208227402, 210327627, 210330699, 212430923, 212432972, 214533196,
	    214536269, 216636493,
	216639565, 216642638, 218742862, 218744910, 220845135, 220848207,
	    222948431, 222951504,
	225051728, 225054800, 227154001, 227157073, 229257298, 229260370,
	    231360594, 231363667,
	233463891, 233466963, 235566164, 235569236, 237669460, 237672533,
	    237675605, 239775829,
	239778902, 241878102, 241881175, 243981399, 243984471, 246084696,
	    246087768, 248187992,
	248190041, 250290265, 250293337, 252393562, 252396634, 254496858,
	    254499931, 256599131
};

/* *********************************************************** */

static const mlib_u32 t_b2xyz[256] = {
	0, 8388608, 16778240, 23069697, 31458305, 39846913, 48236545, 56625153,
	62916609, 71306242, 79694850, 88083458, 96472066, 102764546, 111153155,
	    119541763,
	127931395, 136320003, 142611459, 151000067, 159389700, 167778308,
	    176166916, 182459396,
	190848004, 199236613, 207626245, 216014853, 222306309, 230694917,
	    239084549, 247473158,
	255861766, 262154246, 270542854, 278931462, 287320070, 295709703,
	    302001159, 310389767,
	318779399, 327168007, 335556616, 341848072, 350237704, 358626312,
	    367014920, 375404552,
	381696009, 390084617, 398473225, 406862857, 415251465, 421542922,
	    429932554, 438321162,
	446709770, 455098378, 461390858, 469779467, 478168075, 486557707,
	    494946315, 501237771,
	509626380, 518016012, 526404620, 534793228, 541085708, 549474316,
	    557862925, 566251533,
	574641165, 580932621, 589321229, 597710862, 606099470, 614488078,
	    620780558, 629169166,
	637557774, 645946383, 654336015, 660627471, 669016079, 677405711,
	    685794320, 694182928,
	700474384, 708864016, 717252624, 725641232, 734030865, 740322321,
	    748710929, 757099537,
	765489169, 773877778, 780169234, 788558866, 796947474, 805336082,
	    813724690, 820017171,
	828405779, 836794387, 845184019, 853572627, 859864083, 868252692,
	    876642324, 885030932,
	893419540, 901809172, 908100629, 916489237, 924877845, 933267477,
	    941656085, 947947541,
	956337174, 964725782, 973114390, 981504022, 987795478, 996184087,
	    1004572695, 1012962327,
	1021350935, 1027642391, 1036032023, 1044420632, 1052809240, 1061197848,
	    1067490328, 1075878936,
	1084267545, 1092657177, 1101045785, 1107337241, 1115725849, 1124115481,
	    1132504090, 1140892698,
	1147185178, 1155573786, 1163962394, 1172351003, 1180740635, 1187032091,
	    1195420699, 1203810331,
	1212198939, 1220587548, 1226879004, 1235268636, 1243657244, 1252045852,
	    1260435485, 1266726941,
	1275115549, 1283504157, 1291893789, 1300282397, 1306573854, 1314963486,
	    1323352094, 1331740702,
	1340129310, 1346421790, 1354810399, 1363199007, 1371588639, 1379977247,
	    1386268703, 1394658336,
	1403046944, 1411435552, 1419824160, 1426116640, 1434505248, 1442893857,
	    1451283489, 1459672097,
	1465963553, 1474352161, 1482741794, 1491130402, 1499519010, 1505811490,
	    1514200098, 1522588706,
	1530977315, 1539366947, 1545658403, 1554047011, 1562436643, 1570825252,
	    1579213860, 1585505316,
	1593894948, 1602283556, 1610672164, 1619061797, 1625353253, 1633741861,
	    1642130469, 1650520101,
	1658908710, 1665200166, 1673589798, 1681978406, 1690367014, 1698755622,
	    1705048103, 1713436711,
	1721825319, 1730214951, 1738603559, 1744895016, 1753284648, 1761673256,
	    1770061864, 1778450472,
	1784742952, 1793131561, 1801520169, 1809909801, 1818298409, 1824589865,
	    1832978473, 1841368106,
	1849756714, 1858145322, 1864437802, 1872826410, 1881215019, 1889603627,
	    1897993259, 1904284715,
	1912673323, 1921062955, 1929451564, 1937840172, 1944131628, 1952521260,
	    1960909868, 1969298477,
	1977688109, 1983979565, 1992368173, 2000756781, 2009146413, 2017535022,
	    2023826478, 2032216110
};

/* *********************************************************** */

#ifdef _LITTLE_ENDIAN

#define	SPLIT_S32_U8(x, a0, a1, a2, a3)                         \
	a3 = (x >> 24);                                         \
	a2 = (x & 0xff0000) >> 16;                              \
	a1 = (x & 0x00ff00) >> 8;                               \
	a0 = (x & 0x0000ff)

#else /* _LITTLE_ENDIAN */

#define	SPLIT_S32_U8(x, a0, a1, a2, a3)                         \
	a0 = (x >> 24);                                         \
	a1 = (x & 0xff0000) >> 16;                              \
	a2 = (x & 0x00ff00) >> 8;                               \
	a3 = (x & 0x0000ff)

#endif /* _LITTLE_ENDIAN */

/* *********************************************************** */

#define	MLIB_CONVERT_RGB_U8_1(pd, ps)                           \
	mlib_u32 r0, g0, b0;                                    \
	mlib_u32 s0;                                            \
	                                                        \
	r0 = ps[0];                                             \
	g0 = ps[1];                                             \
	b0 = ps[2];                                             \
	s0 = t_r2xyz[r0] + t_g2xyz[g0] + t_b2xyz[b0];           \
	pd[0] = s0;                                             \
	pd[1] = s0 >> 12;                                       \
	pd[2] = (s0 | ((mlib_s32)s0 >> 31)) >> 23

/* *********************************************************** */

#define	MLIB_CONVERT_RGB_U8_4(pd, ps)                              \
	mlib_u32 s0, s1, s2, s3;                                   \
	mlib_u32 r0, g0, b0, r1, g1, b1, r2, g2, b2, r3, g3, b3;   \
	mlib_u32 r0g0b0r1, g1b1r2g2, b2r3g3b3;                     \
	                                                           \
	r0g0b0r1 = ps[0];                                          \
	g1b1r2g2 = ps[1];                                          \
	b2r3g3b3 = ps[2];                                          \
	SPLIT_S32_U8(r0g0b0r1, r0, g0, b0, r1);                    \
	SPLIT_S32_U8(g1b1r2g2, g1, b1, r2, g2);                    \
	SPLIT_S32_U8(b2r3g3b3, b2, r3, g3, b3);                    \
	s0 = t_r2xyz[r0] + t_g2xyz[g0] + t_b2xyz[b0];              \
	s1 = t_r2xyz[r1] + t_g2xyz[g1] + t_b2xyz[b1];              \
	s2 = t_r2xyz[r2] + t_g2xyz[g2] + t_b2xyz[b2];              \
	s3 = t_r2xyz[r3] + t_g2xyz[g3] + t_b2xyz[b3];              \
	pd[0] = s0;                                                \
	pd[1] = s0 >> 12;                                          \
	pd[2] = (s0 | ((mlib_s32)s0 >> 31)) >> 23;                 \
	pd[3] = s1;                                                \
	pd[4] = s1 >> 12;                                          \
	pd[5] = (s1 | ((mlib_s32)s1 >> 31)) >> 23;                 \
	pd[6] = s2;                                                \
	pd[7] = s2 >> 12;                                          \
	pd[8] = (s2 | ((mlib_s32)s2 >> 31)) >> 23;                 \
	pd[9] = s3;                                                \
	pd[10] = s3 >> 12;                                         \
	pd[11] = (s3 | ((mlib_s32)s3 >> 31)) >> 23

/* *********************************************************** */

#define	MLIB_CONVERT_RGB_U8_4s(rgb, s0, s1, s2, s3)                    \
	{                                                              \
	    mlib_u32 r0, g0, b0, r1, g1, b1, r2, g2, b2, r3, g3, b3;   \
	    mlib_u32 r0g0b0r1, g1b1r2g2, b2r3g3b3;                     \
	                                                               \
	    r0g0b0r1 = ((mlib_s32 *)(rgb))[0];                         \
	    g1b1r2g2 = ((mlib_s32 *)(rgb))[1];                         \
	    b2r3g3b3 = ((mlib_s32 *)(rgb))[2];                         \
	    SPLIT_S32_U8(r0g0b0r1, r0, g0, b0, r1);                    \
	    SPLIT_S32_U8(g1b1r2g2, g1, b1, r2, g2);                    \
	    SPLIT_S32_U8(b2r3g3b3, b2, r3, g3, b3);                    \
	    s0 = t_r2xyz[r0] + t_g2xyz[g0] + t_b2xyz[b0];              \
	    s1 = t_r2xyz[r1] + t_g2xyz[g1] + t_b2xyz[b1];              \
	    s2 = t_r2xyz[r2] + t_g2xyz[g2] + t_b2xyz[b2];              \
	    s3 = t_r2xyz[r3] + t_g2xyz[g3] + t_b2xyz[b3];              \
	}

/* *********************************************************** */

#define	MLIB_CONVERT_RGB_U8_4ll(rgb, y0, cb0, cr0, y1, cb1,     \
	cr1, y2, cb2, cr2, y3, cb3, cr3)                        \
	mlib_u32 s0, s1, s2, s3;                                \
	                                                        \
	MLIB_CONVERT_RGB_U8_4s(rgb, s0, s1, s2, s3);            \
	y0 = s0;                                                \
	cb0 = s0 >> 10;                                         \
	cr0 = s0 >> 21;                                         \
	y1 = s1;                                                \
	cb1 = s1 >> 10;                                         \
	cr1 = s1 >> 21;                                         \
	y2 = s2;                                                \
	cb2 = s2 >> 10;                                         \
	cr2 = s2 >> 21;                                         \
	y3 = s3;                                                \
	cb3 = s3 >> 10;                                         \
	cr3 = s3 >> 21

/* *********************************************************** */

mlib_status
__mlib_ImageColorRGB2XYZ(
    mlib_image *dst,
    const mlib_image *src)
{
/* Rec709 RGB with D64 White Point to CIE XYZ */
	mlib_d64 fmat[9] = { 0.412453, 0.357580, 0.180423,
		0.212671, 0.715160, 0.072169,
		0.019334, 0.119193, 0.950227
	};
	mlib_s32 slb, dlb, xsize, ysize;
	mlib_type dtype;
	mlib_u8 *psrc, *pdst;
	mlib_s32 *ps4;
	mlib_s32 j;

	MLIB_IMAGE_CHECK(dst);
	MLIB_IMAGE_CHECK(src);
	MLIB_IMAGE_FULL_EQUAL(dst, src);

	dtype = mlib_ImageGetType(dst);
	xsize = mlib_ImageGetWidth(dst);
	ysize = mlib_ImageGetHeight(dst);
	dlb = mlib_ImageGetStride(dst);
	pdst = (void *)mlib_ImageGetData(dst);

	slb = mlib_ImageGetStride(src);
	psrc = mlib_ImageGetData(src);

	if (dtype == MLIB_BYTE) {

		for (j = 0; j < ysize; j++) {
			mlib_u8 *ps = psrc,
			    *pd = pdst, *pend = pdst + 3 * xsize;

/* first pixels loop */
			while (((mlib_addr)ps & 3) && (pd < pend)) {
				MLIB_CONVERT_RGB_U8_1(pd, ps);
				pd += 3;
				ps += 3;
			}

			ps4 = (mlib_s32 *)ps;
/* 4 pixel loop */
#ifdef __SUNPRO_C
#pragma pipeloop(0)
#endif /* __SUNPRO_C */
			for (; pd <= (pend - 12); pd += 12) {
				MLIB_CONVERT_RGB_U8_4(pd, ps4);
				ps4 += 3;
			}

			ps = (mlib_u8 *)ps4;
/* last pixels loop */
			for (; pd < pend; pd += 3, ps += 3) {
				MLIB_CONVERT_RGB_U8_1(pd, ps);
			}

			psrc += slb;
			pdst += dlb;
		}

		return (MLIB_SUCCESS);

	} else {

	return (__mlib_ImageColorConvert1(dst, src, fmat));
	}
}

/* *********************************************************** */
