/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_ImageConvClearEdge_Fp.c	9.2	07/10/09 SMI"

/*
 * FUNCTIONS
 *      mlib_c_ImageConvClearEdge  - Set edge of an image to a specific
 *                                        color. (for float-point image)
 *
 * SYNOPSIS
 *      mlib_status mlib_c_ImageConvClearEdge_Fp(mlib_image     *img,
 *                                               mlib_s32       dx_l,
 *                                               mlib_s32       dx_r,
 *                                               mlib_s32       dy_t,
 *                                               mlib_s32       dy_b,
 *                                               const mlib_d64 *color,
 *                                               mlib_s32       cmask)
 *
 * ARGUMENT
 *      img       Pointer to an image.
 *      dx_l      Number of columns on the left side of the
 *                image to be cleared.
 *      dx_r      Number of columns on the right side of the
 *                image to be cleared.
 *      dy_t      Number of rows on the top edge of the
 *                image to be cleared.
 *      dy_b      Number of rows on the top edge of the
 *                image to be cleared.
 *      color     Pointer to the color that the edges are set to.
 *      cmask     Channel mask to indicate the channels to be convolved.
 *                Each bit of which represents a channel in the image. The
 *                channels corresponded to 1 bits are those to be processed.
 *
 * RESTRICTION
 *      img can have 1, 2, 3 or 4 channels of MLIB_FLOAT or MLIB_DOUBLE
 *      data type.
 *
 * DESCRIPTION
 *      Set edge of an image to a specific color.
 *      The unselected channels are not overwritten.
 *      If src and dst have just one channel,
 *      cmask is ignored.
 */

#include <mlib_image.h>
#include <mlib_ImageConvEdge.h>

/* *********************************************************** */

#define	EDGES(chan, type, mask)                                          \
	{                                                                \
	    type *pimg = (type *) mlib_ImageGetData(img);                \
	    type color_i;                                                \
	    mlib_s32 img_stride =                                        \
		mlib_ImageGetStride(img) / sizeof (type);                \
	    mlib_s32 i, j, l;                                            \
	    mlib_s32 testchan;                                           \
	                                                                 \
	    testchan = 1;                                                \
	    for (l = chan - 1; l >= 0; l--) {                            \
		if ((mask & testchan) == 0) {                            \
		    testchan <<= 1;                                      \
		    continue;                                            \
		}                                                        \
		testchan <<= 1;                                          \
		color_i = (type) color[l];                               \
		for (j = 0; j < dx_l; j++) {                             \
		    for (i = dy_t; i < (img_height - dy_b); i++) {       \
			pimg[i * img_stride + l + j * chan] = color_i;   \
		    }                                                    \
		}                                                        \
		for (j = 0; j < dx_r; j++) {                             \
		    for (i = dy_t; i < (img_height - dy_b); i++) {       \
			pimg[i * img_stride + l + (img_width - 1 -       \
			    j) * chan] = color_i;                        \
		    }                                                    \
		}                                                        \
		for (i = 0; i < dy_t; i++) {                             \
		    for (j = 0; j < img_width; j++) {                    \
			pimg[i * img_stride + l + j * chan] = color_i;   \
		    }                                                    \
		}                                                        \
		for (i = 0; i < dy_b; i++) {                             \
		    for (j = 0; j < img_width; j++) {                    \
			pimg[(img_height - 1 - i) * img_stride + l +     \
			    j * chan] = color_i;                         \
		    }                                                    \
		}                                                        \
	    }                                                            \
	}

/* *********************************************************** */

mlib_status
mlib_ImageConvClearEdge_Fp(
    mlib_image *img,
    mlib_s32 dx_l,
    mlib_s32 dx_r,
    mlib_s32 dy_t,
    mlib_s32 dy_b,
    const mlib_d64 *color,
    mlib_s32 cmask)
{
	mlib_s32 img_width = mlib_ImageGetWidth(img);
	mlib_s32 img_height = mlib_ImageGetHeight(img);
	mlib_s32 channel = mlib_ImageGetChannels(img);

	if (dx_l + dx_r > img_width) {
		dx_l = img_width;
		dx_r = 0;
	}

	if (dy_t + dy_b > img_height) {
		dy_t = img_height;
		dy_b = 0;
	}

	if (channel == 1)
		cmask = 1;

	switch (mlib_ImageGetType(img)) {
	case MLIB_FLOAT:
		EDGES(channel, mlib_f32, cmask);
		break;
	case MLIB_DOUBLE:
		EDGES(channel, mlib_d64, cmask);
		break;
	default:
		return (MLIB_FAILURE);
	}

	return (MLIB_SUCCESS);
}

/* *********************************************************** */
