package com.streamsicle.util;

import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.filechooser.*;
import org.ten60.orchextra.*;
/**
 * A Windows-style directory chooser for Swing. As of Java 1.3, using
 * <code>JFileChooser</code> for choosing directories is a fairly
 * miserable ordeal. For example, it is impossible to select a top-level
 * directory with just the mouse.
 * <p>
 * This chooser simplifies the process. It does not yet suppor the
 * ability to create a new directory on the fly.
 * <p>
 * Here is some example code that uses a JDirectoryChooser to select a directory:
 * <pre>
 * JDirectoryChooser chooser = new JDirectoryChooser(null, "Choose a directory");
 * File dir = chooser.getDirectory();
 * System.out.println("Directory chosen: " + dir);
 * </pre>
 * If a default directory is already known, then:
 * <pre>
 * File defaultPath = SOME_DEFAULT_DIRECTORY;
 * JDirectoryChooser chooser = new JDirectoryChooser(null, "Choose a directory", defaultPath);
 * File dir = chooser.getDirectory();
 * System.out.println("Directory chosen: " + dir);
 * </pre>
 *
 * @author John Watkinson
 */

public class JDirectoryChooser extends JDialog implements TreeSelectionListener, ActionListener {

   ///////////////////////////////////////////////////////////////////////////
   // Inner Classes
   ///////////////////////////////////////////////////////////////////////////

   /**
    * A wrapper class for a file that has its <code>toString()</code> method
    * designed to return appropriate info for the directory tree.
    */
   public static class TreeFile
   {
      private File f;

      public TreeFile (File f) {
         this.f = f;
      }

      public File getFile () {
         return f;
      }

      /**
       * Returns the name of the file without path information unless
       * the file has no name in which case the full path information is
       * returned. The latter case happens when the file is a top-level
       * directory.
       */
      public String toString () {
         String name = f.getName();
         if (name.length() == 0) {
            return f.toString();
         }
         return name;
      }
   }

   /**
    * A file filter that selects only directories.
    */
   private static class DirectoryFilter implements FilenameFilter
   {
      public boolean accept (File dir, String name) {
         File f = new File(dir, name);
         return f.isDirectory();
      }
   }

   /**
    * A tree model that assumes that each object in the tree is a directory.
    * This model caches directory information as it is discovered by
    * the user browsing the tree, but in a very optimistic way.
    * It will never update a branch once it has been discovered if the
    * underlying file structure changes. However, it will make sure that
    * the  directory that is eventually returned exists. It also does
    * not flush it's cache until the directory chooser is closed. So,
    * if a large branch of the tree is explored, then closed up, the cache
    * will still contain all the nodes of the branch.
    */
   public static class DirectoryModel implements TreeModel
   {
      /**
       * A special node designating the "root" of the tree
       * (but <i>not</i> the root of the file system -- Windows filesystems
       * have multiple roots.
       */
      private static final String ROOT = "ROOT";

      /**
       * The filesystem roots.
       */
      private File[] roots;

      /**
       * The hashtable that maps explored directories to their children.
       */
      private Hashtable hash;

      private static FilenameFilter filter = new DirectoryFilter();

      public DirectoryModel () {
         roots = FileSystemView.getFileSystemView().getRoots();
         hash = new Hashtable();
      }

      /**
       * Returns the index of the top-level dir, or <code>-1</code> if
       * the file is not a top-level dir.
       */
      private int getTopLevel (File f) {
         for (int i = 0; i < roots.length; i++) {
            if (f.getAbsolutePath().equals(roots[i].getAbsolutePath())) {
               return i;
            }
         }
         return -1;
      }

      /**
       * This tree model will never change, so we will never need to notify a 
       * listener.
       */
      public void addTreeModelListener (TreeModelListener tml) {
         // Who cares
      }

      /**
       * Returns an alphabetically sorted (case insensitive) list of
       * subdirectories. Also caches the result for the life of the chooser.
       */
      private String[] getSubDirs (File f) {
         String[] result;
         result = (String[])hash.get(f);
         if (result == null) {
            result = f.list(filter);
            if (result == null) {
               result = new String[0];
            }
            hash.put(f, result);
         }
         QuickSort.sort(result);
         return result;
      }

      /**
       * Gets the subdirectory of the specified parent directory at the
       * the specified index.
       */
      public Object getChild (Object parent, int index) {
         if (parent instanceof String) {
            return new TreeFile(roots[index]);
         }
         File f = ((TreeFile)parent).getFile();
         File child = new File(f.getAbsolutePath(), getSubDirs(f)[index]);
         return new TreeFile(child);
      }

      /**
       * Returns the number of subdirectories of the specified directory.
       */
      public int getChildCount (Object parent) {
         if (parent instanceof String) {
            return roots.length;
         }
         return getSubDirs(((TreeFile)parent).getFile()).length;
      }

      /**
       * Returns the index of the specified subdirectory in the specified
       * parent directory.
       */
      public int getIndexOfChild(Object parent, Object child) {
         File childFile = ((TreeFile)child).getFile();
         if (parent instanceof String) {
            return getTopLevel(childFile);
         }
         File parentFile = ((TreeFile)parent).getFile();
         String[] list = getSubDirs(parentFile);
         for (int i = 0; i < list.length; i++) {
            if (list[i].compareTo(childFile.getName()) == 0) {
               return i;
            }
         }
         return -1;
      }

      /**
       * Gets the root of the tree (<i>not</i> the root of the filesystem).
       */
      public Object getRoot () {
         return ROOT;
      }

      /**
       * A directory is a leaf if it has no subdirectories.
       */
      public boolean isLeaf (Object node) {
         /*
         return false;
         */
         if (node instanceof String) {
            return false;
         }
         File f = ((TreeFile)node).getFile();
         if (getTopLevel(f) != -1) {
            return false;
         }
         if (getSubDirs(f).length == 0) {
            return true;
         } else {
            return false;
         }
      }

      /**
       * This tree model will never change, so we will never need to notify a 
       * listener.
       */
      public void removeTreeModelListener (TreeModelListener l) {
         // Who cares
      }

      /**
       * This tree model will never change, so we will never need to notify a 
       * listener.
       */
      public void valueForPathChanged (TreePath path, Object newValue) {
         // Who cares
      }
   }

   ///////////////////////////////////////////////////////////////////////////
   // Constants
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Default size of the chooser.
    */
   public static final Dimension DEFAULT_SIZE = new Dimension(400, 400);

   /**
    * OK Action
    */
   public static final String OK_ACTION = "OK";

   /**
    * Cancel Action
    */
   public static final String CANCEL_ACTION = "Cancel";

   ///////////////////////////////////////////////////////////////////////////
   // Fields
   ///////////////////////////////////////////////////////////////////////////

   private JTree tree;
   private JTextField fileField;
   private JButton okButton;
   private JButton cancelButton;
   private File choice = null;

   ///////////////////////////////////////////////////////////////////////////
   // Constructors
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Constructs a new directory chooser with the specified owning 
    * <code>Frame</code> (<code>null</code> <i>is</i> allowed) and the 
    * specified title.
    */
   public JDirectoryChooser(Frame owner, String title) {
      super(owner, title, true);
      initGUI();
   }

   /**
    * Constructs a new directory chooser with the specified owning 
    * <code>Frame</code> (<code>null</code> <i>is</i> allowed), the 
    * specified title, and the specified initial path displayed and
    * selected.
    */
   public JDirectoryChooser(Frame owner, String title, File path) {
      super(owner, title, true);
      initGUI();
      setPath(path);
   }

   ///////////////////////////////////////////////////////////////////////////
   // Protected methods
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Initializes the GUI components.
    */
   protected void initGUI () {
      tree = new JTree(new DirectoryModel());
      tree.setRootVisible(false);
      tree.setEditable(false);
      tree.putClientProperty("JTree.lineStyle", "Angled");
      tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
      tree.addTreeSelectionListener(this);
      tree.setShowsRootHandles(true);
      DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
      renderer.setLeafIcon(new javax.swing.plaf.metal.MetalIconFactory.FolderIcon16());
      tree.setCellRenderer(renderer);
      fileField = new JTextField();
      fileField.setActionCommand(OK_ACTION);
      fileField.addActionListener(this);
      okButton = new JButton("OK");
      okButton.setActionCommand(OK_ACTION);
      okButton.addActionListener(this);
      cancelButton = new JButton("Cancel");
      cancelButton.setActionCommand(CANCEL_ACTION);
      cancelButton.addActionListener(this);
      JPanel mainPanel = new JPanel(new BorderLayout());
      JPanel bottomPanel = new JPanel(new BorderLayout());
      JPanel buttonPanel = new JPanel(new FlowLayout());
      buttonPanel.add(okButton);
      buttonPanel.add(cancelButton);
      bottomPanel.add(fileField, BorderLayout.NORTH);
      bottomPanel.add(buttonPanel, BorderLayout.SOUTH);
      mainPanel.add(new JScrollPane(tree), BorderLayout.CENTER);
      mainPanel.add(bottomPanel, BorderLayout.SOUTH);
      getContentPane().add(mainPanel);
      setSize(DEFAULT_SIZE);
      Toolkit toolkit = Toolkit.getDefaultToolkit();
      Dimension screenSize = toolkit.getScreenSize();
      int x = (screenSize.width - DEFAULT_SIZE.width) / 2;
      int y = (screenSize.height - DEFAULT_SIZE.height) / 2;
      setLocation(x, y);
   }

   /**
    * Displays a tree path (used for debugging).
    */
   protected void showPath (TreePath path) {
      Object[] objs = path.getPath();
      for (int i = 0; i < objs.length; i++) {
         //log.debug(objs[i]);
      }
   }

   /**
    * Sets the initial path of the directory chooser.
    */
   protected void setPath (File dir) {
      TreePath path = new TreePath(DirectoryModel.ROOT);
      Vector paths = new Vector();
      String parent = dir.getAbsolutePath();
      while (parent != null) {
         File f = new File(parent);
         paths.addElement(new TreeFile(f));
         parent = f.getParent();
      }
      for (int i = paths.size() - 1; i >= 0; i--) {
         path = path.pathByAddingChild(paths.elementAt(i));
      }
      tree.setSelectionPath(path);
   }

   ///////////////////////////////////////////////////////////////////////////
   // Public methods
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Runs the dialog and returns the file chosen, or <code>null</code> if
    * the <b>Cancel</b> button was pressed.
    */
   public File getDirectory () {
      show();
      return choice;
   }

   /**
    * Called when a node of the directory tree is selected.
    * This updates the text field below the tree.
    */
   public void valueChanged (TreeSelectionEvent e) {
      File f = ((TreeFile)(e.getPath().getLastPathComponent())).getFile();
      fileField.setText(f.getAbsolutePath());
   }

   /**
    * Called when either the <b>OK</b> or <b>Cancel</b> buttons are
    * pressed, or when <b>Enter</b> is pressed in the text field.
    */
   public void actionPerformed (ActionEvent e) {
      if (e.getActionCommand().equals(OK_ACTION)) {
         choice = new File(fileField.getText());
         if (choice.exists()) {
            dispose();
         } else {
            JOptionPane.showMessageDialog(this, "Directory does not exist.", "Error", JOptionPane.ERROR_MESSAGE);
            choice = null;
         }
      } else {
         choice = null;
         dispose();
      }
   }

   /**
    * Use this to test out the directory chooser.
    * It can be run with no args, or the first argument passed will be
    * the default path to reveal and select.
    */
   public static void main (String[] args) {
      JDirectoryChooser chooser = new JDirectoryChooser(null, "Choose a directory");
      if (args.length > 0) {
         chooser.setPath(new File(args[0]));
      }
      File choice = chooser.getDirectory();
      OrchextraAccessor.log(OrchextraAccessor.INFO, null, "Choice: " + choice);
      System.exit(0);
   }

}
