/*
 *  put your module comment here
 *  formatted with JxBeauty (c) johann.langhofer@nextra.at
 */

package com.streamsicle.fluid;

import java.io.*;
import java.util.*;
import com.streamsicle.*;
import com.streamsicle.util.*;
// Import log4j classes.
import org.ten60.orchextra.*;

/**
 *@author     Matt Hall
 *@author     John Watkinson
 *@created    October 8, 2002
 */
public class InteractiveStream extends MediaInputStream {
    private RequestQueue queue;
    private MediaInputStream current;
    private MP3File currentMedia;
    private MP3File root;
    //  Base directory for MP3Files
    private MP3File randomRoot;
    //  Root file object for selecting random files from
    private Vector availableRandomFiles;
    //  The list of files to select randomly from
    private boolean descriptions;
    private String playDir, hostName, hostDescription, portNumber;
    private String[] files;
    private Vector availableFiles;
    private Hashtable fileHash;
    //  For quicker access to the file objects when adding songs
    private Hashtable fileNameHash;
    //  For quicker access to the file objects when parsing m3u files
    private Random random;
    private IRandomSelector randomSongSelector;
    private boolean skipSong = false;
    private IPlayHistory playHistory;
    // public InteractiveStream theStream = null;
    private IMetaDataListener metaDataListener;
    private Properties props;

    private long numFilesTotal;
    private long numFilesAdded;
    private MP3File lastSongAdded;
    private String parsingPlayDir;


    /**
     *  Creates a playlist stream.
     *
     *@param  properties  Description of the Parameter
     */
    public InteractiveStream(Properties properties) {
        super(null);
        props = properties;
        queue = new RequestQueue();
        descriptions = true;
        random = new Random();
        availableFiles = new Vector();
        fileHash = new Hashtable();
        fileNameHash = new Hashtable();
    }

    ////
    /**
     *  Description of the Method
     */
    public void initPlayDir() {
        // initialize the randomSongSelector here, so it can be updated when
        // the server is refreshed after its initial load
        configureRandomSongSelector();
        File dir = null;
        MP3File reloadRoot = null;
        MP3File reloadRandomRoot = null;
        // set playDir to null if more than one directory exists
        // in the playdir property
        playDir = props.getProperty("streamsicle.playdir");
        // check if any backslashes are used and warn the user if this is the case
        // these shouldn't be converted to forward slashes because *NIX folks
        // should be allowed to use backslashes for escape characters
        if (playDir.indexOf('\\') != -1) {
            System.err.println("Warning: use of backslashes detected in playdir " +
                    "property -- see configuration file for correct usage");
        }
        // check to make sure no one adds any extraneous ;'s to the end
        // of the directory list
        while (playDir.lastIndexOf(";") == playDir.length() - 1) {
            playDir = playDir.substring(0, playDir.length() - 1);
        }
        ;
        String playDirList = playDir;
        if (playDir.indexOf(";") != -1) {
            // more than one directory was specified, use a virtual directory
            playDir = null;
        }
        if (playDir != null) {
            dir = new File(playDir);
            OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Setting root dir to " + dir);
            // begin pre-scan routine
            long scanStart = System.currentTimeMillis();
            numFilesTotal = 0;
            numFilesAdded = 0;
            quickScanDir(dir);
            long scanEnd = System.currentTimeMillis();
            OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Located " + numFilesTotal + " in " +
                    (float) (scanEnd - scanStart) / (float) 1000 + " seconds");
            // end pre-scan routine
            reloadRoot = new MP3File(dir);
        } else {
            OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Setting root dir to virtual directory");
            reloadRoot = new MP3File(null);
            // dir == null
        }
        reloadRandomRoot = reloadRoot;
        //  Select from any of the files avail

        //  Make some new objects to load the file list into,
        //  later we'll swap them with the real ones
        Vector newFileVector = new Vector();
        Hashtable newFileHash = new Hashtable();
        Hashtable newFileNameHash = new Hashtable();
        if (!reloadRoot.isVirtualDirectory()) {
            newFileHash.put(new Integer(reloadRoot.getFileID()), reloadRoot);
            newFileNameHash.put(new String(reloadRoot.getFile().getAbsolutePath().toLowerCase()), reloadRoot);
            parsingPlayDir = playDir;
            addDirectory(dir, reloadRoot, newFileVector, newFileHash, newFileNameHash);
        } else {
            // iterate through all the directories
            int startIndex = 0;
            // iterate through all the directories
            int endIndex = -1;
            do {
                endIndex = playDirList.indexOf(";", endIndex + 1);

                OrchextraAccessor.log(OrchextraAccessor.INFO, this, "adding " +
                        playDirList.substring(startIndex, endIndex == -1 ? playDirList.length() : endIndex) +
                        " as a mount point");
                parsingPlayDir = playDirList.substring(startIndex, endIndex == -1 ? playDirList.length() : endIndex);
                dir = new File(playDirList.substring(startIndex, endIndex == -1 ? playDirList.length() : endIndex));
                // begin pre-scan routine
                long scanStart = System.currentTimeMillis();
                numFilesTotal = 0;
                numFilesAdded = 0;
                quickScanDir(dir);
                long scanEnd = System.currentTimeMillis();
                OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Located " + numFilesTotal + " in " +
                        (float) (scanEnd - scanStart) / (float) 1000 + " seconds");
                // end pre-scan routine
                addDirectory(dir, reloadRoot, newFileVector, newFileHash, newFileNameHash);
                startIndex = endIndex + 1;
            } while (endIndex != -1);
        }
        // Now replace all of the streaming engine objects with the reloaded
        // file list
        queue.reloadQueue(newFileNameHash);
        fileHash = newFileHash;
        fileNameHash = newFileNameHash;
        availableFiles = newFileVector;
        availableRandomFiles = availableFiles;
        root = reloadRoot;
        randomRoot = reloadRandomRoot;
        if (playHistory != null) {
            playHistory.clearPlayHistory();
        }
        MP3File nowPlaying = getCurrent();
        reloadCurrentlyPlaying(newFileNameHash);
    }


    /**
     *  Description of the Method
     *
     *@param  dir  Description of the Parameter
     */
    private void quickScanDir(File dir) {

        if (dir.isDirectory()) {
            String[] files = dir.list();
            if (files != null) {
                for (int i = 0; i < files.length; i++) {
                    File file = new File(dir, files[i]);
                    if (file.isDirectory()) {
                        quickScanDir(file);
                    } else if (file.getAbsolutePath().toLowerCase().endsWith(Constants.CONST_MP3_FILE_SUFFIX)) {
                        numFilesTotal++;
                    }
                }
            }
        }
    }

    /**
     * When a reload is complete, call this to find the song currently playing
     * and change the currently playing song object.
     * @param fileHash
     */
    private void reloadCurrentlyPlaying(Hashtable fileHash) {
        if (getCurrent() != null) {
            MP3File newNowPlaying =
                    (MP3File) fileHash.get(getCurrent().getFile().getAbsolutePath().toLowerCase());
            if (newNowPlaying != null) {
                currentMedia = newNowPlaying;
            }
        }
    }


    /**
     *  Description of the Method
     */
    public void skip() {
        synchronized (this) {
            skipSong = true;
            while (skipSong) {
                try {
                    wait();
                } catch (InterruptedException e) {}
            }
        }
    }


    /**
     *  Add a set of files recursively to the datastore. Creation date:
     *  (1/14/2001 11:38:04 PM)
     *
     *@param  addDir           java.io.File
     *@param  parent           The feature to be added to the Directory
     *      attribute
     *@param  newFileVector    The feature to be added to the Directory
     *      attribute
     *@param  newFileHash      The feature to be added to the Directory
     *      attribute
     *@param  newFileNameHash  The feature to be added to the Directory
     *      attribute
     */
    public void addDirectory(File addDir, MP3File parent, Vector newFileVector, Hashtable newFileHash, Hashtable newFileNameHash) {
        if (addDir.isDirectory()) {
            String[] files = addDir.list();
            Vector toAdd = new Vector();
            if (files != null) {
                for (int i = 0; i < files.length; i++) {
                    File file = new File(addDir, files[i]);
                    if (file.isDirectory()) {
                        // Add this directory too
                        OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Adding directory " + file.getAbsolutePath());
                        // Make an object to represent this directory
                        MP3File newDir = new MP3File(file, parent);
                        newFileVector.addElement(newDir);
                        newFileHash.put(new Integer(newDir.getFileID()), newDir);
                        // Add this directory to the list of directories contained
                        // by the parent
                        //parent.addChildDirectory(newDir);
                        toAdd.addElement(newDir);
                        // Recurse baby!
                        addDirectory(file, newDir, newFileVector, newFileHash, newFileNameHash);
                        if (newDir.getIsARandomRoot()) {
                            parent.setIsARandomRoot(true);
                        }
                    } else {
                        // Add this file
                        MP3File added = addFile(file, parent, newFileVector, newFileHash, newFileNameHash);
                        if (added != null) {
                            toAdd.addElement(added);
                        }
                    }
                }
            }
            Object[] fileList = new Object[toAdd.size()];
            toAdd.copyInto(fileList);
            QuickSort.sort(fileList);
            for (int i = 0; i < fileList.length; i++) {
                MP3File fileToAdd = (MP3File) fileList[i];
                if (fileToAdd.getFile().isDirectory()) {
                    parent.addChildDirectory(fileToAdd);
                } else {
                    parent.addChildMP3File(fileToAdd);
                }
            }
        } else {
            OrchextraAccessor.log(OrchextraAccessor.SEVERE, this, "Something is wrong with the streamsicle.playdir entry " +
                    "in the fluid.config file.  It might not be set to a directory.");
        }
    }


    /**
     *  Trigger the add routine based on a file input. Creation date: (1/13/2001
     *  11:31:23 PM)
     *
     *@param  newFile          The feature to be added to the File attribute
     *@param  parent           The feature to be added to the File attribute
     *@param  newFileVector    The feature to be added to the File attribute
     *@param  newFileHash      The feature to be added to the File attribute
     *@param  newFileNameHash  The feature to be added to the File attribute
     *@return                  Description of the Return Value
     */
    public MP3File addFile(File newFile, MP3File parent, Vector newFileVector, Hashtable newFileHash, Hashtable newFileNameHash) {
        if (newFile.getAbsolutePath().toLowerCase().endsWith(Constants.CONST_MP3_FILE_SUFFIX)) {
            MP3File fileToAdd = new MP3File(newFile, parent);
            newFileVector.addElement(fileToAdd);
            // Moved for sorting.
            //parent.addChildMP3File(fileToAdd);
            newFileHash.put(new Integer(fileToAdd.getFileID()), fileToAdd);
            newFileNameHash.put(new String(fileToAdd.getFile().getAbsolutePath().toLowerCase()), fileToAdd);
            // add the item to the IRandomSelector object as well if the
            // generic RandomSelector object was instantiated
            randomSongSelector.addToAvailableSongs(fileToAdd.getFileID());
            parent.setIsARandomRoot(true);
            lastSongAdded = fileToAdd;
            numFilesAdded++;
            return fileToAdd;
        }
        return null;
    }


    /**
     *  Gets the lastAddedSong attribute of the InteractiveStream object
     *
     *@return    The lastAddedSong value
     */
    public String getLastAddedSong() {
        String songName = null;
        try {
            songName = lastSongAdded.getName();
        } catch (NullPointerException e) {
            //songName = "nothing!";
        }
        return songName;
    }


    /**
     *  Gets the numSongsAdded attribute of the InteractiveStream object
     *
     *@return    The numSongsAdded value
     */
    public long getNumSongsAdded() {
        return numFilesAdded;
    }


    /**
     *  Gets the numTotalSongs attribute of the InteractiveStream object
     *
     *@return    The numTotalSongs value
     */
    public long getNumTotalSongs() {
        return numFilesTotal;
    }


    /**
     *  Gets the parsingFromDir attribute of the InteractiveStream object
     *
     *@return    The parsingFromDir value
     */
    public String getParsingFromDir() {
        return parsingPlayDir;
    }


    /**
     *  Adds a feature to the Random attribute of the InteractiveStream object
     */
    private void addRandom() {
        File thisFile = null;
        do {
            Integer songID = randomSongSelector.getNextSongID();
            thisFile = ((MP3File) fileHash.get(songID)).getFile();
            if ((!thisFile.isDirectory()) && (!thisFile.getName().endsWith(Constants.CONST_M3U_FILE_SUFFIX))) {
                addMedia(((MP3File) fileHash.get(songID)).getFileID());
            }
        } while ((thisFile.isDirectory()) || (thisFile.getName().endsWith(Constants.CONST_M3U_FILE_SUFFIX)));
    }


    /**
     *  Sets the randomRoot attribute of the InteractiveStream object
     *
     *@param  fileID  The new randomRoot value
     */
    public void setRandomRoot(Integer fileID) {
        randomRoot = (MP3File) fileHash.get(fileID);
        // Now build the list of files available from this root point
        availableRandomFiles = new Vector();
        addToRandomList(randomRoot);
        randomSongSelector.setAvailableFiles(availableRandomFiles);
    }


    /**
     *  Adds a feature to the MediaIn attribute of the InteractiveStream object
     *
     *@param  fileID  The feature to be added to the MediaIn attribute
     */
    public void addMediaIn(Integer fileID) {
        MP3File file = (MP3File) fileHash.get(fileID);
        if ((fileID != null) && (file.getFile().isDirectory())) {
            Vector files = file.getMP3FileChildren();
            for (int i = 0; i < files.size(); i++) {
                if (((MP3File) files.elementAt(i)).getFile().getName().toLowerCase().endsWith(Constants.CONST_MP3_FILE_SUFFIX)) {
                    addMedia(((MP3File) files.elementAt(i)).getFileID(), true);
                }
            }
        }
    }


    /**
     *  Adds a feature to the ToRandomList attribute of the InteractiveStream
     *  object
     *
     *@param  addThis  The feature to be added to the ToRandomList attribute
     */
    private void addToRandomList(MP3File addThis) {
        // check to see if this is a virtual directory
        if (addThis == null) {
            addThis = root;
        }
        if ((addThis.isVirtualDirectory()) || (addThis.getFile().isDirectory())) {
            Vector containedFiles = addThis.getMP3FileChildren();
            for (int i = 0; i < containedFiles.size(); i++) {
                availableRandomFiles.addElement((MP3File) containedFiles.elementAt(i));
            }
            Vector containedDirs = addThis.getDirectoryChildren();
            for (int i = 0; i < containedDirs.size(); i++) {
                addToRandomList((MP3File) containedDirs.elementAt(i));
            }
        } else {
            availableRandomFiles.addElement(addThis);
        }
    }


    /**
     *  Gets the next attribute of the InteractiveStream object
     *
     *@return    The next value
     */
    private MP3File getNext() {
        MP3File next = queue.nextSong();
        if (queue.queueSize() == 0) {
            addRandom();
        }
        return next;
    }


    /**
     *  Gets the randomRoot attribute of the InteractiveStream object
     *
     *@return    The randomRoot value
     */
    public MP3File getRandomRoot() {
        return randomRoot;
    }


    /**
     *  Gets the list attribute of the InteractiveStream object
     *
     *@return    The list value
     */
    public Vector getList() {
        return availableFiles;
    }


    /**
     *  Description of the Method
     *
     *@param  fileID  Description of the Parameter
     */
    public void removeFromQueue(int fileID) {
        queue.removeFromQueue(fileID);
        if (queue.queueSize() == 0) {
            addRandom();
        }
    }


    /**
     *  Description of the Method
     *
     *@param  fileID  Description of the Parameter
     */
    public void toTopOfQueue(int fileID) {
        queue.moveToTop(fileID);
    }

    public void toBottomOfQueue(int fileID) {
        queue.moveToBottom(fileID);
    }

    public void moveInQueue(int fileID, int numSpaces) {
        queue.move(fileID, numSpaces);
    }

    public void randomizeQueue() {
        queue.randomize();
    }


    /**
     *  Sets the play history object that play history info will be sent to.
     *
     *@param  playHistory  The new playHistory value
     */
    public void setPlayHistory(IPlayHistory playHistory) {
        this.playHistory = playHistory;
    }


    /**
     *  Returns the object that is currently being used for play history info.
     *
     *@return    The playHistory value
     */
    public IPlayHistory getPlayHistory() {
        return playHistory;
    }


    /**
     *  Adds a song to the current play history, if there is one.
     *
     *@param  song  The feature to be added to the SongToPlayHistory attribute
     */
    public void addSongToPlayHistory(MP3File song) {
        if (playHistory != null) {
            playHistory.addSong(song);
        }
    }


    /**
     *  Adds a filename to the list of media files that this InteractiveStream
     *  handles.
     *
     *@return            The root value
     */
    public MP3File getRoot() {
        return root;
    }


    /**
     *  Gets the queue attribute of the InteractiveStream object
     *
     *@return    The queue value
     */
    public Vector getQueue() {
        return queue.getQueue();
    }


    /**
     *  Gets the queueList attribute of the InteractiveStream object
     *
     *@return    The queueList value
     */
    public String getQueueList() {
        return queue.toString();
    }


    /**
     *  Gets the hostName attribute of the InteractiveStream object
     *
     *@return    The hostName value
     */
    public String getHostName() {
        return hostName;
    }


    /**
     *  Gets the portNumber attribute of the InteractiveStream object
     *
     *@return    The portNumber value
     */
    public String getPortNumber() {
        return portNumber;
    }


    /**
     *  Gets the hostDescription attribute of the InteractiveStream object
     *
     *@return    The hostDescription value
     */
    public String getHostDescription() {
        return hostDescription;
    }


    /**
     *  Gets the current attribute of the InteractiveStream object
     *
     *@return    The current value
     */
    public MP3File getCurrent() {
        return currentMedia;
    }


    /**
     *  Gets the mP3File attribute of the InteractiveStream object
     *
     *@param  fileID  Description of the Parameter
     *@return         The mP3File value
     */
    public MP3File getMP3File(int fileID) {
        return (MP3File) fileHash.get(new Integer(fileID));
    }


    /**
     *  Gets the mP3File attribute of the InteractiveStream object
     *
     *@param  fileName  Description of the Parameter
     *@return           The mP3File value
     */
    public MP3File getMP3File(String fileName) {
        return (MP3File) fileNameHash.get(fileName.toLowerCase());
    }


    /**
     *  Configures the playlist stream. The name of the playlist is read from
     *  the configuration file and the the filenames from that list is loaded.
     *  <P>
     *
     *  The properties that are read from the configuration are: <P>
     *
     *
     *  <UL>
     *    <LI> <B> playliststream.playlist </B> <BR>
     *    The name of the file containing the playlist
     *    <LI> <B> playliststream.loop </B> <BR>
     *    true or false, depending on whether the stream should loop when the
     *    end of the playlist has been reached
     *    <LI> <B> playliststream.first </B> <BR>
     *    The offset number of the first media to play from the playlist
     *    <LI> <B> playliststream.random </B> <BR>
     *    true or false, depending on whether next media to be played should be
     *    picked randomly or in a secquence (setting this to true will imply
     *    loop mode as well)
     *    <LI> <B> playliststream.descriptions </B> <BR>
     *    true or false, depending on whether you want descriptions of the media
     *    files to be printed to standard output
     *  </UL>
     *
     *
     *@exception  StreamingEngineException  Description of the Exception
     */
    public void configure() throws StreamingEngineException {
        try {
            hostName = props.getProperty("streamsicle.host");
            hostDescription = props.getProperty("streamsicle.description");
            portNumber = props.getProperty("streamsicle.port");
            descriptions = props.getProperty("streamsicle.descriptions").equals("true");
            initPlayDir();
            // if initPlayDir didn't add any songs, streamsicle will crash when
            // the random selector tries to add a random song from an available list
            // of songs totaling zero.  catch this exception and gracefully exit.
            try {
                addRandom();
                addRandom();
            } catch (java.lang.ArithmeticException e) {
                // Division by zero
                String message = "The directory specified in the configuration file " +
                        "contained no songs!  Select a valid directory and try again.";
                OrchextraAccessor.log(OrchextraAccessor.SEVERE, this, message);
                StreamingEngineException ex = new StreamingEngineException();
                ex.setUserUnderstandableMessage(message);
                throw ex;
            }
            setCurrent();
            start();
        } catch (StreamingEngineException e) {
            // Just pass this badboy right along.
            throw e;
        } catch (Exception e) {
            OrchextraAccessor.log(OrchextraAccessor.SEVERE, this, "InteractiveStream could not be configured.");
            StreamingEngineException ex = new StreamingEngineException();
            ex.setUserUnderstandableMessage("Couldn't start the stream for an unknown reason.");
            throw ex;
        }
    }


    /**
     *  Called after a GUI reconfiguration.
     */
    public void reconfigure() {
        hostName = props.getProperty("streamsicle.host");
        hostDescription = props.getProperty("streamsicle.description");
        playDir = props.getProperty("streamsicle.playdir");
    }


    /**
     *  Configures the randomSongSelector object
     */
    private void configureRandomSongSelector() {
        String randomizer = props.getProperty("streamsicle.randomizer");
        try {
            randomSongSelector = (IRandomSelector) Class.forName(randomizer).newInstance();
        } catch (Exception e) {
            OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "IRandomSelector could not be configured, using the default");
            e.printStackTrace();
            randomSongSelector = new RandomSelector();
        }
    }


    /**
     *  Adds a filename to the list of media files that this InteractiveStream
     *  handles.
     *
     *@param  fileID     The feature to be added to the Media attribute
     */
    public void addMedia(int fileID) {
        MP3File fileToAdd = (MP3File) fileHash.get(new Integer(fileID));
        QueueItem item = new QueueItem(fileToAdd);
        queue.addToQueue(item);
    }


    /**
     *  Add a file to the queue, specifying that it was a request. Notify the
     *  randomSelector that this song with songID fileID has been requested.
     *  Also, boot any file that was randomly selected and replace it with the
     *  request
     *
     *@param  fileID   The feature to be added to the Media attribute
     *@param  request  The feature to be added to the Media attribute
     */
    public void addMedia(int fileID, boolean request) {
        MP3File fileToAdd = (MP3File) fileHash.get(new Integer(fileID));
        if (fileToAdd != null) {
            QueueItem newItem = new QueueItem(fileToAdd, request);
            queue.addToQueue(newItem);
        }
        randomSongSelector.addRequestedSongID(fileID);
    }
	
	/**
     *  Add a file to the queue, specifying that it was a request. Notify the
     *  randomSelector that this song with songID fileID has been requested.
     *  Also, boot any file that was randomly selected and replace it with the
     *  request
     *
     *@param  fileID   The feature to be added to the Media attribute
     *@param  request  The feature to be added to the Media attribute
     */
    public void addMedia(int fileID, String owner, boolean request) {
        MP3File fileToAdd = (MP3File) fileHash.get(new Integer(fileID));
        if (fileToAdd != null) {
            QueueItem newItem = new QueueItem(fileToAdd, owner, request);
            queue.addToQueue(newItem);
        }
        randomSongSelector.addRequestedSongID(fileID);
    }
	

    /**
     *  Filters which files will be read from the playlist. In its current form
     *  the playlist may contain comments as long as they start with a #. Blank
     *  lines or file names without an extension will not be regarded as valid
     *  files.
     *
     *@param  str  A string to be determined if it is a filename or not
     *@return      Description of the Return Value
     */
    public boolean blank(String str) {
        // Comments start with # and all files must have an extension
        return (str.startsWith("#") || !(str.lastIndexOf('.') > 1));
    }


    /**
     *  The delay that the MediaInputStream will have to wait before reading the
     *  next packet.
     *
     *@return    Between-packet delay or null if there are no more packets to be
     *      read
     */
    public Delay getDelay() {
        // no more files to play (never happens in loop mode)
        if (current == null) {
            return null;
        }
        return current.getDelay();
    }


    /**
     *  Reads a packet from the current MediaInputStream. This method will
     *  internally switch to the next file in the playlist when the end of a
     *  file has been reached. If all media files are corrupted an exception
     *  will be thrown.
     *
     *@return                  A packet read from the media file using the
     *      corresponding read() method in the MediaInputStream handling that
     *      type. Returns null if the end of the files has been reached and the
     *      loop mode is not switched on.
     *@exception  IOException  Description of the Exception
     */
    public byte[] read() throws IOException {
        // no more files to play (never happens in loop mode)
        if (current == null) {
            return null;
        }
        // if all media files are unreadable
        /*
         *  if(numBadMedia >= mediaNames.size())
         *  throw new IOException( "All media files are unreadable" );
         */
        if (skipSong) {
            setCurrent();
            skipSong = false;
            synchronized (this) {
                notifyAll();
            }
        }
        byte[] packet = null;
        // read a packet
        try {
			packet = current.read();
        }
        // the file could not be read
        catch (IOException e) {
			System.err.println("Problem reading source: "+e.getMessage());
            setCurrent();
        }
        // we reached the end of the file
        if (packet == null) {
            setCurrent();
        }
        // all is well
        else {
            return packet;
        }
        // something bad happened, try next file
        return read();
    }


    /**
     *  Sets the current media file to the next in the queue. <P>
     *
     *  The type of stream that this media should be wrapped in is decided
     *  through dynamic class loading and checking of the handlesMedia() method.
     *  <P>
     *
     *  Furthermore, if there is an associated descriptor for this media type,
     *  and descriptions are switched on, it will be instantiated and a
     *  description printed out for possible logging. If no matching descriptor
     *  is found the file name will be printed as a last resort.
     *
     */
    public void setCurrent() {
        MediaInputStream stream = null;
        while (stream == null) {
            MP3File mediaFile = getNext();
            MediaInputStream handler = getHandler(mediaFile.getFile().getAbsolutePath());
            // no handler for this media type considered as bad media
            if (handler != null) {
                // add song to play history before we lose the reference
                addSongToPlayHistory(currentMedia);

                currentMedia = mediaFile;
                sendMetaData(currentMedia.getName());
                if (descriptions) {
                    String desc = getDescription(mediaFile.getFile().getAbsolutePath());
                    //OrchextraAccessor.log(OrchextraAccessor.INFO, this, "<< " + currentMedia.getName());
                }
                stream = handler;
                current = stream;
            }
        }
    }


    /**
     *  Set the MetaDataListener for this stream. Since only one listener is
     *  currently supported, calling this method will replace the previous
     *  listener, if it has been set.
     *
     *@param  listener  The new metaDataListener value
     */
    public void setMetaDataListener(IMetaDataListener listener) {
        // only support one listener for now
        metaDataListener = listener;
    }


    /**
     *  Description of the Method
     *
     *@param  streamTitle  Description of the Parameter
     */
    private void sendMetaData(String streamTitle) {
        if (metaDataListener != null) {
            // DSH TODO: finish this function
            StringBuffer metaDataStrBuf = new StringBuffer();
            metaDataStrBuf.append("StreamTitle=\'");
            metaDataStrBuf.append(streamTitle);
            metaDataStrBuf.append("\';StreamUrl='';");
            // stream url unused

            int metaDataSize = metaDataStrBuf.length();
            int metaDataSizeRounded;
            if ((metaDataSize % 16) == 0) {
                metaDataSizeRounded = metaDataSize;
            } else {
                metaDataSizeRounded = metaDataSize + (16 - (metaDataSize % 16));
            }

            byte[] metaDataBytes = new byte[metaDataSizeRounded + 1];

            // assume that meta data is never larger than 16 * 256 bytes
            metaDataBytes[0] = (byte) (metaDataSizeRounded / 16);

            System.arraycopy(metaDataStrBuf.toString().getBytes(), 0,
                    metaDataBytes, 1, metaDataSize);

            // pad with 0's - - the '+1's are for the size byte in the beginning
            for (int i = metaDataSize + 1; i < metaDataSizeRounded + 1; i++) {
                metaDataBytes[i] = 0;
            }

            metaDataListener.setMetaData(metaDataBytes);
        }
    }


    /**
     *  The playlist in itself will not handle any kind of media.
     *
     *@param  type  Description of the Parameter
     *@return       Always false
     */
    public boolean handlesMedia(String type) {
        return false;
    }


    /**
     *  Finds the handler associated with the specified media file. Using the
     *  extension of the given media filename and dynamic loading the
     *  MediaInputStream subclasses will be probed using the individual
     *  handlesMedia() method. If one is found that handles the kind of media
     *  indicated by the extension it will be loaded and instantiated.
     *
     *@param  mediaName  A media filename
     *@return            A MediaInputStream wrapped around the file handling
     *      this kind of media
     */
    public MediaInputStream getHandler(String mediaName) {
        String extension = mediaName.substring(mediaName.lastIndexOf('.') + 1, mediaName.length());
        String streamName = props.getProperty("fluid.stream");
        try {
            Class sClass = Class.forName(streamName);
            MediaInputStream mis = (MediaInputStream) sClass.newInstance();
            if (mis.handlesMedia(extension)) {
                mis.setInputStream(new FileInputStream(mediaName));
                mis.setProperties(props);
                mis.configure();
                return mis;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     *  Finds the descriptor associated with this media type and returns the
     *  description of the same.
     *
     *@param  mediaName  A media filename
     *@return            A description of the media or null if no suitable
     *      descriptor was found
     */
    public String getDescription(String mediaName) {
        String extension = mediaName.substring(mediaName.lastIndexOf('.') + 1, mediaName.length());
        String descriptor = props.getProperty("fluid.descriptor");
        try {
            /*
			Class dClass = Class.forName(descriptor);
			//Class dClass = String.class.getClassLoader().loadClass(descriptor);
			java.lang.reflect.Constructor c[]=dClass.getConstructors();
			Object arg[]=new Object[1];
			arg[0]=mediaName;
			//Descriptor desc = (Descriptor) dClass.newInstance();
			Descriptor desc= (Descriptor) c[0].newInstance(arg);
			if (desc.describes(extension)) {
                return desc.describe(mediaName);
            }
			 */
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
}


