/*
 * WeightedRandomSelector.java
 *
 * This IRandomSelector implementation provides random songs based on a
 * weighted selection scheme.  In short, the more a song is requested, the more
 * often it will be randomly selected.  In addition, songs requested more
 * recently will have a higher weight than those requested some time ago.
 * Using these two selection schemes together should more or less provide a
 * radio station like selection routine.
 *
 * The long version: songs are chosen based on last request time and on number
 * of requests.  A time sum of all requested songs (in seconds) is computed, and
 * a random number is generated between 0 and the sum of this computed sum.  All
 * items with a lastRequestedTime value greater than this computed value are
 * included in a fair-game list.  The sum of all the number of times
 * requested values is computed based on the items in this fair-game list, and
 * a random number is chosen again.  The item on the list that is less than or
 * equal to the random generated value is chosen.
 *
 * Of course, you don't want the same item to be chosen over and over again.
 * Enter the history object.  If the item is on this list, then it's too soon for
 * it to be chosen again.  The WeightedRandomSelector then falls back on an
 * internally kept RandomSelector, and returns back whatever value the
 * RandomSelector generates.
 *
 * Created on February 16, 2002, 8:17 PM
 */

package com.streamsicle;

import java.math.BigInteger;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Random;
import java.util.Vector;

public class WeightedRandomSelector implements IRandomSelector {

    // internal class used to manage a song's last request time and request freq.
    private class WeightedSelectionItem {
        public Integer songID;
        public int timesRequested = 0;
        public long lastRequestedTime;
    }

    // 12 * 5:00 = 1 hr.  figuring that an average "song" lasts about 5 min,
    // the same song probably shouldn't be heard more than once an hour or so
    private final int NUM_HISTORY_ITEMS = 12;

    // the generic random selector is used when when no suitable weighted
    // song can be generated
    RandomSelector randomSelector = new RandomSelector();
    // a hashtable of songID, WeightedSelectionItem(wsi) key-value pairs
    private Hashtable availableSongs = new Hashtable();
    private Random randomGen = new Random();
    // the first item requested is used as a base time for all other items
    // requested afterwards
    private long initTime = -1;
    // if a weighted song can't be calculated, a default wsi object is generated
    // and used instead
    private WeightedSelectionItem defaultWSI = null;
    // the last NUM_HISTORY_ITEM songs chosen by this random selector
    private Vector songHistory = new Vector();

    // this populates the list used by the random selector to return a random
    // song when the weighted random selector can't compute one
    public void addToAvailableSongs(int songID) {
        randomSelector.addToAvailableSongs(songID);
    }

    /** Notify this randomSelector that the song with songID was just requested
     * by a user
     */
    public void addRequestedSongID(int songID) {
        Integer songIDInteger = new Integer(songID);
        WeightedSelectionItem wsi =
            (WeightedSelectionItem)availableSongs.get(songIDInteger);
        if (wsi == null) {
            // this is the first time this item has been requested, so make a
            // new WeightedSelectionItem for it
            wsi = new WeightedSelectionItem();
            wsi.songID = songIDInteger;
        }
        wsi.timesRequested++;
        // initTime is used as an offset for all other times computed.
        // it prevents this class from ever having generate random BigIntegers
        if (initTime == -1) {
            wsi.lastRequestedTime = 0;
            initTime = (new Date().getTime()) / 1000; // msec -> sec conversion
        } else {
            wsi.lastRequestedTime = (new Date().getTime() - initTime) / 1000;
        }
        updateHashtable(wsi);
        updateSongHistory(wsi.songID.intValue());
    }

    public void setAvailableFiles(Vector availableFiles) {
     randomSelector.setAvailableFiles(availableFiles);
     // update the weighted selection list as well
     availableSongs = new Hashtable();
     initTime = -1;
    }

    // update the songHistory object
    // remove the oldest item and add songID as the newest
    private void updateSongHistory(int songID) {
        if (songHistory.size() == 6) {
            songHistory.removeElementAt(0);
        }
        songHistory.addElement(new Integer(songID));
    }

    // update the hashtable used by this WeightedRandomSelector
    private void updateHashtable(WeightedSelectionItem wsi) {
        availableSongs.remove(wsi.songID);
        availableSongs.put(wsi.songID, wsi);
    }

    /** Request a new song to add to Streamsicle's queue
     */
    public Integer getNextSongID() {
        if (availableSongs.size() <= 1) {
            // only one or fewer have been requested, return a random ID instead
            Integer tempSongID = randomSelector.getNextSongID();
            updateSongHistory(tempSongID.intValue());
            return tempSongID;
        }
        long randomRequestTime = generateRandomTimeValue();
        if (randomRequestTime == -1) {
            // the random root was just reset, return a random value
            Integer tempSongID = randomSelector.getNextSongID();
            updateSongHistory(tempSongID.intValue());
            return tempSongID;
        }
        // get the values from the hashtable that are applicable based on the
        // generated random value
        Vector validTimeValues = getItemsBasedOnTime(randomRequestTime);
        if (validTimeValues.size() == 0) {
            // the generated value was more recent than all the values
            // try to use the default, most recently requested value
            for (Enumeration enum = songHistory.elements(); enum.hasMoreElements(); ) {
                if ( (enum.nextElement()).equals(defaultWSI.songID)) {
                    // the generated songID is in the playhistory list,
                    // so return a random one
                    Integer tempSongID = randomSelector.getNextSongID();
                    updateSongHistory(tempSongID.intValue());
                    return tempSongID;
                }
            }
            // otherwise, return the default songID we saved earlier
            updateSongHistory(defaultWSI.songID.intValue());
            return defaultWSI.songID;
        }
        // return a value from the list based on num. request times
        Integer mostRequestedSongID = getMostRequested(validTimeValues);
        if (mostRequestedSongID != null) {
            updateSongHistory(mostRequestedSongID.intValue());
            return mostRequestedSongID;
        }
        // no songs met the required criteria, return a random one
        Integer tempSongID = randomSelector.getNextSongID();
        updateSongHistory(tempSongID.intValue());
        return tempSongID;
    }

    // return a song id based on the selection criteria described above
    private Integer getMostRequested(Vector validTimeValues) {
        WeightedSelectionItem tempWSI;
        defaultWSI = new WeightedSelectionItem();
        defaultWSI.timesRequested = Integer.MAX_VALUE;
        int requestThreshold = generateRequestValue(validTimeValues);
        for (Enumeration valuesEnum = validTimeValues.elements(); valuesEnum.hasMoreElements(); ) {
            tempWSI = (WeightedSelectionItem)valuesEnum.nextElement();
            if ((tempWSI.timesRequested > requestThreshold) &&
                (tempWSI.timesRequested < defaultWSI.timesRequested) ) {
                   defaultWSI = tempWSI;
            }
        }
        for (Enumeration enum = songHistory.elements(); enum.hasMoreElements(); ) {
            if ( (enum.nextElement()).equals(defaultWSI.songID)) {
                Integer tempSongID = randomSelector.getNextSongID();
                updateSongHistory(tempSongID.intValue());
                return tempSongID;
            }
        }
        return defaultWSI.songID;
    }

    // pull all of the values out of the song list based on randomRequestTime
    private Vector getItemsBasedOnTime(long randomRequestTime) {
        WeightedSelectionItem tempWSI;
        defaultWSI = new WeightedSelectionItem();
        defaultWSI.lastRequestedTime = Long.MIN_VALUE;
        Vector validTimeValues = new Vector();
        for (Enumeration valuesEnum = availableSongs.elements(); valuesEnum.hasMoreElements(); ) {
            tempWSI = (WeightedSelectionItem)valuesEnum.nextElement();
            if (tempWSI.lastRequestedTime > randomRequestTime) {
                validTimeValues.addElement(tempWSI);
            }
            // see if the last time this item was requested is more recent
            // than what the defaultWSI currently has.  if so, swap it
            if (tempWSI.lastRequestedTime > defaultWSI.lastRequestedTime) {
                defaultWSI = tempWSI;
            }
        }
        return validTimeValues;
    }

    // generate a random time value based on the sums of all the items known
    private long generateRandomTimeValue() {
        // first things first, figure out the sum of all of the time elements
        // present in the hashtable
        WeightedSelectionItem wsi;
        long timeSum = 0;
        for (Enumeration valuesEnum = availableSongs.elements(); valuesEnum.hasMoreElements(); ) {
            wsi = (WeightedSelectionItem)valuesEnum.nextElement();
            timeSum += wsi.lastRequestedTime;
        }
        if (timeSum == 0) {
            return -1;
        }
        return Math.abs(randomGen.nextLong()) % timeSum;
    }

    // generate a random request value based on the sums of all values in the
    // validTimeValues list
    private int generateRequestValue(Vector validTimeValues) {
        // find the sums of all the elements first
        WeightedSelectionItem wsi;
        int sumSelectionCount = 0;
        for (Enumeration valuesEnum = availableSongs.elements(); valuesEnum.hasMoreElements(); ) {
            wsi = (WeightedSelectionItem)valuesEnum.nextElement();
            sumSelectionCount += wsi.timesRequested;
        }
        return Math.abs(randomGen.nextInt()) % sumSelectionCount;
    }
}
