package com.streamsicle;

import com.streamsicle.fluid.*;
import com.streamsicle.util.Constants;
import java.io.*;
import java.util.*;
// Import log4j classes.
import org.ten60.orchextra.*;
import com.streamsicle.songinfo.SongInfo;

/**
 *  Title: MP3File Description: Contains all the info about an MP3 file we'll
 *  ever need to know
 *
 *@author     Matt Hall
 *@author     John Watkinson
 *@created    September 17, 2002
 */

public class MP3File {
    // ID3Tag replaced by SongInfo classes
    // private ID3Tag id3tag;
    private SongInfo songInfo;

    private File theFile;
    private int fileID;

    // And now for the structure of the directories storing these files
    // This stuff will just be empty for an actual file because obviously
    // a file can't contain other files or directories.
    Vector containsFiles;
    // Any files that are in this directory
    Vector containsDirectories;
    // Any directories that are in this directory
    MP3File parent;
    // The parent object to this one
    boolean virtualDirectory;
    boolean aRandomRoot;


    /**
     *  Constructor for the MP3File object
     *
     *@param  pFile  Description of the Parameter
     */
    public MP3File(File pFile) {
        setParent(this);
        if (pFile != null) {
            virtualDirectory = false;
        } else {
            virtualDirectory = true;
        }
        init(pFile);
        aRandomRoot = false;
    }


    /**
     *  Constructor for the MP3File object
     *
     *@param  pFile   Description of the Parameter
     *@param  parent  Description of the Parameter
     */
    public MP3File(File pFile, MP3File parent) {
        setParent(parent);
        virtualDirectory = false;
        init(pFile);
        aRandomRoot = false;
    }


    /**
     *  Description of the Method
     *
     *@param  pFile  Description of the Parameter
     */
    public void init(File pFile) {
        this.theFile = pFile;
        this.fileID = Sequence.getSequence().nextVal();

        containsFiles = new Vector();
        containsDirectories = new Vector();

        if (!virtualDirectory) {
            if ((!theFile.isDirectory()) && (!theFile.getName().endsWith(Constants.CONST_M3U_FILE_SUFFIX))) {
                // ID3Tag replaced by SongInfo classes
                // id3tag = new ID3Tag(pFile.getAbsolutePath());
                try {
                    songInfo = SongInfo.createSongInfo(pFile);
                } catch (java.lang.ArrayIndexOutOfBoundsException e) {
                    // bad id3v2 tag
                    try {
                        OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "There was a problem reading the id3v2 tag "
                                + "in the file " + pFile.getCanonicalPath());
                    } catch (java.io.IOException e2) {
                        // screw it -- too many exceptions thrown -- just continue
                    }
                }
            }
        }
    }


    /**
     *  Gets the file attribute of the MP3File object
     *
     *@return    The file value
     */
    public File getFile() {
        return theFile;
    }


    /**
     *  Gets the fileID attribute of the MP3File object
     *
     *@return    The fileID value
     */
    public int getFileID() {
        return fileID;
    }


    /**
     *  Sets the parent attribute of the MP3File object
     *
     *@param  pMP3File  The new parent value
     */
    public void setParent(MP3File pMP3File) {
        this.parent = pMP3File;
    }


    /**
     *  Gets the parent attribute of the MP3File object
     *
     *@return    The parent value
     */
    public MP3File getParent() {
        return this.parent;
    }


    /**
     *  Adds a feature to the ChildMP3File attribute of the MP3File object
     *
     *@param  pMP3File  The feature to be added to the ChildMP3File attribute
     */
    public void addChildMP3File(MP3File pMP3File) {
        containsFiles.addElement(pMP3File);
    }


    /**
     *  Gets the mP3FileChildren attribute of the MP3File object
     *
     *@return    The mP3FileChildren value
     */
    public Vector getMP3FileChildren() {
        return containsFiles;
    }


    /**
     *  Adds a feature to the ChildDirectory attribute of the MP3File object
     *
     *@param  pMP3File  The feature to be added to the ChildDirectory attribute
     */
    public void addChildDirectory(MP3File pMP3File) {
        containsDirectories.addElement(pMP3File);
    }


    /**
     *  Gets the directoryChildren attribute of the MP3File object
     *
     *@return    The directoryChildren value
     */
    public Vector getDirectoryChildren() {
        return containsDirectories;
    }


    /**
     *  Gets the songInfo attribute of the MP3File object
     *
     *@return    The songInfo value
     */
    public SongInfo getSongInfo() {
        return songInfo;
    }


    /**
     *  Gets the isARandomRoot attribute of the MP3File object
     *
     *@return    The isARandomRoot value
     */
    public boolean getIsARandomRoot() {
        return aRandomRoot;
    }


    /**
     *  Sets the isARandomRoot attribute of the MP3File object
     *
     *@param  randomRoot  The new isARandomRoot value
     */
    public void setIsARandomRoot(boolean randomRoot) {
        aRandomRoot = randomRoot;
    }


    /**
     *  getName gets either the ID3 name if found, or just the damn filename
     *
     *@return    The name value
     */

    public String getName() {
        String name = "";
        if (virtualDirectory) {
            return com.streamsicle.util.Constants.CONST_VIRTUAL_DIRECTORY;
        } else {
            if ((songInfo == null) || (songInfo.toString() == null)) {
                // this should only be the case for directories
                if (theFile != null) {
                    if (theFile.getName() != null) {
                        return theFile.getName();
                    }
                }
            } else {
                return songInfo.toString();
            }
        }
        // virtual directory
        return name;
    }


    /**
     *  Gets the album attribute of the MP3File object
     *
     *@return    The album value
     */
    public String getAlbum() {
        // needed to add this to be able to search for album-matches
        if ((songInfo != null) && (songInfo.hasAlbum())) {
            return songInfo.getAlbum();
        } else {
            return "";
        }
    }


    /**
     *  Gets the artist attribute of the MP3File object
     *
     *@return    The artist value
     */
    public String getArtist() {
        if ((songInfo != null) && (songInfo.hasArtist())) {
            return songInfo.getArtist();
        } else {
            return "";
        }
    }


    /**
     *  Gets the timeLength attribute of the MP3File object
     *
     *@return    The timeLength value
     */
    public String getTimeLength() {
        String formattedTime;
        if ((songInfo != null) && (songInfo.hasTimeLength())) {
            Integer songTime = new Integer(songInfo.getTimeLength());
            Calendar cal = new GregorianCalendar();
            cal.clear();
            cal.set(Calendar.SECOND, songTime.intValue() / 1000);
            String minuteVal = "";
            String secondVal = "";

            minuteVal = "" + cal.get(Calendar.MINUTE);
            if (cal.get(Calendar.SECOND) < 10) {
                secondVal = "0" + cal.get(Calendar.SECOND);
            } else {
                secondVal = "" + cal.get(Calendar.SECOND);
            }
            formattedTime = minuteVal + ":" + secondVal;
            if (cal.get(Calendar.HOUR) > 0) {
                if ((int) cal.get(Calendar.MINUTE) == 0) {
                    formattedTime = "0" + formattedTime;
                }
                formattedTime = cal.get(Calendar.HOUR) + ":" + formattedTime;
            }
            return formattedTime;
        } else {
            return "";
        }
    }


    /**
     *  Gets the virtualDirectory attribute of the MP3File object
     *
     *@return    The virtualDirectory value
     */
    public boolean isVirtualDirectory() {
        return virtualDirectory;
    }


    /**
     *  Returns the name of the file (used for sorting).
     *
     *@return    Description of the Return Value
     */
    public String toString() {
        if (theFile != null) {
            return theFile.getName();
        } else {
            return com.streamsicle.util.Constants.CONST_VIRTUAL_DIRECTORY;
        }
    }
}
