/*******************************************************************************
 * Copyright (c) 2009 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementSource;
import org.eclipse.emf.mint.internal.ui.actions.OpenGeneratedAction;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISources;
import org.eclipse.ui.actions.CompoundContributionItem;
import org.eclipse.ui.menus.IMenuService;
import org.eclipse.ui.menus.IWorkbenchContribution;
import org.eclipse.ui.services.IServiceLocator;

/**
 * Abstract base for dynamic menu contributions that provide actions to open
 * generated artifacts related to the current selection, whatever it may be.
 * 
 * <p>
 * Clients may extend this class.
 * </p>
 */
public abstract class AbstractOpenGeneratedMenu extends
		CompoundContributionItem implements IWorkbenchContribution {

	private static final IContributionItem[] NO_ITEMS = {};

	private static final int DEFAULT_MIN_CATEGORY_ITEMS = 2;

	/**
	 * Reference to the service locator obtained in {@link #initialize(IServiceLocator)}.
	 */
	protected IServiceLocator serviceLocator;

	private List<List<IAction>> actions;

	/**
	 * Cached adapter factory reference.
	 */
	protected AdapterFactory adapterFactory;

	/**
	 * Default constructor.
	 */
	protected AbstractOpenGeneratedMenu() {
		super();
	}

	/**
	 * Constructs this menu with the given identifier.
	 * 
	 * @param id menu identifier
	 */
	protected AbstractOpenGeneratedMenu(String id) {
		super(id);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.menus.IWorkbenchContribution#initialize(org.eclipse.ui.services.IServiceLocator)
	 */
	public void initialize(IServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.actions.CompoundContributionItem#getContributionItems()
	 */
	@Override
	protected final IContributionItem[] getContributionItems() {
		if (actions == null) {
			actions = new LinkedList<List<IAction>>();
			IMenuService svc = (IMenuService) serviceLocator
					.getService(IMenuService.class);
			Object selection = svc.getCurrentState().getVariable(
					ISources.ACTIVE_MENU_SELECTION_NAME);
			if (selection instanceof ISelection) {
				Object target = getTarget((ISelection) selection);
				if (target != null) {
					AdapterFactory adapterFactory = getAdapterFactory(target);
					if (adapterFactory != null) {
						Object adapter = adapterFactory.adapt(target,
								IItemJavaElementSource.class);
						if (adapter instanceof IItemJavaElementSource)
							createActions((IItemJavaElementSource) adapter,
									target);
					}
				}
			}
		}

		if (actions.isEmpty())
			return NO_ITEMS;

		ArrayList<IContributionItem> items = new ArrayList<IContributionItem>();
		boolean needsSeparator = false;
		for (List<IAction> group : actions) {
			if (needsSeparator)
				items.add(new Separator());
			else
				needsSeparator = true;

			for (IAction action : group)
				items.add(new ActionContributionItem(action));
		}

		return items.toArray(new IContributionItem[items.size()]);
	}

	/**
	 * Given the selection, returns the object whose generated artifacts
	 * to include in the menu. Clients may override to "unwrap" the selection
	 * in a model-specific way. By default, the first element of the structured
	 * selection is returned.
	 *  
	 * @param selection current selection
	 * @return object whose artifacts to collect
	 */
	protected Object getTarget(ISelection selection) {
		if (selection instanceof IStructuredSelection)
			return ((IStructuredSelection) selection).getFirstElement();

		return null;
	}

	/**
	 * Returns the adapter factory suitable for the given object. The default
	 * implementation calls {@link #createAdapterFactory(Object)} the first time
	 * this method is called, caches the result in {@link #adapterFactory} and
	 * returns it on every subsequent call (regardless of the target).
	 * 
	 * @param target object for which to get the adapter factory
	 * @return adapter factory suitable for the given object
	 */
	protected AdapterFactory getAdapterFactory(Object target) {
		if (adapterFactory == null)
			adapterFactory = createAdapterFactory(target);

		return adapterFactory;
	}

	/**
	 * Creates an instance of an adapter factory suitable for the given target.
	 * The default implementation creates a composed adapter factory using
	 * the global descriptor registry.
	 * 
	 * @param target object for which to create the adapter factory
	 * @return newly created adapter factory
	 */
	protected AdapterFactory createAdapterFactory(Object target) {
		return new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);
	}

	/**
	 * Creates a comparator used for sorting the menu items. The default
	 * implementation returns a new {@link DescriptorComparator}.
	 * 
	 * @param target object for which to create the comparator
	 * @return new comparator
	 */
	protected Comparator<? super IItemJavaElementDescriptor> createDescriptorComparator(
			Object target) {
		return new DescriptorComparator(target);
	}

	/**
	 * Returns the minimum number of menu items per category. The default is 2.
	 * 
	 * @return minimum number of menu items per category
	 */
	protected int getMinCategoryItems() {
		return DEFAULT_MIN_CATEGORY_ITEMS;
	}

	private void createActions(IItemJavaElementSource provider, Object target) {
		List<IItemJavaElementDescriptor> descriptors = new ArrayList<IItemJavaElementDescriptor>(
				provider.getJavaElementDescriptors(target));
		Collections.sort(descriptors, createDescriptorComparator(target));

		LinkedList<IAction> group = null;
		String category = null;
		boolean keepGroups = false;
		int categoryItemCount = 0;
		int minCategoryItems = getMinCategoryItems();
		for (IItemJavaElementDescriptor descriptor : descriptors) {
			String newCategory = descriptor.getCategory(target);
			if (group == null
					|| (category == null ? newCategory != null : !category
							.equals(newCategory))) {
				category = newCategory;
				categoryItemCount = 0;
				group = new LinkedList<IAction>();
				actions.add(group);
			}

			IAction action = createAction(descriptor, target);
			group.add(action);

			if (++categoryItemCount > minCategoryItems)
				keepGroups = true;
		}

		if (!keepGroups) {
			Iterator<List<IAction>> i = actions.iterator();
			if (i.hasNext()) {
				List<IAction> masterGroup = i.next();
				while (i.hasNext()) {
					masterGroup.addAll(i.next());
					i.remove();
				}
			}
		}
	}

	/**
	 * Creates an action for opening a particular generated artifact for a given
	 * target object.
	 * 
	 * @param descriptor descriptor representing the artifact to open
	 * @param target object whose artifact to open
	 * @return action to open the given artifact
	 */
	protected IAction createAction(IItemJavaElementDescriptor descriptor,
			Object target) {
		return new OpenGeneratedAction(descriptor, target);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.action.ContributionItem#dispose()
	 */
	@Override
	public void dispose() {
		if (adapterFactory instanceof IDisposable) {
			((IDisposable) adapterFactory).dispose();
			adapterFactory = null;
		}

		actions = null;
		super.dispose();
	}

	/**
	 * Default comparator used for sorting java element descriptors.
	 */
	protected static class DescriptorComparator implements
			Comparator<IItemJavaElementDescriptor> {

		/**
		 * Cached target object.
		 */
		protected final Object target;

		/**
		 * Creates an instance for the given target.
		 * 
		 * @param target object to which the compared artifact descriptors belong
		 */
		public DescriptorComparator(Object target) {
			this.target = target;
		}

		/* (non-Javadoc)
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(IItemJavaElementDescriptor o1,
				IItemJavaElementDescriptor o2) {
			String category1 = o1.getCategory(target);
			String category2 = o2.getCategory(target);
			if (category1 == null) {
				if (category2 != null)
					return -1;
			} else {
				if (category2 == null)
					return 1;

				int result = category1.compareTo(category2);
				if (result != 0)
					return result;
			}

			String name1 = o1.getDisplayName(target);
			String name2 = o2.getDisplayName(target);
			if (name1 == null)
				return name2 == null ? 0 : -1;

			if (name2 == null)
				return 1;

			return name1.compareTo(name2);
		}
	}
}