"use strict";
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoteStrategy = exports.RemoteTraversal = exports.RemoteConnection = void 0;
/**
 * @author Jorge Bay Gondra
 */
const bytecode_js_1 = __importDefault(require("../process/bytecode.cjs"));
const traversal_strategy_js_1 = require("../process/traversal-strategy.cjs");
const traversal_js_1 = require("../process/traversal.cjs");
/**
 * Represents an abstraction of a "connection" to a "server" that is capable of processing a traversal and
 * returning results.
 */
class RemoteConnection {
    url;
    options;
    /**
     * @param {String} url The resource uri.
     * @param {RemoteConnectionOptions} [options] The connection options.
     */
    constructor(url, options = {}) {
        this.url = url;
        this.options = options;
    }
    /**
     * Determines if the connection is already bound to a session. If so, this indicates that the
     * <code>#createSession()</code> cannot be called so as to produce child sessions.
     * @returns {boolean}
     */
    get isSessionBound() {
        return false;
    }
}
exports.RemoteConnection = RemoteConnection;
/**
 * Represents a traversal as a result of a {@link RemoteConnection} submission.
 */
class RemoteTraversal extends traversal_js_1.Traversal {
    traversers;
    sideEffects;
    constructor(traversers, sideEffects) {
        super(null, null, new bytecode_js_1.default());
        this.traversers = traversers;
        this.sideEffects = sideEffects;
    }
}
exports.RemoteTraversal = RemoteTraversal;
class RemoteStrategy extends traversal_strategy_js_1.TraversalStrategy {
    connection;
    /**
     * Creates a new instance of RemoteStrategy.
     * @param {RemoteConnection} connection
     */
    constructor(connection) {
        // gave this a fqcn that has a local "js:" prefix since this strategy isn't sent as bytecode to the server.
        // this is a sort of local-only strategy that actually executes client side. not sure if this prefix is the
        // right way to name this or not, but it should have a name to identify it.
        super('js:RemoteStrategy');
        this.connection = connection;
    }
    /** @override */
    apply(traversal) {
        if (traversal.traversers) {
            return Promise.resolve();
        }
        return this.connection.submit(traversal.getBytecode()).then(function (remoteTraversal) {
            traversal.sideEffects = remoteTraversal.sideEffects;
            traversal.traversers = remoteTraversal.traversers;
        });
    }
}
exports.RemoteStrategy = RemoteStrategy;
