/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.meecrowave.jta;

import jakarta.annotation.Priority;
import jakarta.interceptor.AroundInvoke;
import jakarta.interceptor.Interceptor;
import jakarta.interceptor.InvocationContext;
import jakarta.transaction.HeuristicMixedException;
import jakarta.transaction.HeuristicRollbackException;
import jakarta.transaction.InvalidTransactionException;
import jakarta.transaction.NotSupportedException;
import jakarta.transaction.RollbackException;
import jakarta.transaction.SystemException;
import jakarta.transaction.Transaction;
import jakarta.transaction.Transactional;
import jakarta.transaction.TransactionalException;

@Interceptor
@Transactional(Transactional.TxType.REQUIRES_NEW)
@Priority(200)
public class RequiredNewInterceptor extends InterceptorBase {
    @AroundInvoke
    public Object intercept(final InvocationContext ic) throws Exception {
        return super.intercept(ic);
    }

    @Override
    protected boolean isNewTransaction(final State state) {
        return state.old == null;
    }

    @Override
    protected State start() {
        try {
            final Transaction transaction = transactionManager.suspend();
            transactionManager.begin();
            return new State(transaction, transactionManager.getTransaction());
        } catch (final SystemException | NotSupportedException se) {
            throw new TransactionalException(se.getMessage(), se);
        }
    }

    @Override
    protected void commit(final State state) {
        try {
            if (state.old != state.current) {
                try {
                    state.current.commit();
                } catch (final HeuristicMixedException | HeuristicRollbackException | RollbackException | SystemException e) {
                    throw new TransactionalException(e.getMessage(), e);
                }
            }
        } finally {
            if (state.old != null) {
                try {
                    transactionManager.resume(state.old);
                } catch (final InvalidTransactionException | SystemException e) {
                    throw new TransactionalException(e.getMessage(), e);
                }
            }
        }
    }
}
