/* $Id: object.c,v 1.8 2005/12/28 23:54:29 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Interface to an abstract Postgres object
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <access/hash.h>
#include <catalog/pg_cast.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_operator.h>
#include <lib/stringinfo.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/geo_decls.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/record.h>
#include <pypg/error.h>

Datum
PyPg_datumCopy(Datum d, bool typbyval, int typlen)
{
	Datum rd = 0;
	PythonMemoryContext(rd = datumCopy(d, typbyval, typlen));
	return(rd);
}

PyObj
PyPgObject_cast(PyObj target, PyObj ob)
{
	TypeCacheEntry *tc = PyPgObject_FetchTypeCache(ob);
	HeapTuple tup = NULL;
	Oid castor = InvalidOid;
	Datum rd = 0;
	PyObj rob;

	Assert(PyPgType_Check(target) && PyPgObject_Check(ob));

	PG_TRY();
	{
		Oid from, to;

		to = PyPgType_FetchOid(target);
		from = PyPgObject_FetchTypeOid(ob);
		tup = SearchSysCache(CASTSOURCETARGET,
			ObjectIdGetDatum(from),
			ObjectIdGetDatum(to),
			0, 0
		);
		if (tup == NULL) goto tryend;
		castor = CASTSTRUCT(tup)->castfunc;
		ReleaseSysCache(tup);
		if (OidIsValid(castor))
			rd = OidFunctionCall1(castor, PyPgObject_FetchDatum(ob));
		else
			rd = PyPgObject_FetchDatum(ob);
tryend:;
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
		if (OidIsValid(castor) && !tc->typbyval && rd != 0)
			pfree(DatumGetPointer(rd));
	}
	PG_END_TRY();

	if (PyErr_Occurred() || tup == NULL) return(NULL);

	rob = PyPgObject_New(target, rd);
	if (OidIsValid(castor) && !tc->typbyval)
		pfree(DatumGetPointer(rd));

	return(rob);
}

PyObj
PyPgObject_typinput(PyObj subtype, PyObj ob)
{
	PyObj typ_element = PyPgType_FetchElementType(subtype);
	PyObj obstr, rob = NULL;
	char *str;
	Datum rd = 0;
	bool isnull = false;

	if (!PyPgType_Check(subtype))
	{
		PyErr_Format(PyExc_TypeError,
			"typinput requires a Postgres.Type "
			"as the first argument, given '%s'", subtype->ob_type->tp_name);
		return(NULL);
	}

	/*
	 * XXX: Unicode it then string it?
	 */
	obstr = PyObject_Str(ob);
	if (obstr == NULL) return(NULL);

	str = PyString_AS_STRING(obstr);
	PG_TRY();
	{
		FunctionCallInfoData fcinfo = {0,};
		fcinfo.flinfo = &PyPgType_FetchTypeInput(subtype);
		fcinfo.isnull = false;
		fcinfo.nargs = 3;
		fcinfo.arg[0] = PointerGetDatum(PyString_AS_STRING(obstr));
		fcinfo.arg[1] = -1;
		fcinfo.arg[2] = -1;
		if (typ_element != NULL)
		{
			fcinfo.arg[1] = ObjectIdGetDatum(PyPgType_FetchOid(typ_element));
			fcinfo.arg[2] = -1;
		}
		rd = FunctionCallInvoke(&fcinfo);
		isnull = fcinfo.isnull;
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();
	Py_DECREF(obstr);

	if (!PyErr_Occurred())
	{
		if (isnull)
		{
			rob = Py_None;
			Py_INCREF(rob);
		}
		else
		{
			TypeCacheEntry *tc = PyPgType_FetchTypeCache(subtype);
			rob = PyPgObject_New(subtype, rd);
			if (!tc->typbyval) pfree(DatumGetPointer(rd));
		}
	}

	return(rob);
}

PyObj
PyPgObject_typreceive(PyObj subtype, PyObj ob)
{
	PyObj obstr, rob = NULL;
	StringInfoData buf;
	Datum rd = 0;
	bool isnull = false;

	obstr = PyObject_Str(ob);
	if (obstr == NULL) return(NULL);

	PG_TRY();
	{
		FmgrInfo *flinfo = &PyPgType_FetchTypeInput(subtype);
		FunctionCallInfoData fcinfo = {0,};
		fcinfo.flinfo = flinfo;
		fcinfo.isnull = false;
		fcinfo.nargs = 1;

		initStringInfo(&buf);
		appendBinaryStringInfo(&buf,
			PyString_AS_STRING(obstr),
			PyString_GET_SIZE(obstr)
		);

		fcinfo.arg[0] = PointerGetDatum(&buf);
		rd = FunctionCallInvoke(&fcinfo);
		isnull = fcinfo.isnull;
	}
	PYPG_CATCH_END();

	Py_DECREF(obstr);
	pfree(buf.data);

	if (!PyErr_Occurred())
	{
		if (isnull)
		{
			rob = Py_None;
			Py_INCREF(rob);
		}
		else
		{
			TypeCacheEntry *tc = PyPgType_FetchTypeCache(subtype);
			rob = PyPgObject_New(subtype, rd);
			if (!tc->typbyval) pfree(DatumGetPointer(rd));
		}
	}

	return(rob);
}

PyObj
PyPgObject_typoutput(PyObj self)
{
	PyObj rob = NULL;
	char *str = NULL;

	PG_TRY();
	{
		FmgrInfo *flinfo = &PyPgObject_FetchTypeOutput(self);
		Datum d = PyPgObject_FetchDatum(self);
		FunctionCallInfoData fcinfo = {0,};
		fcinfo.flinfo = flinfo;
		fcinfo.isnull = false;
		fcinfo.nargs = 3;

		fcinfo.arg[0] = d;
		fcinfo.arg[1] = 0;
		fcinfo.arg[2] = -1;
		str = (char *) FunctionCallInvoke(&fcinfo);
		rob = PyString_FromString(str);
		pfree(str);
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
		Py_XDECREF(rob);
	}
	PG_END_TRY();

	return(rob);
}

PyObj
PyPgObject_typsend(PyObj self)
{
	PyObj rob = NULL;

	PG_TRY();
	{
		Datum rd, d = PyPgObject_FetchDatum(self);
		FunctionCallInfoData fcinfo = {0,};
		fcinfo.flinfo = &PyPgObject_FetchTypeSend(self);
		fcinfo.isnull = false;
		fcinfo.nargs = 1;
		fcinfo.arg[0] = d;
		rd = FunctionCallInvoke(&fcinfo);
		/**
		 * WARNING: Assumes untoasted bytea is returned.
		 **/
		rob = PyString_FromStringAndSize(VARDATA(rd), VARSIZE(rd) - VARHDRSZ);
		pfree(DatumGetPointer(rd));
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
		Py_XDECREF(rob);
	}
	PG_END_TRY();

	return(rob);
}

static PyObj
typcast(PyObj self, PyObj ob)
{
	PyObj rob;
	rob = PyPgObject_cast(self, ob);
	if (rob == NULL && !PyErr_Occurred())
	{
		PyErr_Format(PyExc_LookupError,
			"no explicit cast for transforming a %s into a %s",
			((PyTypeObject *) self)->tp_name,
			ob->ob_type->tp_name);
	}
	return(rob);
}

static PyMethodDef PyPgObject_Methods[] = {
	{"typinput", (PyCFunction) PyPgObject_typinput, METH_O | METH_CLASS,
	PyDoc_STR("create an instance using the type's typinput function")},
	{"typoutput", (PyCFunction) PyPgObject_typoutput, METH_NOARGS,
	PyDoc_STR("create a cstring using the type's typoutput function")},
	{"typreceive", (PyCFunction) PyPgObject_typreceive, METH_O | METH_CLASS,
	PyDoc_STR("create an instance using the type's typreceive function")},
	{"typsend", (PyCFunction) PyPgObject_typsend, METH_NOARGS,
	PyDoc_STR("create a bytea using the type's typsend function")},
	{"typcast", (PyCFunction) typcast, METH_O | METH_CLASS,
	PyDoc_STR("explicitly cast a Postgres.Object instance")},
	{NULL}
};

static PyMemberDef PyPgObject_Members[] = {
	{"type", T_OBJECT, offsetof(struct PyPgObject, ob_type), RO,
	PyDoc_STR("the object's type [DEPRECATING, DONT USE]")},
	{NULL}
};

#define fetch_opOid(oprname, oprleft, oprright) \
LookupOperName(stringToQualifiedNameList(oprname, NULL), oprleft, oprright, 1)

PyObj
PyPgObject_Operate(const char *op, PyObj former, PyObj latter)
{
	Oid opOid = InvalidOid, opCode = InvalidOid, opRes = InvalidOid;
	HeapTuple opTup = NULL;

	volatile PyObj rob = NULL;
	volatile Datum rd = 0;

	Oid strong_typoid;
	Oid weak_typoid = InvalidOid;
	Datum weak_datum = 0;

	/* 'strong' is the PyPgObject that establishes the type */
	PyObj strong, weak;

	if (PyPgObject_Check(former))
	{
		strong = former;
		weak = latter;
	}
	else if (latter != NULL && PyPgObject_Check(latter))
	{
		weak = former;
		strong = latter;
	}
	else
	{
		PyErr_SetString(PyExc_TypeError,
			"cannot operate on non-Postgres.Objects");
		return(NULL);
	}

	strong_typoid = PyPgObject_FetchTypeOid(strong);

	if (weak)
	{
		if (weak == Py_True || weak == Py_False)
		{
			weak_typoid = BOOLOID;
			weak_datum = (weak == Py_True) ? 1 : 0;
			Py_INCREF(weak);
		}
		else
		{
			if (!PyPgObject_Check(weak))
			{
				PyObj new_weak;
				new_weak = Py_Call((PyObj) PyPgObject_FetchType(strong), weak);
				if (new_weak == NULL) return(NULL);
				weak = new_weak;
			}
			else
			{
				Py_INCREF(weak);
			}
			weak_datum = PyPgObject_FetchDatum(weak);
			weak_typoid = PyPgObject_FetchTypeOid(weak);
			opOid = fetch_opOid(op, strong_typoid, weak_typoid);
		}
	}
	else
		opOid = fetch_opOid(op, InvalidOid, strong_typoid);

	if ((opTup = SearchSysCache(OPEROID, opOid, 0, 0, 0)))
	{
		Form_pg_operator opStruct = OPERSTRUCT(opTup);
		opCode = opStruct->oprcode;
		opRes = opStruct->oprresult;
		ReleaseSysCache(opTup);
	}
	else
	{
		PyErr_Format(PyExc_LookupError, "no operator %s", op);
		Py_XDECREF(weak);
		return(NULL);
	}

	{
		struct FmgrInfo flinfo = {0,};
		struct FunctionCallInfoData fcid = {0,};
		fmgr_info(opCode, &flinfo);
		fcid.flinfo = &flinfo;
		fcid.nargs = 1;

		fcid.arg[0] = PyPgObject_FetchDatum(strong);
		fcid.argnull[0] = false;
		if (weak)
		{
			++fcid.nargs;
			fcid.arg[1] = weak_datum;
			fcid.argnull[1] = false;
		}
		PgError_TRAP(rd = FunctionCallInvoke(&fcid));
	}
	Py_XDECREF(weak);

	if (!PyErr_Occurred())
		rob = PyPgObject_FromTypeOidAndDatum(opRes, rd);

	return(rob);
}

static PyObj
obj_add(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("+", self, with));
}

static PyObj
obj_subtract(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("-", self, with));
}

static PyObj
obj_multiply(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("*", self, with));
}

static PyObj
obj_divide(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("/", self, with));
}

static PyObj
obj_remainder(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("%", self, with));
}

static PyObj
obj_divmod(PyObj self, PyObj with)
{
	PyObj div, mod, rob;
	rob = PyTuple_New(2);
	if (rob == NULL) return(NULL);

	div = PyPgObject_Operate("/", self, with);
	if (div == NULL) goto error;
	PyTuple_SET_ITEM(rob, 0, div);
	mod = PyPgObject_Operate("%", self, with);
	if (mod == NULL) goto error;
	PyTuple_SET_ITEM(rob, 1, mod);

	return(rob);
error:
	Py_DECREF(rob);
	return(NULL);
}

static PyObj
obj_power(PyObj self, PyObj with, PyObj andthis)
{
	return(PyPgObject_Operate("^", self, with));
}

static PyObj
obj_negative(PyObj self)
{
	return(PyPgObject_Operate("-", self, NULL));
}

static PyObj
obj_positive(PyObj self)
{
	return(PyPgObject_Operate("+", self, NULL));
}

static PyObj
obj_absolute(PyObj self)
{
	Oid typoid = PyPgObject_FetchTypeOid(self);
	volatile Oid procoid, roid;
	volatile HeapTuple procTuple;
	volatile Datum rd = 0;
	volatile PyObj rob = NULL;

	procoid = LookupFuncName(stringToQualifiedNameList("abs", NULL),
			1, &(typoid), true);
	if (procoid == InvalidOid)
	{
		PyErr_Format(PyExc_LookupError,
				"no such function named 'abs' for type %u", typoid);
		return(NULL);
	}
	procTuple = SearchSysCache(PROCOID, procoid, 0, 0, 0);
	if (procTuple == NULL)
	{
		PyErr_Format(PyExc_LookupError,
				"no procedure with Oid %u", procoid);
		return(NULL);
	}
	
	roid = PROCSTRUCT(procTuple)->prorettype;
	PgError_TRAP(rd = OidFunctionCall1(procoid, PyPgObject_FetchDatum(self)));
	ReleaseSysCache(procTuple);

	if (!PyErr_Occurred())
	{
		rob = PyPgObject_FromTypeOidAndDatum(roid, rd);
		if (PyPgObject_ShouldFree(rob))
			pfree(DatumGetPointer(rd));
	}

	return(rob);
}

static int
obj_nonzero(PyObj self)
{
	return((int) PyPgObject_FetchDatum(self));
}

static PyObj
obj_invert(PyObj self)
{
	return(PyPgObject_Operate("~", self, NULL));
}

static PyObj
obj_lshift(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("<<", self, with));
}

static PyObj
obj_rshift(PyObj self, PyObj with)
{
	return(PyPgObject_Operate(">>", self, with));
}

static PyObj
obj_and(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("&", self, with));
}

static PyObj
obj_xor(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("#", self, with));
}

static PyObj
obj_or(PyObj self, PyObj with)
{
	return(PyPgObject_Operate("|", self, with));
}

static PyObj
obj_oct(PyObj self)
{
	PyObj lo, rob = NULL;
	lo = self->ob_type->tp_as_number->nb_long(self);
	if (lo)
	{
		rob = lo->ob_type->tp_as_number->nb_oct(lo);
		DECREF(lo);
	}
	return(rob);
}

static PyObj
obj_hex(PyObj self)
{
	PyObj lo, rob = NULL;
	lo = self->ob_type->tp_as_number->nb_long(self);
	if (lo)
	{
		rob = lo->ob_type->tp_as_number->nb_hex(lo);
		DECREF(lo);
	}
	return(rob);
}

static PyNumberMethods PyPgObjectAsNumber = {
	obj_add,			/* nb_add */
	obj_subtract,	/* nb_subtract */
	obj_multiply,	/* nb_multiply */
	obj_divide,		/* nb_divide */
	obj_remainder,	/* nb_remainder */
	obj_divmod,		/* nb_divmod */
	obj_power,		/* nb_power */
	obj_negative,	/* nb_negative */
	obj_positive,	/* nb_positive */
	obj_absolute,	/* nb_absolute */
	obj_nonzero,	/* nb_nonzero */
	obj_invert,		/* nb_invert */
	obj_lshift,		/* nb_lshift */
	obj_rshift,		/* nb_rshift */
	obj_and,			/* nb_and */
	obj_xor,			/* nb_xor */
	obj_or,			/* nb_or */
	NULL,				/* nb_coerce */
	NULL,				/* nb_int */
	NULL,				/* nb_long */
	NULL,				/* nb_float */
	obj_oct,			/* nb_oct */
	obj_hex,			/* nb_hex */

	NULL,				/* nb_inplace_add */
	NULL,				/* nb_inplace_subtract */
	NULL,				/* nb_inplace_multiply */
	NULL,				/* nb_inplace_divide */
	NULL,				/* nb_inplace_remainder */
	NULL,				/* nb_inplace_power */
	NULL,				/* nb_inplace_lshift */
	NULL,				/* nb_inplace_rshift */
	NULL,				/* nb_inplace_and */
	NULL,				/* nb_inplace_xor */
	NULL,				/* nb_inplace_or */

	NULL,				/* nb_floor_divide */
	NULL,				/* nb_true_divide */
	NULL,				/* nb_inplace_floor_divide */
	NULL,				/* nb_inplace_true_divide */
};

int
obj_getreadbuffer(PyObj self, int segment, void **out)
{
	int typlen, typbyval, ret = 0;

	if (segment != 0)
	{
		PyErr_SetString(PyExc_SystemError,
			"Postgres.Object buffers only have one segment");
		return(-1);
	}

	typlen = PyPgObject_FetchTypeCache(self)->typlen;
	typbyval = PyPgObject_FetchTypeCache(self)->typbyval;

	if (typbyval)
	{
		*out = &PyPgObject_FetchDatum(self);
		ret = typlen;
	}
	else
	{
		Datum ob_datum = PyPgObject_FetchDatum(self);
		switch (typlen)
		{
			case -1:
				/*
				 * Skip the VARLENA header
				 */
				*out = (void *) VARDATA(ob_datum);
				ret = (int) VARSIZE(ob_datum) - VARHDRSZ;
			break;

			case -2:
				ret = strlen((char *) ob_datum);
				*out = DatumGetPointer(ob_datum);
			break;

			default:
				ret = typlen;
				*out = DatumGetPointer(ob_datum);
			break;
		}
	}

	return(ret);
}

int
obj_getsegcount(PyObj self, int *length)
{
	if (length != NULL)
	{
		int typlen, typbyval;
		typlen = PyPgObject_FetchTypeCache(self)->typlen;
		typbyval = PyPgObject_FetchTypeCache(self)->typbyval;

		if (typbyval)
			*length = typlen;
		else
		{
			Datum ob_datum = PyPgObject_FetchDatum(self);
			switch (typlen)
			{
				case -1:
					*length = (int) VARSIZE(ob_datum) - VARHDRSZ;
				break;

				case -2:
					*length = strlen((char *) ob_datum);
				break;

				default:
					*length = typlen;
				break;
			}
		}
	}

	return(1);
}

static PyBufferProcs PyPgObjectBufferProcs = {
	obj_getreadbuffer, /* bf_getreadbuffer */
	NULL, /* bf_getwritebuffer */
	obj_getsegcount, /* bf_getsegcount */
	NULL, /* bf_getcharbuffer */
};

static int
obj_compare(PyObj self, PyObj with)
{
	int ret = -1;

	if (self == Py_None || with == Py_None)
		ret = -1;
	else if (self == with)
		ret = 0;
	else
	{
		PyObj withd;

		if (!PyPgObject_Check(with))
		{
			withd = PyPgObject_FromPyPgTypeAndPyObject(
							(PyObj) PyPgObject_FetchType(self), with);
			if (withd == NULL)
				return(-1);
		}
		else
		{
			withd = with;
			INCREF(withd);
		}

		if (PyPgObject_FetchDatum(self) == PyPgObject_FetchDatum(withd) &&
			 PyPgObject_FetchTypeOid(self) == PyPgObject_FetchTypeOid(withd))
			ret = 0;
		else
		{
			PyObj et;
			et = PyPgObject_Operate("=", self, withd);
			if (et == NULL)
			{
				DECREF(withd);
				return(-1);
			}
			if (PyPgObject_FetchDatum(et)) ret = 0;
			DECREF(et);
			
			if (ret == -1)
			{
				PyObj lt;
				lt = PyPgObject_Operate("<", self, withd);
				if (lt == NULL)
				{
					DECREF(withd);
					return(-1);
				}
				if (!PyPgObject_FetchDatum(lt)) ret = 1;

				DECREF(lt);
			}
		}

		DECREF(withd);
	}

	return(ret);
}

static long
obj_hash(PyObj self)
{
	Datum ob_datum = PyPgObject_FetchDatum(self);
	TypeCacheEntry *tc = PyPgObject_FetchTypeCache(self);
	int2 typlen = tc->typlen;
	long rv = 0;

	if (tc->typbyval)
	{
		rv = ((long) ob_datum);
	}
	else if (typlen > -1 && typlen <= sizeof(long))
	{
		rv = (*((long *) DatumGetPointer(ob_datum)));
	}
	else
	{
		int len;
		switch(typlen)
		{
			case -2:
				len = strlen((char *) DatumGetPointer(ob_datum));
			break;

			case -1:
				len = VARSIZE(ob_datum) - VARHDRSZ;
				ob_datum = PointerGetDatum(VARDATA(ob_datum));
			break;

			default:
				len = (int) typlen;
			break;
		}
		rv = hash_any((unsigned char *) DatumGetPointer(ob_datum), len);
	}

	return(rv);
}

static PyObj
obj_repr(PyObj self)
{
	PyObj rob = NULL;
	char *str;

	PG_TRY();
	{
		FunctionCallInfoData fcinfo = {0,};
		fcinfo.flinfo = &PyPgObject_FetchTypeOutput(self);
		fcinfo.nargs = 3;
		fcinfo.arg[0] = PyPgObject_FetchDatum(self);
		fcinfo.arg[1] = 0;
		fcinfo.arg[2] = -1;
		str = (char *) FunctionCallInvoke(&fcinfo);
		rob = PyString_FromFormat("'%s'::%s", str, self->ob_type->tp_name);
		pfree(str);
	}
	PG_CATCH();
	{
		Py_XDECREF(rob);
		PyErr_SetPgError();
	}
	PG_END_TRY();

	return(rob);
}

static PyObj
obj_richcompare(PyObj self, PyObj with, int comparison)
{
	char *op = NULL;
	PyObj rob;

	if (!PyPgObject_Check(with) && !PyPgObject_Check(self))
	{
		Py_INCREF(Py_False);
		return(Py_False);
	}

	if (with == Py_None)
	{
		switch (comparison)
		{
			case Py_NE:
				rob = Py_True;
			break;
			default:
				rob = Py_False;
			break;
		}
		Py_INCREF(rob);
		return(rob);
	}

	switch (comparison)
	{
		case Py_LT:
			op = "<";
		break;

		case Py_LE:
			op = "<=";
		break;

		case Py_EQ:
			op = "=";
		break;

		case Py_NE:
			op = "!=";
		break;

		case Py_GT:
			op = ">";
		break;

		case Py_GE:
			op = ">=";
		break;

		default:
			PyErr_Format(PyExc_TypeError,
					"unknown comparison operation(%d)", comparison);
			return(NULL);
		break;
	}

	rob = PyPgObject_Operate(op, self, with);
	return(rob);
}

static void
obj_dealloc(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	if (PyPgObject_ShouldFree(self) && d != 0)
	{
		PyPgObject_FixDatum(self, 0);
		pfree(DatumGetPointer(d));
	}
	self->ob_type->tp_free(self);
}

static PyObj
obj_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj obdata, rob;
	TypeCacheEntry *tc;
	Size sz;
	int2 typlen;
	char *str;
	Datum dat;

	if (!PyPgType_Check(subtype))
	{
		PyErr_Format(PyExc_TypeError,
			"first argument must be a Postgres.Type, not a '%s'", 
			subtype->ob_type->tp_name
		);
		return(NULL);
	}

	if (!PyArg_ParseTuple(args, "O", &obdata))
		return(NULL);

	if (!PyString_Check(obdata))
	{
		PyErr_Format(PyExc_TypeError,
				"second argument must be a string, not a '%s'", 
				obdata->ob_type->tp_name
		);
		return(NULL);
	}

	sz = PyObject_Length(obdata);
	str = PyString_AS_STRING(obdata);
	tc = PyPgType_FetchTypeCache(subtype);
	typlen = tc->typlen;

	switch (typlen)
	{
		/* Specifically sized */
		case -1:
		{
			struct varattrib *varl;
			Size tsz = sz + VARHDRSZ;

			varl = MemoryContextAlloc(PythonMemoryContext, tsz);
			VARATT_SIZEP(varl) = tsz;
			memcpy(VARDATA(varl), str, sz);

			dat = PointerGetDatum(varl);
		}
		break;

		/* NULL terminated */
		case -2:
			dat = PointerGetDatum(pstrdup(str));
		break;

		default:
		{
			char *dst;
			if (sz != typlen)
			{
				PyErr_Format(PyExc_OverflowError,
					"the type's length is %d bytes, "
					"but the given data is %lu bytes", typlen, (unsigned long) sz
				);
				return(NULL);
			}
			
			if (tc->typbyval)
				dst = (char *) &dat;
			else
			{
				dst = MemoryContextAlloc(PythonMemoryContext, typlen);
				dat = PointerGetDatum(dst);
			}
			memcpy(dst, str, typlen);
		}
		break;
	}

	rob = subtype->tp_alloc(subtype, 0);
	PyPgObject_FixDatum(rob, dat);
	return(rob);
}

PyDoc_STRVAR(PyPgObject_Type_Doc, "Postgres object interface type");

PyPgTypeObject PyPgObject_Type = {
{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,										/* ob_size */
	"Postgres.Object",				/* tp_name */
	sizeof(struct PyPgObject),		/* tp_basicsize */
	0,										/* tp_itemsize */
	obj_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	obj_compare,						/* tp_compare */
	obj_repr,							/* tp_repr */
	&PyPgObjectAsNumber,				/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	obj_hash,							/* tp_hash */
	NULL,									/* tp_call */
	PyPgObject_typoutput,			/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	&PyPgObjectBufferProcs,			/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE,			   /* tp_flags */
	PyPgObject_Type_Doc,				/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	obj_richcompare,					/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgObject_Methods,				/* tp_methods */
	PyPgObject_Members,				/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	obj_new,								/* tp_new */
	NULL,									/* tp_free */
},
};

PyObj
PyPgObject_Initialize(PyObj self, Datum d)
{
	register TypeCacheEntry *tc = PyPgObject_FetchTypeCache(self);
	if (self == NULL) return(NULL);

	if (!tc->typbyval)
		d = PyPg_datumCopy(d, false, tc->typlen);
	PyPgObject_FixDatum(self, d);
	return(self);
}

PyObj
PyPgObject_FromPyPgTypeAndDatum(PyObj to, Datum dat)
{
	TypeCacheEntry *tc = PyPgType_FetchTypeCache(to);
	PyObj rob;

	switch (tc->type_id)
	{
		case BOOLOID:
		{
			rob = dat ? Py_True : Py_False;
			Py_INCREF(rob);
		}
		break;

		default:
			rob = PyPgObject_New(to, dat);
		break;
	}

	return(rob);
}

PyObj
PyPgObject_FromTypeOidAndDatum(Oid typeoid, Datum d)
{
	PyObj typ, rob;

	typ = PyPgType_FromOid(typeoid);
	rob = PyPgObject_FromPyPgTypeAndDatum(typ, d);
	DECREF(typ);

	return(rob);
}

PyObj
PyPgObject_FromPyPgTypeAndHeapTuple(PyObj to, HeapTuple ht)
{
	HeapTupleHeader hth;
	PyObj rob;

	if (!PyPgType_IsComposite(to))
	{
		PyErr_SetString(PyExc_TypeError,
			"Postgres.Type is not a composite type");
		return(NULL);
	}

	hth = HeapTupleHeader_FromHeapTuple(ht);
	HeapTupleHeaderSetTypeId(hth, PyPgType_FetchOid(to));
	rob = PyPgObject_New(to, PointerGetDatum(hth));

	return(rob);
}

PyObj
pseudo_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyErr_Format(PyExc_TypeError,
		"psuedo-type '%s' cannot be instantiated", subtype->ob_type->tp_name);
	return(NULL);
}

PyDoc_STRVAR(PyPgPsuedo_Type_Doc, "Psuedo-types basetype");

PyPgTypeObject PyPgPseudo_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"pseudo",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE,						/* tp_flags */
	PyPgPsuedo_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	pseudo_new,									/* tp_new */
},
};

PyObj
PyPgArbitrary_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	static char *kwlist[] = {"source", NULL};
	PyObj ob;
	if (!PyPgType_Check(subtype))
	{
		PyErr_Format(PyExc_TypeError,
			"arbitrary.__new__ requires a Postgres.Type instance "
			"as the first argument, given '%s'", subtype->ob_type->tp_name);
		return(NULL);
	}

	if (!PyArg_ParseTupleAndKeywords(args, kw, "O", kwlist, &ob))
		return(NULL);

	if (ob->ob_type == subtype)
	{
		Py_INCREF(ob);
		return(ob);
	}

	return(PyPgObject_typinput((PyObj) subtype, ob));
}

PyDoc_STRVAR(PyPgArbitrary_Type_Doc, "Arbitrary object basetypes");

PyPgTypeObject PyPgArbitrary_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"arbitrary",								/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE,						/* tp_flags */
	PyPgArbitrary_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
/*
 * vim: ts=3:sw=3:noet:
 */
