//
// AbstractConfiguration.cpp
//
// $Id: //poco/1.4/Util/src/AbstractConfiguration.cpp#2 $
//
// Library: Util
// Package: Configuration
// Module:  AbstractConfiguration
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Util/AbstractConfiguration.h"
#include "Poco/Util/ConfigurationView.h"
#include "Poco/Exception.h"
#include "Poco/NumberParser.h"
#include "Poco/NumberFormatter.h"
#include "Poco/String.h"


using Poco::FastMutex;
using Poco::NotFoundException;
using Poco::SyntaxException;
using Poco::CircularReferenceException;
using Poco::NumberParser;
using Poco::NumberFormatter;
using Poco::icompare;


namespace Poco {
namespace Util {


AbstractConfiguration::AbstractConfiguration(): 
	_depth(0),
	_eventsEnabled(true)
{
}


AbstractConfiguration::~AbstractConfiguration()
{
}


bool AbstractConfiguration::hasProperty(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	return getRaw(key, value);
}


bool AbstractConfiguration::hasOption(const std::string& key) const
{
	return hasProperty(key);
}


bool AbstractConfiguration::has(const std::string& key) const
{
	return hasProperty(key);
}

	
std::string AbstractConfiguration::getString(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return internalExpand(value);
	else
		throw NotFoundException(key);
}

	
std::string AbstractConfiguration::getString(const std::string& key, const std::string& defaultValue) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return internalExpand(value);
	else
		return defaultValue;
}


std::string AbstractConfiguration::getRawString(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return value;
	else
		throw NotFoundException(key);
}

	
std::string AbstractConfiguration::getRawString(const std::string& key, const std::string& defaultValue) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return value;
	else
		return defaultValue;
}

	
int AbstractConfiguration::getInt(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return parseInt(internalExpand(value));
	else
		throw NotFoundException(key);
}

	
int AbstractConfiguration::getInt(const std::string& key, int defaultValue) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return parseInt(internalExpand(value));
	else
		return defaultValue;
}


double AbstractConfiguration::getDouble(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return NumberParser::parseFloat(internalExpand(value));
	else
		throw NotFoundException(key);
}

	
double AbstractConfiguration::getDouble(const std::string& key, double defaultValue) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return NumberParser::parseFloat(internalExpand(value));
	else
		return defaultValue;
}


bool AbstractConfiguration::getBool(const std::string& key) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return parseBool(internalExpand(value));
	else
		throw NotFoundException(key);
}


bool AbstractConfiguration::getBool(const std::string& key, bool defaultValue) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string value;
	if (getRaw(key, value))
		return parseBool(internalExpand(value));
	else
		return defaultValue;
}


void AbstractConfiguration::setString(const std::string& key, const std::string& value)
{
	setRawWithEvent(key, value);
}

	
void AbstractConfiguration::setInt(const std::string& key, int value)
{
	setRawWithEvent(key, NumberFormatter::format(value));
}


void AbstractConfiguration::setDouble(const std::string& key, double value)
{
	setRawWithEvent(key, NumberFormatter::format(value));
}


void AbstractConfiguration::setBool(const std::string& key, bool value)
{
	setRawWithEvent(key, value ? "true" : "false");
}


void AbstractConfiguration::keys(Keys& range) const
{
	FastMutex::ScopedLock lock(_mutex);

	std::string key;
	range.clear();
	enumerate(key, range);
}


void AbstractConfiguration::keys(const std::string& key, Keys& range) const
{
	FastMutex::ScopedLock lock(_mutex);

	range.clear();
	enumerate(key, range);
}


const AbstractConfiguration* AbstractConfiguration::createView(const std::string& prefix) const
{
	return new ConfigurationView(prefix, const_cast<AbstractConfiguration*>(this));
}


AbstractConfiguration* AbstractConfiguration::createView(const std::string& prefix)
{
	return new ConfigurationView(prefix, this);
}


namespace
{
	class AutoCounter
	{
	public:
		AutoCounter(int& count): _count(count)
		{
			++_count;
		}
		
		~AutoCounter()
		{
			--_count;
		}
		
	private:
		int& _count;
	};
}


std::string AbstractConfiguration::expand(const std::string& value) const
{
	FastMutex::ScopedLock lock(_mutex);

	return internalExpand(value);
}


void AbstractConfiguration::remove(const std::string& key)
{
	if (_eventsEnabled)
	{
		propertyRemoving(this, key);
	}
	{
		FastMutex::ScopedLock lock(_mutex);

		removeRaw(key);
	}
	if (_eventsEnabled)
	{
		propertyRemoved(this, key);
	}
}


void AbstractConfiguration::enableEvents(bool enable)
{
	_eventsEnabled = enable;
}

	
bool AbstractConfiguration::eventsEnabled() const
{
	return _eventsEnabled;
}


void AbstractConfiguration::removeRaw(const std::string& key)
{
	throw Poco::NotImplementedException("removeRaw()");
}


std::string AbstractConfiguration::internalExpand(const std::string& value) const
{
	AutoCounter counter(_depth);
	if (_depth > 10) throw CircularReferenceException("Too many property references encountered");
	return uncheckedExpand(value);
}


std::string AbstractConfiguration::uncheckedExpand(const std::string& value) const
{
	std::string result;
	std::string::const_iterator it  = value.begin();
	std::string::const_iterator end = value.end();
	while (it != end)
	{
		if (*it == '$')
		{
			++it;
			if (it != end && *it == '{')
			{
				++it;
				std::string prop;
				while (it != end && *it != '}') prop += *it++;
				if (it != end) ++it;
				std::string value;
				if (getRaw(prop, value))
				{
					result.append(internalExpand(value));
				}
				else
				{
					result.append("${");
					result.append(prop);
					result.append("}");
				}
			}
			else result += '$';
		}
		else result += *it++;
	}
	return result;
}


int AbstractConfiguration::parseInt(const std::string& value)
{
	if (value.compare(0, 2, "0x") == 0)
		return NumberParser::parseHex(value.substr(2));
	else
		return NumberParser::parse(value);
}


bool AbstractConfiguration::parseBool(const std::string& value)
{
	int n;
	if (NumberParser::tryParse(value, n))
		return n != 0;
	else if (icompare(value, "true") == 0)
		return true;
	else if (icompare(value, "yes") == 0)
		return true;
	else if (icompare(value, "on") == 0)
		return true;
	else if (icompare(value, "false") == 0)
		return false;
	else if (icompare(value, "no") == 0)
		return false;
	else if (icompare(value, "off") == 0)
		return false;
	else 
		throw SyntaxException("Cannot convert to boolean", value);
}


void AbstractConfiguration::setRawWithEvent(const std::string& key, std::string value)
{
	KeyValue kv(key, value);
	if (_eventsEnabled)
	{
		propertyChanging(this, kv);
	}
	{
		FastMutex::ScopedLock lock(_mutex);

		setRaw(key, value);
	}
	if (_eventsEnabled)
	{
		propertyChanged(this, kv);
	}
}


} } // namespace Poco::Util
