#!/usr/local/bin/luac53 -p --
--[[
--@brief  Quasi Robot Language; Lua Library
--@date   Sun,06 Sep,2015
--@date   Sun,20 Sep,2015
--@date   Wed,28 Oct,2015 - Thu,29 Oct,2015
--@date   Sun,22 Nov,2015
--@date   Sun,13 Dec,2015
--@date   Sat,19 Dec,2015
--@date   Sun,27 Dec,2015
--@date   Sun,10 Jan,2016
--@date   Mon,17 Jul,2017
--@date   Tue,15 Aug,2017
--@date   Sun,03 Sep,2017
--@date   Fri,05 Jan,2018
--@date   Sat,14 Apr,2018
--@date   Sat,12 May,2018
--@date   Sat,24 Nov,2018
--@date   Sun,25 Nov,2018
--@date   Sun,02 Dec,2018
--@date   Sun,09 Dec,2018
--@date   Sat,25 Jan,2020
--@date   Sat,18 Apr,2020
--@date   Thu,23 Apr,2020
--@date   Sun,04 Oct,2020
--@date   Sun,11 Oct,2020
--@date   Sat,17 Oct,2020
--@date   Wed,21 Oct,2020
--@date   Tue,08 Mar,2022
--@date   Fri,11 Mar,2022
--@author Copyright(C)2015-2022 G-HAL
--]]

local SuspendSystemState = require "LuaData/suspend";



-- Constructor and Destructor

function qrl_chip_outofrange(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:OutOfRange');
  if enable_debugger then
    dbg();
  end;

  if (0 < g_QRL_SystemVariable["SubroutinePtr"]) then
    g_QRL_SystemVariable["SubroutinePtr"] = g_QRL_SystemVariable["SubroutinePtr"] - 1;
    return g_QRL_SystemVariable["Subroutine"][g_QRL_SystemVariable["SubroutinePtr"] + 1];
  end;
  return '0000';
end;


local function qrl_chip_begin(Mek, GB)
  -- initialization
  -- print("Hello QRL!!");
  math.randomseed(os.time());

  InitMaps();
  ClearHotMaps();

  A_Star.Init();

  return;
end;

local function qrl_chip_end(Mek, GB)
  -- finalization
  -- print("Goodbye QRL!!");
  return;
end;


function qrl_chip_start(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:start');
  qrl_chip_begin(Mek, GB);
  return;
end;

function qrl_chip_resume(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:resume');
  SuspendSystemState.Resume(GB, GetSuspendFileName(GB, Mek));
  qrl_chip_begin(Mek, GB);
  return;
end;

function qrl_chip_hibernate(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:hibernate');
  SuspendSystemState.Hibernate(GetSuspendFileName(GB, Mek));
  return;
end;

function qrl_chip_destroy(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:destroy');
  SuspendSystemState.Destroy(GetSuspendFileName(GB, Mek));
  qrl_chip_end(Mek, GB);
  return;
end;

function qrl_chip_stop(Mek, GB)
  PrintTraceMessage(GB, Mek, '000-000:stop');
  qrl_chip_end(Mek, GB);
  return;
end;



-- Command Execution Function

function qrl_chip_exec(GB, Mek)
  -- Reload Ammo
  local t;
  for t = 1, g_QRL_SystemVariable["ExecQueue_Reload_Queue_Ptr"], 1 do
    local Weapon = g_QRL_SystemVariable["ExecQueue_Reload_Queue_Weapon"][t];
    local Ammo = g_QRL_SystemVariable["ExecQueue_Reload_Queue_Ammo"][t];
    if (nil ~= Weapon) then
      if (nil ~= Ammo) then
        InstallAmmo(GB, Mek, Weapon, Ammo);
      else
        ReloadAmmo(GB, Mek, Weapon);
      end;
    end;
  end;
  g_QRL_SystemVariable["ExecQueue_Reload_Queue_Ptr"] = 0;

  -- Attack Target
  if ((nil ~= g_QRL_SystemVariable["ExecQueue_Attack_Weapon"]) and (nil ~= g_QRL_SystemVariable["ExecQueue_Attack_Target"])) then
    if g_QRL_SystemVariable["ExecQueue_Attack_Mode_AIAttacker"] then
      AIAttacker(GB, Mek, g_QRL_SystemVariable["ExecQueue_Attack_Weapon"], g_QRL_SystemVariable["ExecQueue_Attack_Target"]);
    else
      AttackerFrontEnd(GB, Mek, g_QRL_SystemVariable["ExecQueue_Attack_Weapon"], g_QRL_SystemVariable["ExecQueue_Attack_Target"], g_QRL_SystemVariable["ExecQueue_Attack_AtOp"]);
    end;
  end;
  g_QRL_SystemVariable["ExecQueue_Attack_Mode_AIAttacker"] = false;
  g_QRL_SystemVariable["ExecQueue_Attack_Weapon"] = nil;
  g_QRL_SystemVariable["ExecQueue_Attack_Target"] = nil;
  g_QRL_SystemVariable["ExecQueue_Attack_AtOp"] = 0;

  -- Do Action
  local function Exec_NONE(GB, Mek)
    return;
  end;

  local function Exec_PrepAction(GB, Mek)
    PrepAction(GB, Mek, g_QRL_SystemVariable["ExecQueue_Type_PrepAction"]);
    return;
  end;

  local function Exec_PrepActionStopAndGearUp(GB, Mek)
    PrepAction(GB, Mek, NAV_Stop);
    GearUp(Mek);
    return;
  end;

  local function Exec_GearUp(GB, Mek)
    GearUp(Mek);
    return;
  end;

  local function Exec_MoveMode(GB, Mek)
    Mek_SetNAtt_Action_MoveMode(Mek, g_QRL_SystemVariable["ExecQueue_Type_MoveMode"]);
    return;
  end;

  local function Exec_Transformation(GB, Mek)
    DoTransformation(GB, Mek, g_QRL_SystemVariable["ExecQueue_Type_Transformation"]);
    return;
  end;

  local function Exec_PurgeParts(GB, Mek)
    DoPurgeParts(GB, Mek, g_QRL_SystemVariable["ExecQueue_Type_PurgeParts"]);
    return;
  end;

  local function Exec_Repair(GB, Mek)
    AIRepair(GB, Mek, g_QRL_SystemVariable["ExecQueue_Type_Repair_Target"], g_QRL_SystemVariable["ExecQueue_Type_Repair_Limit"], g_QRL_SystemVariable["ExecQueue_Type_Repair_Skill"]);
    return;
  end;

  local function Exec_Eject(GB, Mek)
    AI_Eject(Mek, GB);
    return;
  end;

  local switch = {
    default                                    = function(GB, Mek) print('ERROR: qrl_chip_exec() Failed.'); end;
    [c_ExecQueue_Type_NONE]                    = function(GB, Mek) Exec_NONE(GB, Mek); end;
    [c_ExecQueue_Type_PrepAction]              = function(GB, Mek) Exec_PrepAction(GB, Mek); end;
    [c_ExecQueue_Type_PrepActionStopAndGearUp] = function(GB, Mek) Exec_PrepActionStopAndGearUp(GB, Mek); end;
    [c_ExecQueue_Type_GearUp]                  = function(GB, Mek) Exec_GearUp(GB, Mek); end;
    [c_ExecQueue_Type_MoveMode]                = function(GB, Mek) Exec_MoveMode(GB, Mek); end;
    [c_ExecQueue_Type_Transformation]          = function(GB, Mek) Exec_Transformation(GB, Mek); end;
    [c_ExecQueue_Type_PurgeParts]              = function(GB, Mek) Exec_PurgeParts(GB, Mek); end;
    [c_ExecQueue_Type_Repair]                  = function(GB, Mek) Exec_Repair(GB, Mek); end;
    [c_ExecQueue_Type_Eject]                   = function(GB, Mek) Exec_Eject(GB, Mek); end;
  };
  (switch[g_QRL_SystemVariable["ExecQueue_Type"]] or switch.default)(GB, Mek);
  g_QRL_SystemVariable["ExecQueue_Type"] = c_ExecQueue_Type_NONE;

  -- Waiting for Results
  if (0 < g_QRL_SystemVariable["ExecQueue_WaitAMinute"]) then
    WaitAMinute(GB, Mek, g_QRL_SystemVariable["ExecQueue_WaitAMinute"]);
  end;
  g_QRL_SystemVariable["ExecQueue_WaitAMinute"] = 0;

  return;
end;

-- [ End of File ]
