/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.content.BroadcastReceiver;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.EditTextPreference;
import android.preference.ListPreference;
import android.preference.PreferenceActivity;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.simple_battery_logger.service.DeleteAllCsvFileCompletedReceiver;
import org.routine_work.simple_battery_logger.service.DeleteAllDataCompletedReceiver;
import org.routine_work.simple_battery_logger.service.ServiceUtils;
import org.routine_work.simple_battery_logger.util.Log;
import org.routine_work.simple_battery_logger.util.SystemResourceUtils;

/**
 *
 * @author sawai
 */
public class BatteryLoggerPreferenceActivity extends PreferenceActivity
	implements SharedPreferences.OnSharedPreferenceChangeListener
{

	private static final String LOG_TAG = "simple-battery-logger";
	private String loggingServiceEnabledKey;
	private String ignoreVoltageChangeOnlyKey;
	private String dataKeepingPeriodKey;
	private String csvAutoExportEnabledKey;
	private String csvExportDirectoryKey;
	private String csvSortOrderKey;
	private BroadcastReceiver deleteAllDataCompletedReceiver = new DeleteAllDataCompletedReceiver();
	private BroadcastReceiver deleteAllCsvFileCompletedReceiver = new DeleteAllCsvFileCompletedReceiver();

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "onCreate() Hello");
		super.onCreate(savedInstanceState);

		addPreferencesFromResource(R.xml.preferences);
		loggingServiceEnabledKey = getString(R.string.logging_service_enabled_key);
		ignoreVoltageChangeOnlyKey = getString(R.string.ignore_voltage_change_only_key);
		dataKeepingPeriodKey = getString(R.string.data_keeping_period_key);
		csvAutoExportEnabledKey = getString(R.string.csv_auto_export_enabled_key);
		csvExportDirectoryKey = getString(R.string.csv_export_directory_key);
		csvSortOrderKey = getString(R.string.csv_sort_order_key);

		updateSummary();

		Log.v(LOG_TAG, "onCreate() Bye");
	}

	@Override
	protected void onResume()
	{
		super.onResume();

		// register preference listener
		SharedPreferences preferences = getPreferenceScreen().getSharedPreferences();
		preferences.registerOnSharedPreferenceChangeListener(this);

		// register broadcast receiver
		IntentFilter intentFilter;
		intentFilter = new IntentFilter();
		intentFilter.addAction(DeleteAllDataCompletedReceiver.DELETE_ALL_DATA_COMPLETED_ACTION);
		registerReceiver(deleteAllDataCompletedReceiver, intentFilter);

		intentFilter = new IntentFilter();
		intentFilter.addAction(DeleteAllCsvFileCompletedReceiver.DELETE_ALL_CSV_FILE_COMPLETED_ACTION);
		registerReceiver(deleteAllCsvFileCompletedReceiver, intentFilter);
	}

	@Override
	protected void onPause()
	{
		// unregister broadcast receiver
		unregisterReceiver(deleteAllDataCompletedReceiver);
		unregisterReceiver(deleteAllCsvFileCompletedReceiver);

		// unregister preference listener
		SharedPreferences preferences = getPreferenceScreen().getSharedPreferences();
		preferences.unregisterOnSharedPreferenceChangeListener(this);

		super.onPause();
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = false;

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.quit_menuitem:
				setResult(AppConstants.RESULT_QUIT);
				finish();
				result = true;
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					Intent dashboardIntent = new Intent(this, DashboardActivity.class);
					dashboardIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
					startActivity(dashboardIntent);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}
		return result;
	}

	public void onSharedPreferenceChanged(SharedPreferences prefs, String key)
	{
		Log.v(LOG_TAG, "onSharedPreferenceChanged(" + key + ") Hello");
		if (loggingServiceEnabledKey.equals(key)
			|| ignoreVoltageChangeOnlyKey.equals(key))
		{
			ServiceUtils.initializeLoggingService(this);
		}
		else if (dataKeepingPeriodKey.equals(key))
		{
			ServiceUtils.initializeDeleteOldDataTask(this);
		}
		else if (csvAutoExportEnabledKey.equals(key))
		{
			ServiceUtils.initializeExportCsvFileTask(this);
		}

		updateSummary(key);

		Log.v(LOG_TAG, "onSharedPreferenceChanged() Bye");
	}

	private void updateSummary()
	{
		updateSummary(null);
	}

	private void updateSummary(String key)
	{
		if (key == null || key.equals(dataKeepingPeriodKey))
		{
			ListPreference dataKeepingPeriodPreference = (ListPreference) getPreferenceScreen().findPreference(dataKeepingPeriodKey);
			CharSequence entry = dataKeepingPeriodPreference.getEntry();
			dataKeepingPeriodPreference.setSummary(entry);
		}

		if (key == null || key.equals(csvExportDirectoryKey))
		{
			EditTextPreference csvExportDirectoryPreference = (EditTextPreference) getPreferenceScreen().findPreference(csvExportDirectoryKey);
			String csvExportDirectory = csvExportDirectoryPreference.getText();
			csvExportDirectoryPreference.setSummary(csvExportDirectory);
		}

		if (key == null || key.equals(csvSortOrderKey))
		{
			ListPreference csvSortOrderListPreference = (ListPreference) getPreferenceScreen().findPreference(csvSortOrderKey);
			CharSequence sortOrder = csvSortOrderListPreference.getValue();
			String summary = null;
			if (DBConstants.SORT_ASCENDING.equals(sortOrder))
			{
				summary = getString(R.string.csv_sort_order_summary_ascending);
			}
			else if (DBConstants.SORT_DESCENDING.equals(sortOrder))
			{
				summary = getString(R.string.csv_sort_order_summary_descending);
			}
			csvSortOrderListPreference.setSummary(summary);
		}
	}
}
