"""
Sphinx configuration file for CMake quick doc

It's used on build server to generate quick doc for specified CMake.
See https://jetbrains.team/p/ij/repositories/intellij-teamcity-config/files/.teamcity/src/cidr/externalTools/cmake/CIDR_ExternalTools_CMakeQuickDoc.kt

Example command to generate quick doc using this config:
    export \
        CMAKE_QUICK_DOC_CMAKE_SOURCES="<cmake-source-dir>" \
        CMAKE_QUICK_DOC_CMAKE_VERSION="<cmake-version>" \
        CMAKE_QUICK_DOC_THEME_PATH="<quick-doc-theme-dir>"

    sphinx-build \
        -b quickdoc \
        -j auto \
        -v \
        -c "<quick-doc-theme-dir>" \
        "<cmake-source-dir>/Help" \
        "<tmp-dir>"
"""

import os
import sys


def read_environ_safe(variable, default=None):
    try:
        return os.environ[variable]
    except KeyError:
        return default


# path to CMake sources
cmake_sources_path = os.environ["CMAKE_QUICK_DOC_CMAKE_SOURCES"]
# path to Sphinx theme used by quick doc (`CIDR/cmake-psi/src/com/jetbrains/cmake/documentation/sphinx`)
theme_path = os.environ["CMAKE_QUICK_DOC_THEME_PATH"]
# CMake version for this documentation or None for latest
cmake_version = read_environ_safe('CMAKE_QUICK_DOC_CMAKE_VERSION')

sys.path.insert(0, cmake_sources_path + '/Utilities/Sphinx')
sys.path.insert(0, theme_path + '/quick_doc')
sys.path.insert(0, theme_path)

if cmake_version is None:
    cmake_external_url = 'https://cmake.org/cmake/help/latest/'
else:
    cmake_external_url = F'https://cmake.org/cmake/help/v{cmake_version}/'

cmake_mapped_dirs = [
  'command',
  'variable',
  'module',
  'policy',
  'prop_gbl',
  'prop_dir',
  'prop_tgt',
  'prop_sf',
  'prop_inst',
  'prop_test',
  'prop_cache',
]

source_suffix = '.rst'
master_doc = 'index'

project = 'CMake'

language = 'en'
primary_domain = 'cmake'
highlight_language = 'none'

exclude_patterns = [
  'dev',  # ignore developer-only documentation
]

extensions = [
  'cmake',  # For cmake extension see CMake sources: Utilities/Sphinx/cmake.py
  'builder'  # see CMAKE_QUICK_DOC_THEME_PATH/builder.py
]

nitpicky = True
smartquotes = False

html_theme_path = [theme_path]  # template path
html_theme = 'quick_doc'  # template name in `sphinx` directory

embedded = True  # generated pages will be shown embedded, not in the browser
html_use_index = False  # don't generate index.html
html_copy_source = False  # don't copy sources into `_sources` directory
html_show_sourcelink = False  # don't put `Show Source` link onto page
html_show_copyright = False  # no copyright needed
html_show_search_summary = False  # no search summary needed
html_show_sphinx = False  # don't add 'Created using Sphinx'
html_use_opensearch = ''  # no OpenSearch
html_sidebars = {'**': []}  # try to remove all default sidebars

html_permalinks = False  # no need for permalinks on every paragraph
