/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.util.io;

import com.intellij.openapi.Forceable;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.ThrowableNotNullFunction;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ExceptionUtil;
import com.intellij.util.SmartList;
import com.intellij.util.SystemProperties;
import com.intellij.util.ThrowableRunnable;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.io.Bits;
import com.intellij.util.io.DirectBufferWrapper;
import com.intellij.util.io.FilePageCache;
import com.intellij.util.io.IOStatistics;
import com.intellij.util.io.OpenChannelsCache;
import com.intellij.util.io.PagedFileStorageCache;
import com.intellij.util.io.PersistentHashMapValueStorage;
import com.intellij.util.io.StorageLockContext;
import com.intellij.util.io.storage.AbstractStorage;
import com.intellij.util.lang.CompoundRuntimeException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class PagedFileStorage
implements Forceable {
    static final Logger LOG = Logger.getInstance(PagedFileStorage.class);
    static final OpenChannelsCache CHANNELS_CACHE = new OpenChannelsCache(SystemProperties.getIntProperty("paged.file.storage.open.channel.cache.capacity", 400));
    public static final int MB = 0x100000;
    public static final int BUFFER_SIZE = FilePageCache.BUFFER_SIZE;
    @NotNull
    private static final ThreadLocal<byte[]> ourTypedIOBuffer = ThreadLocal.withInitial(() -> new byte[8]);
    private static final StorageLockContext ourDefaultContext = new StorageLockContext(false);
    @NotNull
    public static final ThreadLocal<StorageLockContext> THREAD_LOCAL_STORAGE_LOCK_CONTEXT = new ThreadLocal();
    @NotNull
    private final StorageLockContext myStorageLockContext;
    private final boolean myNativeBytesOrder;
    private long myStorageIndex;
    @NotNull
    private final PagedFileStorageCache myLastAccessedBufferCache = new PagedFileStorageCache();
    @NotNull
    private final Path myFile;
    private final boolean myReadOnly;
    private final Object myInputStreamLock = new Object();
    protected final int myPageSize;
    protected final boolean myValuesAreBufferAligned;
    private volatile boolean isDirty;
    private volatile long mySize = -1L;
    private static final int MAX_FILLER_SIZE = 8192;

    public PagedFileStorage(@NotNull Path file2, @Nullable StorageLockContext storageLockContext, int pageSize, boolean valuesAreBufferAligned, boolean nativeBytesOrder) {
        this.myFile = file2;
        this.myReadOnly = PersistentHashMapValueStorage.CreationTimeOptions.READONLY.get() == Boolean.TRUE;
        StorageLockContext context = THREAD_LOCAL_STORAGE_LOCK_CONTEXT.get();
        if (context != null) {
            if (storageLockContext != null && storageLockContext != context) {
                throw new IllegalStateException();
            }
            storageLockContext = context;
        }
        this.myStorageLockContext = storageLockContext != null ? storageLockContext : ourDefaultContext;
        this.myPageSize = Math.max(pageSize > 0 ? pageSize : BUFFER_SIZE, AbstractStorage.PAGE_SIZE);
        this.myValuesAreBufferAligned = valuesAreBufferAligned;
        this.myStorageIndex = this.myStorageLockContext.getBufferCache().registerPagedFileStorage(this);
        this.myNativeBytesOrder = nativeBytesOrder;
    }

    public int getPageSize() {
        return this.myPageSize;
    }

    public void lockRead() {
        this.myStorageLockContext.lockRead();
    }

    public void unlockRead() {
        this.myStorageLockContext.unlockRead();
    }

    public void lockWrite() {
        this.myStorageLockContext.lockWrite();
    }

    public void unlockWrite() {
        this.myStorageLockContext.unlockWrite();
    }

    @NotNull
    public StorageLockContext getStorageLockContext() {
        return this.myStorageLockContext;
    }

    @NotNull
    Path getFile() {
        return this.myFile;
    }

    public boolean isNativeBytesOrder() {
        return this.myNativeBytesOrder;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public <R> R readInputStream(@NotNull ThrowableNotNullFunction<? super InputStream, R, ? extends IOException> consumer) throws IOException {
        Object object = this.myInputStreamLock;
        synchronized (object) {
            try {
                return (R)this.useChannel(ch -> {
                    ch.position(0L);
                    return consumer.fun(Channels.newInputStream(ch));
                }, true);
            }
            catch (NoSuchFileException ignored) {
                return consumer.fun(new ByteArrayInputStream(ArrayUtil.EMPTY_BYTE_ARRAY));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public <R> R readChannel(@NotNull ThrowableNotNullFunction<? super ReadableByteChannel, R, ? extends IOException> consumer) throws IOException {
        Object object = this.myInputStreamLock;
        synchronized (object) {
            try {
                return (R)this.useChannel(ch -> {
                    ch.position(0L);
                    return consumer.fun(ch);
                }, true);
            }
            catch (NoSuchFileException ignored) {
                return consumer.fun(Channels.newChannel(new ByteArrayInputStream(ArrayUtil.EMPTY_BYTE_ARRAY)));
            }
        }
    }

    <R> R useChannel(@NotNull OpenChannelsCache.ChannelProcessor<R> processor, boolean read) throws IOException {
        if (this.myStorageLockContext.useChannelCache()) {
            return CHANNELS_CACHE.useChannel(this.myFile, processor, read);
        }
        this.myStorageLockContext.getBufferCache().incrementUncachedFileAccess();
        try (OpenChannelsCache.ChannelDescriptor desc = new OpenChannelsCache.ChannelDescriptor(this.myFile, read);){
            R r = processor.process(desc.getChannel());
            return r;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void putInt(long addr, int value) throws IOException {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            DirectBufferWrapper buffer = this.getBuffer(page);
            try {
                buffer.putInt(page_offset, value);
            }
            finally {
                buffer.unlock();
            }
        } else {
            Bits.putInt(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, value);
            this.put(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 4);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getInt(long addr) throws IOException {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            DirectBufferWrapper buffer = this.getReadOnlyBuffer(page);
            try {
                int n = buffer.getInt(page_offset);
                return n;
            }
            finally {
                buffer.unlock();
            }
        }
        this.get(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 4);
        return Bits.getInt(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0);
    }

    public int getOffsetInPage(long addr) {
        return (int)(addr % (long)this.myPageSize);
    }

    public DirectBufferWrapper getByteBuffer(long address, boolean modify) throws IOException {
        long page = address / (long)this.myPageSize;
        assert (page >= 0L && page <= 0xFFFFFFFFL) : address + " in " + this.myFile;
        return this.getBufferWrapper(page, modify);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void putLong(long addr, long value) throws IOException {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            DirectBufferWrapper buffer = this.getBuffer(page);
            try {
                buffer.putLong(page_offset, value);
            }
            finally {
                buffer.unlock();
            }
        } else {
            Bits.putLong(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, value);
            this.put(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 8);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getLong(long addr) throws IOException {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            DirectBufferWrapper buffer = this.getReadOnlyBuffer(page);
            try {
                long l = buffer.getLong(page_offset);
                return l;
            }
            finally {
                buffer.unlock();
            }
        }
        this.get(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 8);
        return Bits.getLong(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte get(long index) throws IOException {
        long page = index / (long)this.myPageSize;
        int offset = (int)(index % (long)this.myPageSize);
        DirectBufferWrapper buffer = this.getReadOnlyBuffer(page);
        try {
            byte by = buffer.get(offset);
            return by;
        }
        finally {
            buffer.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void put(long index, byte value) throws IOException {
        long page = index / (long)this.myPageSize;
        int offset = (int)(index % (long)this.myPageSize);
        DirectBufferWrapper buffer = this.getBuffer(page);
        try {
            buffer.put(offset, value);
        }
        finally {
            buffer.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void get(long index, byte[] dst, int offset, int length) throws IOException {
        long i = index;
        int o = offset;
        int l = length;
        while (l > 0) {
            long page = i / (long)this.myPageSize;
            int page_offset = (int)(i % (long)this.myPageSize);
            int page_len = Math.min(l, this.myPageSize - page_offset);
            DirectBufferWrapper buffer = this.getReadOnlyBuffer(page);
            try {
                buffer.readToArray(dst, o, page_offset, page_len);
            }
            finally {
                buffer.unlock();
            }
            l -= page_len;
            o += page_len;
            i += (long)page_len;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void put(long index, byte[] src, int offset, int length) throws IOException {
        long i = index;
        int o = offset;
        int l = length;
        while (l > 0) {
            long page = i / (long)this.myPageSize;
            int page_offset = (int)(i % (long)this.myPageSize);
            int page_len = Math.min(l, this.myPageSize - page_offset);
            DirectBufferWrapper buffer = this.getBuffer(page);
            try {
                buffer.putFromArray(src, o, page_offset, page_len);
            }
            finally {
                buffer.unlock();
            }
            l -= page_len;
            o += page_len;
            i += (long)page_len;
        }
    }

    public void close() throws IOException {
        SmartList exceptions = new SmartList();
        ContainerUtil.addIfNotNull(exceptions, ExceptionUtil.runAndCatch((ThrowableRunnable<? extends Exception>)((ThrowableRunnable)() -> this.force())));
        ContainerUtil.addIfNotNull(exceptions, ExceptionUtil.runAndCatch((ThrowableRunnable<? extends Exception>)((ThrowableRunnable)() -> {
            this.unmapAll();
            this.myStorageLockContext.getBufferCache().removeStorage(this.myStorageIndex);
            this.myStorageIndex = -1L;
        })));
        ContainerUtil.addIfNotNull(exceptions, ExceptionUtil.runAndCatch((ThrowableRunnable<? extends Exception>)((ThrowableRunnable)() -> CHANNELS_CACHE.closeChannel(this.myFile))));
        if (!exceptions.isEmpty()) {
            throw new IOException(new CompoundRuntimeException(exceptions));
        }
    }

    private void unmapAll() {
        this.myStorageLockContext.getBufferCache().unmapBuffersForOwner(this);
        this.myLastAccessedBufferCache.clear();
    }

    public void resize(long newSize) throws IOException {
        long oldSize;
        if (Files.exists(this.myFile, new LinkOption[0])) {
            oldSize = Files.size(this.myFile);
        } else {
            Files.createDirectories(this.myFile.getParent(), new FileAttribute[0]);
            oldSize = 0L;
        }
        if (oldSize == newSize && oldSize == this.length()) {
            return;
        }
        this.resizeFile(newSize);
        long delta = newSize - oldSize;
        if (delta > 0L) {
            this.fillWithZeros(oldSize, delta);
        }
    }

    private void resizeFile(long newSize) throws IOException {
        this.mySize = -1L;
        try (RandomAccessFile raf = new RandomAccessFile(this.myFile.toFile(), "rw");){
            raf.setLength(newSize);
        }
        this.mySize = newSize;
    }

    private void fillWithZeros(long from, long length) throws IOException {
        byte[] buff = new byte[8192];
        Arrays.fill(buff, (byte)0);
        while (length > 0L) {
            int filled = Math.min((int)length, 8192);
            this.put(from, buff, 0, filled);
            length -= (long)filled;
            from += (long)filled;
        }
    }

    public final long length() {
        long size = this.mySize;
        if (size == -1L) {
            if (Files.exists(this.myFile, new LinkOption[0])) {
                try {
                    this.mySize = size = Files.size(this.myFile);
                }
                catch (IOException e) {
                    LOG.error(e);
                }
            } else {
                size = 0L;
                this.mySize = 0L;
            }
        }
        return size;
    }

    private DirectBufferWrapper getBuffer(long page) throws IOException {
        return this.getBufferWrapper(page, true);
    }

    private DirectBufferWrapper getReadOnlyBuffer(long page) throws IOException {
        return this.getBufferWrapper(page, false);
    }

    private DirectBufferWrapper getBufferWrapper(long page, boolean modify) throws IOException {
        DirectBufferWrapper wrapper;
        do {
            wrapper = this.doGetBufferWrapper(page, modify);
            assert (this == wrapper.getFile());
        } while (!wrapper.tryLock());
        return wrapper;
    }

    @NotNull
    private DirectBufferWrapper doGetBufferWrapper(long page, boolean modify) throws IOException {
        DirectBufferWrapper pageFromCache = this.myLastAccessedBufferCache.getPageFromCache(page);
        if (this.myReadOnly && modify) {
            throw new IOException("Read-only storage can't be modified");
        }
        if (pageFromCache != null) {
            this.myStorageLockContext.getBufferCache().incrementFastCacheHitsCount();
            return pageFromCache;
        }
        if (page < 0L || page > 0xFFFFFFFFL) {
            throw new AssertionError(page);
        }
        if (this.myStorageIndex == -1L) {
            throw new IOException("storage is already closed; path " + this.myFile);
        }
        DirectBufferWrapper byteBufferWrapper = this.myStorageLockContext.getBufferCache().get(this.myStorageIndex | page, !modify);
        this.myLastAccessedBufferCache.updateCache(page, byteBufferWrapper);
        return byteBufferWrapper;
    }

    void markDirty() {
        if (!this.isDirty) {
            this.isDirty = true;
        }
    }

    public void ensureCachedSizeAtLeast(long size) {
        if (this.mySize < size) {
            this.mySize = size;
        }
    }

    public boolean isReadOnly() {
        return this.myReadOnly;
    }

    boolean useNativeByteOrder() {
        return this.myNativeBytesOrder;
    }

    private static byte[] getThreadLocalTypedIOBuffer() {
        return ourTypedIOBuffer.get();
    }

    @Override
    public void force() throws IOException {
        long finished;
        long started;
        long l = started = IOStatistics.DEBUG ? System.currentTimeMillis() : 0L;
        if (this.isDirty) {
            this.myStorageLockContext.getBufferCache().flushBuffersForOwner(this);
            this.isDirty = false;
        }
        if (IOStatistics.DEBUG && (finished = System.currentTimeMillis()) - started > 100L) {
            IOStatistics.dump("Flushed " + this.myFile + " for " + (finished - started));
        }
    }

    @Override
    public boolean isDirty() {
        return this.isDirty;
    }

    public String toString() {
        return "PagedFileStorage[" + this.myFile + "]";
    }
}

