/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.execution.process.impl;

import com.intellij.execution.ExecutionException;
import com.intellij.execution.configurations.GeneralCommandLine;
import com.intellij.execution.process.ProcessInfo;
import com.intellij.execution.process.ProcessOutput;
import com.intellij.execution.process.impl.CSVReader;
import com.intellij.execution.util.ExecUtil;
import com.intellij.openapi.application.Application;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.application.PathManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.NullableFunction;
import com.intellij.util.PathUtil;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class ProcessListUtil {
    private static final Logger LOG = Logger.getInstance(ProcessListUtil.class);
    private static final String WIN_PROCESS_LIST_HELPER_FILENAME = "WinProcessListHelper.exe";
    public static final List<@NlsSafe String> COMM_LIST_COMMAND = List.of("/bin/ps", "-a", "-x", "-o", "pid,state,user,comm");
    public static final List<@NlsSafe String> COMMAND_LIST_COMMAND = List.of("/bin/ps", "-a", "-x", "-o", "pid,state,user,command");

    public static ProcessInfo @NotNull [] getProcessList() {
        List<ProcessInfo> result = ProcessListUtil.doGetProcessList();
        return result.toArray(ProcessInfo.EMPTY_ARRAY);
    }

    @NotNull
    private static List<ProcessInfo> doGetProcessList() {
        if (SystemInfo.isWindows) {
            List<ProcessInfo> result = ProcessListUtil.getProcessListUsingWinProcessListHelper();
            if (result != null) {
                return result;
            }
            LOG.info("Cannot get process list via WinProcessListHelper.exe, fallback to wmic");
            result = ProcessListUtil.getProcessListUsingWindowsWMIC();
            if (result != null) {
                return result;
            }
            LOG.info("Cannot get process list via wmic, fallback to tasklist");
            result = ProcessListUtil.getProcessListUsingWindowsTaskList();
            if (result != null) {
                return result;
            }
            LOG.error("Cannot get process list via wmic and tasklist");
        } else if (SystemInfo.isUnix) {
            List<ProcessInfo> result = SystemInfo.isMac ? ProcessListUtil.getProcessListOnMac() : ProcessListUtil.getProcessListOnUnix();
            if (result != null) {
                return result;
            }
            LOG.error("Cannot get process list");
        } else {
            LOG.error("Cannot get process list, unexpected platform: " + SystemInfo.OS_NAME);
        }
        return Collections.emptyList();
    }

    @Nullable
    private static List<ProcessInfo> parseCommandOutput(@NotNull List<@NlsSafe String> command, @NotNull NullableFunction<? super String, ? extends List<ProcessInfo>> parser) {
        return ProcessListUtil.parseCommandOutput(command, parser, null);
    }

    @Nullable
    private static List<ProcessInfo> parseCommandOutput(@NotNull List<@NlsSafe String> command, @NotNull NullableFunction<? super String, ? extends List<ProcessInfo>> parser, @Nullable Charset charset) {
        String output;
        try {
            ProcessOutput processOutput;
            int exitCode;
            GeneralCommandLine commandLine = new GeneralCommandLine(command);
            if (charset != null) {
                commandLine.withCharset(charset);
            }
            if ((exitCode = (processOutput = ExecUtil.execAndGetOutput(commandLine)).getExitCode()) != 0) {
                LOG.error("Cannot get process list, command '" + StringUtil.join(command, " ") + "' exited with code " + exitCode + ", stdout:\n" + processOutput.getStdout() + "\nstderr:\n" + processOutput.getStderr());
            }
            output = processOutput.getStdout();
        }
        catch (ExecutionException e) {
            LOG.error("Cannot get process list", e);
            return null;
        }
        return (List)parser.fun((Object)output);
    }

    @Nullable
    private static List<ProcessInfo> getProcessListOnUnix() {
        File proc = new File("/proc");
        File[] processes = proc.listFiles();
        if (processes == null) {
            LOG.error("Cannot read /proc, not mounted?");
            return null;
        }
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        for (File each : processes) {
            List<String> cmdline;
            int pid = StringUtil.parseInt(each.getName(), -1);
            if (pid == -1) continue;
            try (FileInputStream stream = new FileInputStream(new File(each, "cmdline"));){
                String cmdlineString = new String(FileUtil.loadBytes(stream), StandardCharsets.UTF_8);
                cmdline = StringUtil.split(cmdlineString, "\u0000");
            }
            catch (IOException e) {
                continue;
            }
            if (cmdline.isEmpty()) continue;
            String executablePath = null;
            try {
                File exe = new File(each, "exe");
                if (!exe.getAbsolutePath().equals(exe.getCanonicalPath())) {
                    executablePath = exe.getCanonicalPath();
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            result.add(new ProcessInfo(pid, StringUtil.join(cmdline, " "), PathUtil.getFileName(cmdline.get(0)), StringUtil.join(cmdline.subList(1, cmdline.size()), " "), executablePath));
        }
        return result;
    }

    @Nullable
    private static List<ProcessInfo> getProcessListOnMac() {
        return ProcessListUtil.parseCommandOutput(COMM_LIST_COMMAND, (NullableFunction<? super String, ? extends List<ProcessInfo>>)((NullableFunction)commandOnly -> ProcessListUtil.parseCommandOutput(COMMAND_LIST_COMMAND, (NullableFunction<? super String, ? extends List<ProcessInfo>>)((NullableFunction)full -> ProcessListUtil.parseMacOutput(commandOnly, full)))));
    }

    @Nullable
    public static List<ProcessInfo> parseMacOutput(@NotNull String commandOnly, @NotNull String full) {
        List<MacProcessInfo> commands = ProcessListUtil.doParseMacOutput(commandOnly);
        List<MacProcessInfo> fulls = ProcessListUtil.doParseMacOutput(full);
        if (commands == null || fulls == null) {
            return null;
        }
        Int2ObjectOpenHashMap idToCommand = new Int2ObjectOpenHashMap();
        for (MacProcessInfo each : commands) {
            idToCommand.put(each.pid, (Object)each.commandLine);
        }
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        for (MacProcessInfo each : fulls) {
            String command;
            if (!idToCommand.containsKey(each.pid) || !each.commandLine.equals(command = (String)idToCommand.get(each.pid)) && !each.commandLine.startsWith(command + " ")) continue;
            String name = PathUtil.getFileName(command);
            String args2 = each.commandLine.substring(command.length()).trim();
            result.add(new ProcessInfo(each.pid, each.commandLine, name, args2, command));
        }
        return result;
    }

    @Nullable
    public static List<ProcessInfo> parseLinuxOutputMacStyle(@NotNull String commandOnly, @NotNull String full) {
        List<MacProcessInfo> commands = ProcessListUtil.doParseMacOutput(commandOnly);
        if (commands == null) {
            LOG.debug("Failed to parse commands output: ", commandOnly);
            return null;
        }
        List<MacProcessInfo> fulls = ProcessListUtil.doParseMacOutput(full);
        if (fulls == null) {
            LOG.debug("Failed to parse comm output: ", full);
            return null;
        }
        Int2ObjectOpenHashMap idToCommand = new Int2ObjectOpenHashMap();
        for (MacProcessInfo each : commands) {
            idToCommand.put(each.pid, (Object)each.commandLine);
        }
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        for (MacProcessInfo each : fulls) {
            if (!idToCommand.containsKey(each.pid)) continue;
            String command = (String)idToCommand.get(each.pid);
            String name = PathUtil.getFileName(command);
            String args2 = each.commandLine.startsWith(command) ? each.commandLine.substring(command.length()).trim() : each.commandLine;
            result.add(new ProcessInfo(each.pid, each.commandLine, name, args2, command));
        }
        return result;
    }

    @Nullable
    private static List<MacProcessInfo> doParseMacOutput(@NlsSafe String output) {
        ArrayList<MacProcessInfo> result = new ArrayList<MacProcessInfo>();
        String[] lines = StringUtil.splitByLinesDontTrim(output);
        if (lines.length == 0) {
            return null;
        }
        @NlsSafe String header = lines[0];
        int pidStart = header.indexOf("PID");
        if (pidStart == -1) {
            return null;
        }
        int statStart = header.indexOf("S", pidStart);
        if (statStart == -1) {
            return null;
        }
        int userStart = header.indexOf("USER", statStart);
        if (userStart == -1) {
            return null;
        }
        int commandStart = header.indexOf("COMM", userStart);
        if (commandStart == -1) {
            return null;
        }
        for (int i = 1; i < lines.length; ++i) {
            String line = lines[i];
            try {
                String state;
                int pid = StringUtil.parseInt(line.substring(0, statStart).trim(), -1);
                if (pid == -1 || (state = line.substring(statStart, userStart).trim()).contains("Z")) continue;
                String user = line.substring(userStart, commandStart).trim();
                String commandLine = line.substring(commandStart).trim();
                result.add(new MacProcessInfo(pid, commandLine, user, state));
                continue;
            }
            catch (Exception e) {
                LOG.error("Can't parse line '" + line + "'", e);
            }
        }
        return result;
    }

    @Nullable
    private static List<ProcessInfo> getProcessListUsingWinProcessListHelper() {
        Path exeFile = ProcessListUtil.findWinProcessListHelperFile();
        if (exeFile == null) {
            return null;
        }
        return ProcessListUtil.parseCommandOutput(Collections.singletonList(exeFile.toAbsolutePath().toString()), (NullableFunction<? super String, ? extends List<ProcessInfo>>)((NullableFunction)ProcessListUtil::parseWinProcessListHelperOutput), StandardCharsets.UTF_8);
    }

    private static void logErrorTestSafe(@NonNls String message) {
        Application application = ApplicationManager.getApplication();
        if (application == null || application.isUnitTestMode()) {
            LOG.warn(message);
        } else {
            LOG.error(message);
        }
    }

    @Nullable
    private static String unescapeString(@Nullable String str) {
        if (str == null) {
            return null;
        }
        StringBuilder builder = new StringBuilder();
        for (int index = 0; index < str.length(); ++index) {
            if (str.charAt(index) == '\\') {
                if (index == str.length() - 1) {
                    ProcessListUtil.logErrorTestSafe("Invalid escaped string: backslash at the last position");
                    LOG.debug(str);
                    return null;
                }
                switch (str.charAt(index + 1)) {
                    case '\\': {
                        builder.append('\\');
                        break;
                    }
                    case 'n': {
                        builder.append('\n');
                        break;
                    }
                    case 'r': {
                        builder.append('\r');
                        break;
                    }
                    default: {
                        ProcessListUtil.logErrorTestSafe("Invalid character after an escape symbol: " + str.charAt(index + 1));
                        LOG.debug(str);
                        return null;
                    }
                }
                ++index;
                continue;
            }
            builder.append(str.charAt(index));
        }
        return builder.toString();
    }

    @Nullable
    private static String removePrefix(String str, @NonNls String prefix) {
        if (str.startsWith(prefix)) {
            return str.substring(prefix.length());
        }
        ProcessListUtil.logErrorTestSafe("Can't remove prefix \"" + prefix + "\"");
        LOG.debug(str);
        return null;
    }

    @Nullable
    static List<ProcessInfo> parseWinProcessListHelperOutput(@NotNull String output) {
        String[] lines = StringUtil.splitByLines(output, false);
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        if (lines.length % 3 != 0) {
            ProcessListUtil.logErrorTestSafe("Broken output of WinProcessListHelper.exe: output line count is not a multiple of 3");
            LOG.debug(output);
            return null;
        }
        int processCount = lines.length / 3;
        for (int i = 0; i < processCount; ++i) {
            String args2;
            int offset = i * 3;
            String idString = ProcessListUtil.removePrefix(lines[offset], "pid:");
            int id = StringUtil.parseInt(idString, -1);
            if (id == -1) {
                ProcessListUtil.logErrorTestSafe("Broken output of WinProcessListHelper.exe: process ID is not a number: " + lines[offset]);
                LOG.debug(output);
                return null;
            }
            if (id == 0) continue;
            String name = ProcessListUtil.unescapeString(ProcessListUtil.removePrefix(lines[offset + 1], "name:"));
            if (name == null) {
                ProcessListUtil.logErrorTestSafe("Failed to read a process name: " + lines[offset + 1]);
                LOG.debug(output);
                return null;
            }
            if (name.isEmpty()) continue;
            String commandLine = ProcessListUtil.unescapeString(ProcessListUtil.removePrefix(lines[offset + 2], "cmd:"));
            if (commandLine == null) {
                ProcessListUtil.logErrorTestSafe("Failed to read a process command line: " + lines[offset + 2]);
                LOG.debug(output);
                return null;
            }
            if (commandLine.isEmpty()) {
                commandLine = name;
                args2 = "";
            } else {
                args2 = ProcessListUtil.extractCommandLineArgs(commandLine, name);
            }
            result.add(new ProcessInfo(id, commandLine, name, args2));
        }
        return result;
    }

    @NotNull
    private static String extractCommandLineArgs(@NotNull String fullCommandLine, @NotNull String executableName) {
        List<String> commandLineList = StringUtil.splitHonorQuotes(fullCommandLine, ' ');
        if (commandLineList.isEmpty()) {
            return "";
        }
        String first = StringUtil.unquoteString(commandLineList.get(0));
        if (StringUtil.endsWithIgnoreCase(first, executableName)) {
            List<String> argsList = commandLineList.subList(1, commandLineList.size());
            return StringUtil.join(argsList, " ");
        }
        return "";
    }

    @Nullable
    private static Path findWinProcessListHelperFile() {
        try {
            return PathManager.findBinFileWithException(WIN_PROCESS_LIST_HELPER_FILENAME);
        }
        catch (RuntimeException e) {
            LOG.error(e);
            return null;
        }
    }

    @Nullable
    static List<ProcessInfo> getProcessListUsingWindowsWMIC() {
        return ProcessListUtil.parseCommandOutput(Arrays.asList("wmic.exe", "path", "win32_process", "get", "Caption,Processid,Commandline,ExecutablePath"), (NullableFunction<? super String, ? extends List<ProcessInfo>>)((NullableFunction)ProcessListUtil::parseWMICOutput));
    }

    @Nullable
    static List<ProcessInfo> parseWMICOutput(@NotNull String output) {
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        String[] lines = StringUtil.splitByLinesDontTrim(output);
        if (lines.length == 0) {
            return null;
        }
        String header = lines[0];
        int commandLineStart = header.indexOf("CommandLine");
        if (commandLineStart == -1) {
            return null;
        }
        int pidStart = header.indexOf("ProcessId");
        if (pidStart == -1) {
            return null;
        }
        int executablePathStart = header.indexOf("ExecutablePath");
        if (executablePathStart == -1) {
            return null;
        }
        for (int i = 1; i < lines.length; ++i) {
            String line = lines[i];
            int pid = StringUtil.parseInt(line.substring(pidStart).trim(), -1);
            if (pid == -1 || pid == 0) continue;
            String executablePath = line.substring(executablePathStart, pidStart).trim();
            String name = line.substring(0, commandLineStart).trim();
            if (name.isEmpty()) continue;
            String commandLine = line.substring(commandLineStart, executablePathStart).trim();
            String args2 = "";
            if (commandLine.isEmpty()) {
                commandLine = name;
            } else {
                args2 = ProcessListUtil.extractCommandLineArgs(commandLine, name);
            }
            result.add(new ProcessInfo(pid, commandLine, name, args2, executablePath));
        }
        return result;
    }

    @Nullable
    static List<ProcessInfo> getProcessListUsingWindowsTaskList() {
        return ProcessListUtil.parseCommandOutput(Arrays.asList("tasklist.exe", "/fo", "csv", "/nh", "/v"), (NullableFunction<? super String, ? extends List<ProcessInfo>>)((NullableFunction)ProcessListUtil::parseListTasksOutput));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    static List<ProcessInfo> parseListTasksOutput(@NotNull String output) {
        ArrayList<ProcessInfo> result = new ArrayList<ProcessInfo>();
        CSVReader reader = new CSVReader(new StringReader(output));
        try {
            while (true) {
                String[] next;
                if ((next = reader.readNext()) != null) {
                    String name;
                    if (next.length < 2) {
                        List<ProcessInfo> list = null;
                        return list;
                    }
                    int pid = StringUtil.parseInt(next[1], -1);
                    if (pid == -1 || (name = next[0]).isEmpty()) continue;
                    result.add(new ProcessInfo(pid, name, name, ""));
                    continue;
                }
                break;
            }
        }
        catch (IOException e) {
            LOG.error("Cannot parse listtasks output", e);
            List<ProcessInfo> list = null;
            return list;
        }
        finally {
            try {
                reader.close();
            }
            catch (IOException iOException) {}
        }
        return result;
    }

    private static class MacProcessInfo {
        final int pid;
        final String commandLine;
        final String user;
        final String state;

        MacProcessInfo(int pid, String commandLine, String user, String state) {
            this.pid = pid;
            this.commandLine = commandLine;
            this.user = user;
            this.state = state;
        }
    }
}

