pragma Ada_2012;
pragma Style_Checks (Off);

with Interfaces.C; use Interfaces.C;
with SDL_stdinc_h;
with stddef_h;
with System;

package SDL_cpuinfo_h is

   SDL_CACHELINE_SIZE : constant := 128;  --  ../../inc10/SDL2/SDL_cpuinfo.h:114

  --  Simple DirectMedia Layer
  --  Copyright (C) 1997-2019 Sam Lantinga <slouken@libsdl.org>
  --  This software is provided 'as-is', without any express or implied
  --  warranty.  In no event will the authors be held liable for any damages
  --  arising from the use of this software.
  --  Permission is granted to anyone to use this software for any purpose,
  --  including commercial applications, and to alter it and redistribute it
  --  freely, subject to the following restrictions:
  --  1. The origin of this software must not be misrepresented; you must not
  --     claim that you wrote the original software. If you use this software
  --     in a product, an acknowledgment in the product documentation would be
  --     appreciated but is not required.
  --  2. Altered source versions must be plainly marked as such, and must not be
  --     misrepresented as being the original software.
  --  3. This notice may not be removed or altered from any source distribution.
  -- 

  --*
  -- *  \file SDL_cpuinfo.h
  -- *
  -- *  CPU feature detection for SDL.
  --  

  -- Need to do this here because intrin.h has C++ code in it  
  -- Visual Studio 2005 has a bug where intrin.h conflicts with winnt.h  
  -- Many of the intrinsics SDL uses are not implemented by clang with Visual Studio  
  -- altivec.h redefining bool causes a number of problems, see bugs 3993 and 4392, so you need to explicitly define SDL_ENABLE_ALTIVEC_H to have it included.  
  -- Visual Studio doesn't define __ARM_ARCH, but _M_ARM (if set, always 7), and _M_ARM64 (if set, always 1).  
  -- Set up for C function definitions, even when using C++  
  -- This is a guess for the cacheline size used for padding.
  -- * Most x86 processors have a 64 byte cache line.
  -- * The 64-bit PowerPC processors have a 128 byte cache line.
  -- * We'll use the larger value to be generally safe.
  --  

  --*
  -- *  This function returns the number of CPU cores available.
  --  

   function SDL_GetCPUCount return int  -- ../../inc10/SDL2/SDL_cpuinfo.h:119
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_GetCPUCount";

  --*
  -- *  This function returns the L1 cache line size of the CPU
  -- *
  -- *  This is useful for determining multi-threaded structure padding
  -- *  or SIMD prefetch sizes.
  --  

   function SDL_GetCPUCacheLineSize return int  -- ../../inc10/SDL2/SDL_cpuinfo.h:127
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_GetCPUCacheLineSize";

  --*
  -- *  This function returns true if the CPU has the RDTSC instruction.
  --  

   function SDL_HasRDTSC return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:132
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasRDTSC";

  --*
  -- *  This function returns true if the CPU has AltiVec features.
  --  

   function SDL_HasAltiVec return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:137
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasAltiVec";

  --*
  -- *  This function returns true if the CPU has MMX features.
  --  

   function SDL_HasMMX return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:142
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasMMX";

  --*
  -- *  This function returns true if the CPU has 3DNow! features.
  --  

   function SDL_Has3DNow return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:147
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_Has3DNow";

  --*
  -- *  This function returns true if the CPU has SSE features.
  --  

   function SDL_HasSSE return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:152
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasSSE";

  --*
  -- *  This function returns true if the CPU has SSE2 features.
  --  

   function SDL_HasSSE2 return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:157
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasSSE2";

  --*
  -- *  This function returns true if the CPU has SSE3 features.
  --  

   function SDL_HasSSE3 return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:162
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasSSE3";

  --*
  -- *  This function returns true if the CPU has SSE4.1 features.
  --  

   function SDL_HasSSE41 return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:167
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasSSE41";

  --*
  -- *  This function returns true if the CPU has SSE4.2 features.
  --  

   function SDL_HasSSE42 return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:172
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasSSE42";

  --*
  -- *  This function returns true if the CPU has AVX features.
  --  

   function SDL_HasAVX return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:177
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasAVX";

  --*
  -- *  This function returns true if the CPU has AVX2 features.
  --  

   function SDL_HasAVX2 return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:182
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasAVX2";

  --*
  -- *  This function returns true if the CPU has AVX-512F (foundation) features.
  --  

   function SDL_HasAVX512F return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:187
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasAVX512F";

  --*
  -- *  This function returns true if the CPU has NEON (ARM SIMD) features.
  --  

   function SDL_HasNEON return SDL_stdinc_h.SDL_bool  -- ../../inc10/SDL2/SDL_cpuinfo.h:192
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_HasNEON";

  --*
  -- *  This function returns the amount of RAM configured in the system, in MB.
  --  

   function SDL_GetSystemRAM return int  -- ../../inc10/SDL2/SDL_cpuinfo.h:197
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_GetSystemRAM";

  --*
  -- * \brief Report the alignment this system needs for SIMD allocations.
  -- *
  -- * This will return the minimum number of bytes to which a pointer must be
  -- *  aligned to be compatible with SIMD instructions on the current machine.
  -- *  For example, if the machine supports SSE only, it will return 16, but if
  -- *  it supports AVX-512F, it'll return 64 (etc). This only reports values for
  -- *  instruction sets SDL knows about, so if your SDL build doesn't have
  -- *  SDL_HasAVX512F(), then it might return 16 for the SSE support it sees and
  -- *  not 64 for the AVX-512 instructions that exist but SDL doesn't know about.
  -- *  Plan accordingly.
  --  

   function SDL_SIMDGetAlignment return stddef_h.size_t  -- ../../inc10/SDL2/SDL_cpuinfo.h:211
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_SIMDGetAlignment";

  --*
  -- * \brief Allocate memory in a SIMD-friendly way.
  -- *
  -- * This will allocate a block of memory that is suitable for use with SIMD
  -- *  instructions. Specifically, it will be properly aligned and padded for
  -- *  the system's supported vector instructions.
  -- *
  -- * The memory returned will be padded such that it is safe to read or write
  -- *  an incomplete vector at the end of the memory block. This can be useful
  -- *  so you don't have to drop back to a scalar fallback at the end of your
  -- *  SIMD processing loop to deal with the final elements without overflowing
  -- *  the allocated buffer.
  -- *
  -- * You must free this memory with SDL_FreeSIMD(), not free() or SDL_free()
  -- *  or delete[], etc.
  -- *
  -- * Note that SDL will only deal with SIMD instruction sets it is aware of;
  -- *  for example, SDL 2.0.8 knows that SSE wants 16-byte vectors
  -- *  (SDL_HasSSE()), and AVX2 wants 32 bytes (SDL_HasAVX2()), but doesn't
  -- *  know that AVX-512 wants 64. To be clear: if you can't decide to use an
  -- *  instruction set with an SDL_Has*() function, don't use that instruction
  -- *  set with memory allocated through here.
  -- *
  -- * SDL_AllocSIMD(0) will return a non-NULL pointer, assuming the system isn't
  -- *  out of memory.
  -- *
  -- *  \param len The length, in bytes, of the block to allocated. The actual
  -- *             allocated block might be larger due to padding, etc.
  -- * \return Pointer to newly-allocated block, NULL if out of memory.
  -- *
  -- * \sa SDL_SIMDAlignment
  -- * \sa SDL_SIMDFree
  --  

   function SDL_SIMDAlloc (len : stddef_h.size_t) return System.Address  -- ../../inc10/SDL2/SDL_cpuinfo.h:246
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_SIMDAlloc";

  --*
  -- * \brief Deallocate memory obtained from SDL_SIMDAlloc
  -- *
  -- * It is not valid to use this function on a pointer from anything but
  -- *  SDL_SIMDAlloc(). It can't be used on pointers from malloc, realloc,
  -- *  SDL_malloc, memalign, new[], etc.
  -- *
  -- * However, SDL_SIMDFree(NULL) is a legal no-op.
  -- *
  -- * \sa SDL_SIMDAlloc
  --  

   procedure SDL_SIMDFree (ptr : System.Address)  -- ../../inc10/SDL2/SDL_cpuinfo.h:259
   with Import => True, 
        Convention => C, 
        External_Name => "SDL_SIMDFree";

  -- vi: set ts=4 sw=4 expandtab:  
  -- Ends C function definitions when using C++  
  -- vi: set ts=4 sw=4 expandtab:  
end SDL_cpuinfo_h;
