
--
-- Copyright (C) 2022  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--

with system;
with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

with interfaces.c;
use type interfaces.c.unsigned_short;


with ada.finalization;
with unchecked_deallocation;

	with text_io;

with matutils;  use matutils;


package body mroomobj is -- for rectangular textured interiors

-- this variation maps multiple copies of texture to walls



procedure initialize( rm: in out room ) is
begin
	rm.vert := new varray;
	rm.txuv := new tarray;
	rm.elem := new earray;
	rm.norm := new narray;
end initialize;

procedure nfree is new unchecked_deallocation(narray,nap);
procedure vfree is new unchecked_deallocation(varray,vap);
procedure tfree is new unchecked_deallocation(tarray,tap);
procedure efree is new unchecked_deallocation(earray,eap);

procedure finalize( rm: in out room ) is
begin
	vfree( rm.vert );
	tfree( rm.txuv );
	efree( rm.elem );
	nfree( rm.norm );
	--text_io.put_line("room Free");
end finalize;













procedure setroomwithfloorgap( 
	rm: in out room; 
	dx,dy,dz, z1,z2, xc,yc,zc, xr,yr,zr : float ) is

xm,xp,ym,yp,zm1,zp1, zm2,zp2 : float;

	j,k : integer := 0;
	jj : glushort;

	ax,ay,az,bx,by,bz,nx,ny,nz: float;


	-- dungeon dimensions xr,yr,zr = 10x3x20
	-- lev=3:
	--dx : constant float := 0.25;
	--dy : constant float := 0.25;
	--dz : constant float := 0.25;
	-- z1=3.0  z2=16.0  

begin

	xm  := xc-xr;
	xp  := xc+xr;
	ym  := yc-yr;
	yp  := yc+yr;

	zm1 := zc-zr; -- -20
	zp1 := z1; --  3
	zm2 := z2; -- 16
	zp2 := zc+zr; -- +20

	if    z1>=z2 then
		text_io.put_line("mroomobj: Z1 > Z2");
		raise program_error; 
	end if;
	if zc-zr>=z1 then 
		text_io.put_line("mroomobj: ZC-ZR >= Z1");
		raise program_error; 
	end if;
	if z2>=zc+zr then 
		text_io.put_line("mroomobj: Z2 >= ZC+ZR");
		raise program_error; 
	end if;

	rm.gap:=true;

	-- front Z+ ccw inward normal
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zp2; --LR
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zp2; --UR
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zp2; --UL
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zp2; --LL

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- lf(we) X dn = So = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=0;
	rm.txuv(j+2):=xm*dx;  rm.txuv(j+1):=ym*dy;
	rm.txuv(j+4):=xm*dx;  rm.txuv(j+3):=yp*dy;
	rm.txuv(j+6):=xp*dx;  rm.txuv(j+5):=yp*dy;
	rm.txuv(j+8):=xp*dx;  rm.txuv(j+7):=ym*dy;


	-- top Y+ ccw inward normal
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=yp;  rm.vert(k+ 3):=zp2; --NE
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm1; --SE
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm1; --SW
	rm.vert(k+10):=xp;  rm.vert(k+11):=yp;  rm.vert(k+12):=zp2; --NW

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- we X no = Dn = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=8;
	rm.txuv(j+1):=xm*dx;  rm.txuv(j+2):=zp2*dz;
	rm.txuv(j+3):=xm*dx;  rm.txuv(j+4):=zm1*dz;
	rm.txuv(j+5):=xp*dx;  rm.txuv(j+6):=zm1*dz;
	rm.txuv(j+7):=xp*dx;  rm.txuv(j+8):=zp2*dz;



	-- back Z- ccw inward normal
	rm.vert(k+ 1):=xp;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm1; --LR
	rm.vert(k+ 4):=xp;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm1; --UR
	rm.vert(k+ 7):=xm;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm1; --UL
	rm.vert(k+10):=xm;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm1; --LL

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- lf(ea) X dn = No = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=16;
	rm.txuv(j+2):=xp*dx;  rm.txuv(j+1):=ym*dy;
	rm.txuv(j+4):=xp*dx;  rm.txuv(j+3):=yp*dy;
	rm.txuv(j+6):=xm*dx;  rm.txuv(j+5):=yp*dy;
	rm.txuv(j+8):=xm*dx;  rm.txuv(j+7):=ym*dy;


	-- bottom A  Y- ccw inward normal
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm1; --SE
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=ym;  rm.vert(k+ 6):=zp1; --NE
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=ym;  rm.vert(k+ 9):=zp1; --NW
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm1; --SW

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- we X so = Up = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=24;
	rm.txuv(j+1):=xm*dx;  rm.txuv(j+2):=zm1*dz;
	rm.txuv(j+3):=xm*dx;  rm.txuv(j+4):=zp1*dz;
	rm.txuv(j+5):=xp*dx;  rm.txuv(j+6):=zp1*dz;
	rm.txuv(j+7):=xp*dx;  rm.txuv(j+8):=zm1*dz;



	-- right (east) X- ccw inward normal
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm1; --LR
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm1; --UR
	rm.vert(k+ 7):=xm;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zp2; --UL
	rm.vert(k+10):=xm;  rm.vert(k+11):=ym;  rm.vert(k+12):=zp2; --LL

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- lf(no) X dn = We = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=32;
	rm.txuv(j+1):=ym*dy;  rm.txuv(j+2):=zm1*dz;
	rm.txuv(j+3):=yp*dy;  rm.txuv(j+4):=zm1*dz;
	rm.txuv(j+5):=yp*dy;  rm.txuv(j+6):=zp2*dz;
	rm.txuv(j+7):=ym*dy;  rm.txuv(j+8):=zp2*dz;


	-- left (west) X+ ccw inward normal
	rm.vert(k+ 1):=xp;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zp2; --LR
	rm.vert(k+ 4):=xp;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zp2; --UR
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm1; --UL
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm1; --LL

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- lf(so) X dn = Ea = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=40;
	rm.txuv(j+1):=ym*dy;  rm.txuv(j+2):=zp2*dz;
	rm.txuv(j+3):=yp*dy;  rm.txuv(j+4):=zp2*dz;
	rm.txuv(j+5):=yp*dy;  rm.txuv(j+6):=zm1*dz;
	rm.txuv(j+7):=ym*dy;  rm.txuv(j+8):=zm1*dz;


	-- bottom B  Y- ccw inward normal
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm2; --SE
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=ym;  rm.vert(k+ 6):=zp2; --NE
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=ym;  rm.vert(k+ 9):=zp2; --NW
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm2; --SW

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- we X so = Up = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------


	k:=k+12;

	j:=48;
	rm.txuv(j+1):=xm*dx;  rm.txuv(j+2):=zm2*dz;
	rm.txuv(j+3):=xm*dx;  rm.txuv(j+4):=zp2*dz;
	rm.txuv(j+5):=xp*dx;  rm.txuv(j+6):=zp2*dz;
	rm.txuv(j+7):=xp*dx;  rm.txuv(j+8):=zm2*dz;



	-- element indices:
	for i in 0..6 loop
		jj:=glushort(i*4);
		j := i*6;
		rm.elem(j+1):=jj+0;
		rm.elem(j+2):=jj+1;
		rm.elem(j+3):=jj+2;
		rm.elem(j+4):=jj+2;
		rm.elem(j+5):=jj+3;
		rm.elem(j+6):=jj+0;
	end loop;


end setroomwithfloorgap;















procedure setrect( rm: in out room;  xc,yc,zc, xr,yr,zr : float ) is

xm,xp,ym,yp,zm,zp : float;

	k, j : integer := 0;
	jj : glushort;
	ax,ay,az,bx,by,bz,nx,ny,nz: float;

begin
	xm  := xc-xr;
	xp  := xc+xr;
	ym  := yc-yr;
	yp  := yc+yr;
	zm  := zc-zr;
	zp  := zc+zr;


	rm.gap:=false;

	-- front
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zp;
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zp;
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zp;
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zp;

-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(ax,ay,az, bx,by,bz, nx,ny,nz);
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------

	k:=k+12;

	j:=0;
	rm.txuv(j+1):=xm;  rm.txuv(j+2):=ym;
	rm.txuv(j+3):=xp;  rm.txuv(j+4):=ym;
	rm.txuv(j+5):=xp;  rm.txuv(j+6):=yp;
	rm.txuv(j+7):=xm;  rm.txuv(j+8):=yp;




	-- top
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=yp;  rm.vert(k+ 3):=zp;
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm;
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm;
	rm.vert(k+10):=xp;  rm.vert(k+11):=yp;  rm.vert(k+12):=zp;
	
-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(ax,ay,az, bx,by,bz, nx,ny,nz);
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------

	k:=k+12;

	j:=8;
	rm.txuv(j+1):=xm;  rm.txuv(j+2):=zm;
	rm.txuv(j+3):=xp;  rm.txuv(j+4):=zm;
	rm.txuv(j+5):=xp;  rm.txuv(j+6):=zp;
	rm.txuv(j+7):=xm;  rm.txuv(j+8):=zp;





	-- back
	rm.vert(k+ 1):=xp;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm;
	rm.vert(k+ 4):=xp;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm;
	rm.vert(k+ 7):=xm;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm;
	rm.vert(k+10):=xm;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm;
	
-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(ax,ay,az, bx,by,bz, nx,ny,nz);
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------

	k:=k+12;

	j:=16;
	rm.txuv(j+1):=xm;  rm.txuv(j+2):=ym;
	rm.txuv(j+3):=xp;  rm.txuv(j+4):=ym;
	rm.txuv(j+5):=xp;  rm.txuv(j+6):=yp;
	rm.txuv(j+7):=xm;  rm.txuv(j+8):=yp;




	-- bottom
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm;
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=ym;  rm.vert(k+ 6):=zp;
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=ym;  rm.vert(k+ 9):=zp;
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm;
	
-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(ax,ay,az, bx,by,bz, nx,ny,nz);
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------

	k:=k+12;

	j:=24;
	rm.txuv(j+1):=xm;  rm.txuv(j+2):=zm;
	rm.txuv(j+3):=xp;  rm.txuv(j+4):=zm;
	rm.txuv(j+5):=xp;  rm.txuv(j+6):=zp;
	rm.txuv(j+7):=xm;  rm.txuv(j+8):=zp;





	-- right
	rm.vert(k+ 1):=xm;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zm;
	rm.vert(k+ 4):=xm;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zm;
	rm.vert(k+ 7):=xm;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zp;
	rm.vert(k+10):=xm;  rm.vert(k+11):=ym;  rm.vert(k+12):=zp;
	
-------- begin insert ---------------------------------------
	ax:=rm.vert(k+1)-rm.vert(k+4);
	ay:=rm.vert(k+2)-rm.vert(k+5);
	az:=rm.vert(k+3)-rm.vert(k+6);
	bx:=rm.vert(k+7)-rm.vert(k+4);
	by:=rm.vert(k+8)-rm.vert(k+5);
	bz:=rm.vert(k+9)-rm.vert(k+6);
	cross(ax,ay,az, bx,by,bz, nx,ny,nz);
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rm.norm(k+ 1):=nx;  rm.norm(k+ 2):=ny;  rm.norm(k+ 3):=nz;
	rm.norm(k+ 4):=nx;  rm.norm(k+ 5):=ny;  rm.norm(k+ 6):=nz;
	rm.norm(k+ 7):=nx;  rm.norm(k+ 8):=ny;  rm.norm(k+ 9):=nz;
	rm.norm(k+10):=nx;  rm.norm(k+11):=ny;  rm.norm(k+12):=nz;
-------- end insert ---------------------------------------

	k:=k+12;

	j:=32;
	rm.txuv(j+1):=ym;  rm.txuv(j+2):=zm;
	rm.txuv(j+3):=yp;  rm.txuv(j+4):=zm;
	rm.txuv(j+5):=yp;  rm.txuv(j+6):=zp;
	rm.txuv(j+7):=ym;  rm.txuv(j+8):=zp;




	-- left
	rm.vert(k+ 1):=xp;  rm.vert(k+ 2):=ym;  rm.vert(k+ 3):=zp;
	rm.vert(k+ 4):=xp;  rm.vert(k+ 5):=yp;  rm.vert(k+ 6):=zp;
	rm.vert(k+ 7):=xp;  rm.vert(k+ 8):=yp;  rm.vert(k+ 9):=zm;
	rm.vert(k+10):=xp;  rm.vert(k+11):=ym;  rm.vert(k+12):=zm;

	j:=40;
	rm.txuv(j+1):=ym;  rm.txuv(j+2):=zm;
	rm.txuv(j+3):=yp;  rm.txuv(j+4):=zm;
	rm.txuv(j+5):=yp;  rm.txuv(j+6):=zp;
	rm.txuv(j+7):=ym;  rm.txuv(j+8):=zp;






	-- element indices:
	for i in 0..5 loop
		jj:=glushort(i*4);
		j := i*6;
		rm.elem(j+1):=jj+0;
		rm.elem(j+2):=jj+1;
		rm.elem(j+3):=jj+2;
		rm.elem(j+4):=jj+2;
		rm.elem(j+5):=jj+3;
		rm.elem(j+6):=jj+0;
	end loop;


end setrect;



--
-- note:  the shaders for these objects must have two 
-- input "layouts", as well as whatever uniforms are needed:
--
-- layout(location=0) in vec3 vertPosName
-- layout(location=1) in vec3 vertRgbName
--
-- ...where their actual names can be whatever is convenient
--
use gl;
use glext;
use glext.binding;
use gl.binding;

procedure draw( rm: room;  vertbuff, uvbuff, elembuff : gluint ) is
	nv, nt, ne : integer; -- no gap => (72,48,36);  gap=>(84,56,42)
begin

	if rm.gap then
		nv:=nvert;
		nt:=nuv;
		ne:=nelm;
	else
		nv:=72;
		nt:=48;
		ne:=36;
	end if;

	-- 0th attribute:  vertices
	glBindBuffer(gl_array_buffer, vertbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nv), rm.vert(1)'address, gl_static_draw);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0,3,gl_float,gl_false,0, system.null_address);

	-- 1st attribute:  texture UV
	glBindBuffer(gl_array_buffer, uvbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nt), rm.txuv(1)'address, gl_static_draw);
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1,2,gl_float,gl_false,0, system.null_address);

	-- element indices:
	glBindBuffer(gl_element_array_buffer, elembuff);
	glBufferData(gl_element_array_buffer, glsizeiptr(2*ne), rm.elem(1)'address, gl_static_draw);

	glEnable(gl_blend);
	glBlendFunc(gl_src_alpha, gl_one_minus_src_alpha);

	glDrawElements( gl_triangles, glint(nv), gl_unsigned_short, system.null_address );

	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);

end draw;



-- this expects Lighting-capable shaders
procedure ldraw( rm: room;  vertbuff, uvbuff, normbuff, elembuff : gluint ) is
	nv, nt, nn, ne : integer; -- no gap => (72,48,72,36);  gap=>(84,56,84,42)
begin

	if rm.gap then
		nv:=nvert;
		nt:=nuv;
		nn:=nnorm;
		ne:=nelm;
	else
		nv:=72;
		nt:=48;
		nn:=nv;
		ne:=36;
	end if;

	-- 0th attribute:  vertices
	glBindBuffer(gl_array_buffer, vertbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nv), rm.vert(1)'address, gl_static_draw);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0,3,gl_float,gl_false,0, system.null_address);

	-- 1st attribute:  texture UV
	glBindBuffer(gl_array_buffer, uvbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nt), rm.txuv(1)'address, gl_static_draw);
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1,2,gl_float,gl_false,0, system.null_address);

	-- 2nd attribute:  normals
	glBindBuffer(gl_array_buffer, normbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nn), rm.norm(1)'address, gl_static_draw);
	glEnableVertexAttribArray(2);
	glVertexAttribPointer(2,3,gl_float,gl_false,0, system.null_address);

	-- element indices:
	glBindBuffer(gl_element_array_buffer, elembuff);
	glBufferData(gl_element_array_buffer, glsizeiptr(2*ne), rm.elem(1)'address, gl_static_draw);

	glEnable(gl_blend);
	glBlendFunc(gl_src_alpha, gl_one_minus_src_alpha);

	glDrawElements( gl_triangles, glint(nv), gl_unsigned_short, system.null_address );

	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);
	glDisableVertexAttribArray(2);

end ldraw;





end mroomobj;

