#!/bin/bash -f
#
# ユーザーパスワードの有効期限設定用スクリプト
#
# Developed by Yasuhiro Fujitsuki 2010/3
#
# 返り値：
# 0 = 成功
# 1 = 失敗
# 9 = ユーザー作成を実行せず
# 99 = Usage 表示/オプション設定ミス
#

###################################
# 返り値定義 / 条件を変える場合、この部分を変更する
SUCCESS=0
FAILED=1
PASSED=9
USAGE=99
###################################
# 環境変数
# 実行ファイル
CHAGE=/usr/bin/chage
GREP=/bin/grep
WC=/usr/bin/wc 
WC_L="${WC} -l"
# 設定ファイル
ETC_PASSWD=/etc/passwd
###################################


# 引数確認
# 引数が指定数無い場合は usage 表示を行い、終了
if [ $# -lt 2 ]; then
  echo 引数指定がされていません。
  echo 
  echo usage \> $0 [User Name] [Expire Date] \<Inactive Date\>
  echo
  echo "[User Name]     ... ユーザーアカウント"
  echo "[Expire Date]   ... ファイル有効期限"
  echo "                    有効日を数値で設定"
  echo "                    30 = 30 日"
  echo "                     x = 最大値(99999日/システムデフォルト値と同等)"
  echo "<Inactive Date> ... パスワード期限切れからアカウントロックがかかるまでの日数"
  echo "                    省略 = デフォルト値"
  echo "                      10 = パスワード有効期限切れ、10日後にアカウントロックする"
  echo "                       x    = 有効にしない"
  echo
  exit $USAGE
fi

# 引数の読み込み
USER_NAME=$1
EXPIRE=$2
INACTIVE=$3

# ユーザーの存在確認
if [ `${GREP} ^${USER_NAME}: ${ETC_PASSWD} | ${WC_L} ` -eq 0 ]; then
  echo \#\#\# ユーザー ${USER_NAME} が存在しません。
  exit $PASSED
fi


# 実行オプション用変数
EXEC_OPTION=""

# ユーザーパスワードの有効期限確認

if [ $EXPIRE = "x" ]; then
  EXPIRE=99999
elif [ `echo $EXPIRE | $GREP "^[[:digit:]+]" | $WC_L` -ne 1 ]; then
  echo \#\#\# パスワード有効期限設定が不正です。\(設定値: $EXPIRE\)
  exit $FAILED
fi

echo
echo \#\#\# パスワード有効期間 : $EXPIRE

# 実行オプション設定
EXEC_OPTION="-M $EXPIRE"


# ユーザーのアカウントロック設定確認

if [ "$INACTIVE" = "x" ]; then
  INACTIVE=never
  # 実行オプション設定
  EXEC_OPTION="$EXEC_OPTION -I $INACTIVE"
  echo \#\#\# パスワード有効期間 : $INACTIVE
elif [ "$INACTIVE" = "" ]; then
  echo \#\#\# アカウントロックは定義されていません。
  echo \#\#\# アカウントロックはデフォルト値となります。
elif [ `echo $INACTIVE | $GREP "^[[:digit:]+]" | $WC_L` -ne 1 ]; then
  echo \#\#\# パスワード有効期限設定が不正です。\(設定値: $INACTIVE\)
  exit $FAILED
else 
  # 実行オプション設定
  EXEC_OPTION="$EXEC_OPTION -I $INACTIVE"
  echo \#\#\# パスワード有効期間 : $INACTIVE
fi


# ユーザーパスワードの有効期限・アカウントロック設定

# アカウントロックを行わない場合
echo \#\#\# パスワード有効期限を設定します。
echo \# $CHAGE $EXEC_OPTION $USER_NAME
$CHAGE $EXEC_OPTION $USER_NAME


# コマンド成功確認

RESULT=$?

if [ $RESULT -ne $SUCCESS ]; then
  echo \#\#\# パスワード有効期限設定に失敗しました。
  echo Error Code : $RESULT
  exit $FAILED
fi

echo 
echo \#\#\# パスワード有効期限情報
echo \# $CHAGE -l $CHAGE
$CHAGE -l $USER_NAME 

exit $SUCCESS

