/*
 * (C) Copyright 2003
 * Gerry Hamel, geh@ti.com, Texas Instruments
 *
 * Based on
 * linux/drivers/usb/device/bi/omap.c
 * TI OMAP1510 USB bus interface driver
 *
 * Author: MontaVista Software, Inc.
 *	   source@mvista.com
 *	   (C) Copyright 2002
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <asm/io.h>
#include <usbdevice.h>
#include <usb/mb9bfxxx_udc.h>
#include <usb/udc.h>

#include "ep0.h"


#define UDC_MAX_ENDPOINTS	     5 /* Number of endpoints on this UDC */

/* Some kind of debugging output... */
#if 0
#define UDCDBG(str)
#define UDCDBGA(fmt,args...)
#else  /* The bugs still exists... */
#define UDCDBG(str) serial_printf("[%s] %s:%d: " str "\n", __FILE__,__FUNCTION__,__LINE__)
#define UDCDBGA(fmt,args...) serial_printf("[%s] %s:%d: " fmt "\n", __FILE__,__FUNCTION__,__LINE__, ##args)
#endif

#if 0
#define UDCREG(name)
#define UDCREGL(name)
#else  /* The bugs still exists... */
#define UDCREG(name)	 serial_printf("%s():%d: %s[%08x]=%.4x\n",__FUNCTION__,__LINE__, (#name), name, inw(name))	/* For 16-bit regs */
#define UDCREGL(name)	 serial_printf("%s():%d: %s[%08x]=%.8x\n",__FUNCTION__,__LINE__, (#name), name, inl(name))	/* For 32-bit regs */
#endif


static struct urb *ep0_urb = NULL;

static struct usb_device_instance *udc_device;	/* Used in interrupt handler */
static u16 udc_devstat = 0;	/* UDC status (DEVSTAT) */
static u32 udc_interrupts = 0;
static const u32 episaddr[] = {USB_EP0IS, USB_EP1S, USB_EP2S, USB_EP3S, USB_EP4S, USB_EP5S};
static const u32 eposaddr[] = {USB_EP0OS, USB_EP1S, USB_EP2S, USB_EP3S, USB_EP4S, USB_EP5S};

static void udc_stall_ep (unsigned int ep_addr);
int fm3_vbus_detect(void);

static struct usb_endpoint_instance *mb9bfxxx_find_ep(int ep)
{
	int i;

	for (i = 0; i < udc_device->bus->max_endpoints; i++) {
		if (udc_device->bus->endpoint_array[i].endpoint_address == ep)
			return &udc_device->bus->endpoint_array[i];
	}
	return NULL;
}

static void mb9bfxxx_configure_endpoints(struct usb_device_instance *device)
{
	int ep;
	struct usb_bus_instance *bus;
	struct usb_endpoint_instance *endpoint;
	int ep_addr;
	int packet_size;
	int attributes;

	bus = device->bus;

	for (ep = 0; ep < bus->max_endpoints; ep++) {
		endpoint = bus->endpoint_array + ep;
		ep_addr = endpoint->endpoint_address;
		if ((ep_addr & USB_ENDPOINT_DIR_MASK) == USB_DIR_IN) {
			/* IN endpoint */
			packet_size = endpoint->tx_packetSize;
			attributes = endpoint->tx_attributes;
		} else {
			/* OUT endpoint */
			packet_size = endpoint->rcv_packetSize;
			attributes = endpoint->rcv_attributes;
		}

		if (!ep) {
			writew(0x0040, USB_EP0C);
		} else if ((ep_addr & USB_ENDPOINT_DIR_MASK) == USB_DIR_IN) {
			writew(0x8000 |attributes << 13| packet_size, USB_EP0C + ep * 4);
		} else {
			writew(0x9000 |attributes << 13| packet_size, USB_EP0C + ep * 4);
		}
	}
}

static void mb9bfxxx_write_noniso_tx_fifo(struct usb_endpoint_instance
					   *endpoint)
{
	struct urb *urb = endpoint->tx_urb;
	int ep = endpoint->endpoint_address & USB_ENDPOINT_NUMBER_MASK;

	if (urb) {
		unsigned int last, i;

		UDCDBGA ("urb->buffer %p, buffer_length %d, actual_length %d",
			 urb->buffer, urb->buffer_length, urb->actual_length);
		if ((last =
		     MIN (urb->actual_length - endpoint->sent,
			  endpoint->tx_packetSize))) {
			u8 *cp = urb->buffer + endpoint->sent;

			UDCDBGA ("endpoint->sent %d, tx_packetSize %d, last %d", endpoint->sent, endpoint->tx_packetSize, last);
			for (i = 0; i < last; i += 2) {
				writeb(*cp++, USB_EP0DTL + ep * 4);
				if ((last - i) > 1)
					writeb(*cp++, USB_EP0DTH + ep * 4);
			}
		}
		i = readw(USB_EP0C + ep * 4) & 0x1ff;
		i |= last;
		writew(i, USB_EP0C + ep * 4);
		endpoint->last = last;
	}
}

static int mb9bfxxx_read_noniso_rx_fifo(struct usb_endpoint_instance
					 *endpoint)
{
	struct urb *urb = endpoint->rcv_urb;
	int len = 0;
	int ep = endpoint->endpoint_address & USB_ENDPOINT_NUMBER_MASK;

	if (urb) {
		unsigned char *cp = urb->buffer + urb->actual_length;
		int i;

		len = readw(episaddr[ep]) & 0x1ff;
		for (i = 0; i < len; i += 2) {
			*cp++ = readb(USB_EP0DTL + ep * 4);
			if ((len - i) > 1)
				*cp++ = readb(USB_EP0DTH + ep * 4);
		}
	}
	return len;
}

/* udc_state_transition_up
 * udc_state_transition_down
 *
 * Helper functions to implement device state changes.	The device states and
 * the events that transition between them are:
 *
 *				STATE_ATTACHED
 *				||	/\
 *				\/	||
 *	DEVICE_HUB_CONFIGURED			DEVICE_HUB_RESET
 *				||	/\
 *				\/	||
 *				STATE_POWERED
 *				||	/\
 *				\/	||
 *	DEVICE_RESET				DEVICE_POWER_INTERRUPTION
 *				||	/\
 *				\/	||
 *				STATE_DEFAULT
 *				||	/\
 *				\/	||
 *	DEVICE_ADDRESS_ASSIGNED			DEVICE_RESET
 *				||	/\
 *				\/	||
 *				STATE_ADDRESSED
 *				||	/\
 *				\/	||
 *	DEVICE_CONFIGURED			DEVICE_DE_CONFIGURED
 *				||	/\
 *				\/	||
 *				STATE_CONFIGURED
 *
 * udc_state_transition_up transitions up (in the direction from STATE_ATTACHED
 * to STATE_CONFIGURED) from the specified initial state to the specified final
 * state, passing through each intermediate state on the way.  If the initial
 * state is at or above (i.e. nearer to STATE_CONFIGURED) the final state, then
 * no state transitions will take place.
 *
 * udc_state_transition_down transitions down (in the direction from
 * STATE_CONFIGURED to STATE_ATTACHED) from the specified initial state to the
 * specified final state, passing through each intermediate state on the way.
 * If the initial state is at or below (i.e. nearer to STATE_ATTACHED) the final
 * state, then no state transitions will take place.
 *
 * These functions must only be called with interrupts disabled.
 */
static void udc_state_transition_up(usb_device_state_t initial,
				    usb_device_state_t final)
{
	if (initial < final) {
		switch (initial) {
		case STATE_ATTACHED:
			usbd_device_event_irq (udc_device,
					       DEVICE_HUB_CONFIGURED, 0);
			if (final == STATE_POWERED)
				break;
		case STATE_POWERED:
			usbd_device_event_irq (udc_device, DEVICE_RESET, 0);
			if (final == STATE_DEFAULT)
				break;
		case STATE_DEFAULT:
			usbd_device_event_irq (udc_device,
					       DEVICE_ADDRESS_ASSIGNED, 0);
			if (final == STATE_ADDRESSED)
				break;
		case STATE_ADDRESSED:
			usbd_device_event_irq (udc_device, DEVICE_CONFIGURED,
					       0);
		case STATE_CONFIGURED:
			break;
		default:
			break;
		}
	}
}

static void udc_state_transition_down(usb_device_state_t initial,
				      usb_device_state_t final)
{
	if (initial > final) {
		switch (initial) {
		case STATE_CONFIGURED:
			usbd_device_event_irq (udc_device, DEVICE_DE_CONFIGURED, 0);
			if (final == STATE_ADDRESSED)
				break;
		case STATE_ADDRESSED:
			usbd_device_event_irq (udc_device, DEVICE_RESET, 0);
			if (final == STATE_DEFAULT)
				break;
		case STATE_DEFAULT:
			usbd_device_event_irq (udc_device, DEVICE_POWER_INTERRUPTION, 0);
			if (final == STATE_POWERED)
				break;
		case STATE_POWERED:
			usbd_device_event_irq (udc_device, DEVICE_HUB_RESET, 0);
		case STATE_ATTACHED:
			break;
		default:
			break;
		}
	}
}

/* Handle all device state changes.
 */
static void mb9bfxxx_udc_state_changed(void)
{
	u16 bits;
	u16 devstat = readb(USB_UDCS);

	devstat |= (fm3_vbus_detect() & 1) << 6;
	UDCDBGA ("state changed, devstat %x, old %x", devstat, udc_devstat);

	bits = devstat ^ udc_devstat;
	if (bits) {
		if (bits & USB_UDCS_ATT) {
			if (devstat & USB_UDCS_ATT) {
				UDCDBG ("device attached and powered");
				udc_state_transition_up (udc_device->device_state, STATE_POWERED);
			} else {
				UDCDBG ("device detached or unpowered");
				udc_state_transition_down (udc_device->device_state, STATE_ATTACHED);
			}
		}
		if (bits & USB_UDCS_BRST) {
			if (devstat & USB_UDCS_BRST) {
				UDCDBG ("device reset in progess");
				udc_state_transition_down (udc_device->device_state, STATE_POWERED);
			} else {
				UDCDBG ("device entering default state");
				udc_state_transition_up (udc_device->device_state, STATE_DEFAULT);
			}
		}
		if (bits & USB_UDCS_SUSP) {
			if (devstat & USB_UDCS_SUSP) {
				UDCDBG ("entering suspended state");
				usbd_device_event_irq (udc_device, DEVICE_BUS_INACTIVE, 0);
			}
		}
		if (bits & USB_UDCS_WKUP) {
			UDCDBGA ("remote wakeup %s", (devstat & USB_UDCS_WKUP)
				 ? "enabled" : "disabled");
		}
		if (bits & USB_UDCS_CONF) {
			if (devstat & USB_UDCS_CONF) {
				UDCDBG ("device configured");
				/* The ep0_recv_setup function generates the
				 * DEVICE_CONFIGURED event when a
				 * USB_REQ_SET_CONFIGURATION setup packet is
				 * received, so we should already be in the
				 * state STATE_CONFIGURED.
				 */
				udc_state_transition_up (udc_device->device_state, STATE_CONFIGURED);
			}
		}
	}

	/* Clear interrupt source */
	outb(0x00, USB_UDCS);

	/* Save current DEVSTAT */
	udc_devstat = devstat;
}

/* Handle SETUP USB interrupt.
 */
static void mb9bfxxx_udc_setup(struct usb_endpoint_instance *endpoint)
{
	const int setup_pktsize = 8;
	unsigned char *datap =
		(unsigned char *) &ep0_urb->device_request;
	int i;
	UDCDBG ("-> Entering device setup");

	/* Read control request data */
	for (i = 0; i < setup_pktsize; i += 2) {
		*datap++ = readb(USB_EP0DTL);
		*datap++ = readb(USB_EP0DTH);
	}
	datap -= setup_pktsize;
	
	UDCDBGA ("EP0 setup read [%x %x %x %x %x %x %x %x]",
		 *(datap + 0), *(datap + 1), *(datap + 2),
		 *(datap + 3), *(datap + 4), *(datap + 5),
		 *(datap + 6), *(datap + 7));
	
	/* Try to process setup packet */
	if (ep0_recv_setup (ep0_urb)) {
		/* Not a setup packet, stall next EP0 transaction */
		udc_stall_ep (0);
		UDCDBG ("can't parse setup packet, still waiting for setup");
		return;
	}

	/* Check direction */
	if ((ep0_urb->device_request.bmRequestType & USB_REQ_DIRECTION_MASK)
	    == USB_REQ_HOST2DEVICE) {
		UDCDBG ("control write on EP0");
		if (le16_to_cpu (ep0_urb->device_request.wLength)) {
			/* We don't support control write data stages.
			 * The only standard control write request with a data
			 * stage is SET_DESCRIPTOR, and ep0_recv_setup doesn't
			 * support that so we just stall those requests.  A
			 * function driver might support a non-standard
			 * write request with a data stage, but it isn't
			 * obvious what we would do with the data if we read it
			 * so we'll just stall it.  It seems like the API isn't
			 * quite right here.
			 */
			/* Stall this request */
			UDCDBG ("Stalling unsupported EP0 control write data "
				"stage.");
			udc_stall_ep (0);
		}
	} else {
		UDCDBG ("control read on EP0");
		/* The ep0_recv_setup function has already placed our response
		 * packet data in ep0_urb->buffer and the packet length in
		 * ep0_urb->actual_length.
		 */
		endpoint->tx_urb = ep0_urb;
		endpoint->sent = 0;
		/* Write packet data to the FIFO.  omap1510_write_noniso_tx_fifo
		 * will update endpoint->last with the number of bytes written
		 * to the FIFO.
		 */
		mb9bfxxx_write_noniso_tx_fifo(endpoint);
		writew(readw(USB_EP0OS) & ~(1 << 10), USB_EP0OS);
		readw(USB_EP0OS);
	}

	UDCDBG ("<- Leaving device setup");
}

/* Handle endpoint 0 RX interrupt
 * This routine implements TRM Figure 14-16.
 */
static void mb9bfxxx_udc_ep0_rx(void)
{
	UDCDBG ("RX on EP0");
	/* select EP0 rx FIFO */
	if ((ep0_urb->device_request.bmRequestType
	     & USB_REQ_DIRECTION_MASK) == USB_REQ_HOST2DEVICE) {
		/* This rx interrupt must be for a control write data
		 * stage packet.
		 *
		 * We don't support control write data stages.
		 * We should never end up here.
		 */
		
		writew(readw(USB_EP0IS) & ~0x0400, USB_EP0IS);
		UDCDBG ("Stalling unexpected EP0 control write "
			"data stage packet");
		udc_stall_ep (0);
	} else {
		/* This rx interrupt must be for a control read status
		 * stage packet.
		 */
		UDCDBG ("ACK on EP0 control read status stage packet");
	}
}

/* Handle endpoint 0 TX interrupt
 */
static void mb9bfxxx_udc_ep0_tx(void)
{
	struct usb_device_request *request = &ep0_urb->device_request;
	struct usb_endpoint_instance *endpoint =
		mb9bfxxx_find_ep(0);

	UDCDBG ("TX on EP0");
	/* This tx interrupt must be for a control read data
	 * stage packet.
	 */
	int wLength = le16_to_cpu (request->wLength);
	
	/* Update our count of bytes sent so far in this
	 * transfer.
	 */
	endpoint->sent += endpoint->last;

	/* We are finished with this transfer if we have sent
	 * all of the bytes in our tx urb (urb->actual_length)
	 * unless we need a zero-length terminating packet.  We
	 * need a zero-length terminating packet if we returned
	 * fewer bytes than were requested (wLength) by the host,
	 * and the number of bytes we returned is an exact
	 * multiple of the packet size endpoint->tx_packetSize.
	 */
	if ((endpoint->sent == ep0_urb->actual_length)
	    && ((ep0_urb->actual_length == wLength)
		|| (endpoint->last !=
		    endpoint->tx_packetSize))) {
		/* Done with control read data stage. */
		UDCDBG ("control read data stage complete");
		writew(0x8400, USB_EP0IS);
		udelay(1);
		writew(0x0000, USB_EP0IS);
	} else {
		/* We still have another packet of data to send
		 * in this control read data stage or else we
		 * need a zero-length terminating packet.
		 */
		UDCDBG ("ACK control read data stage packet");
		writew(0x8400, USB_EP0IS);
		mb9bfxxx_write_noniso_tx_fifo(endpoint);
		writew(0x0000, USB_EP0IS);
	}
}

/* Handle RX transaction on non-ISO endpoint.
 * This function implements TRM Figure 14-27.
 * The ep argument is a physical endpoint number for a non-ISO OUT endpoint
 * in the range 1 to 15.
 */
static void mb9bfxxx_udc_epn_rx(int ep)
{
	int nbytes;
	struct usb_endpoint_instance *endpoint =
		mb9bfxxx_find_ep (ep);
	
	nbytes = mb9bfxxx_read_noniso_rx_fifo (endpoint);
	usbd_rcv_complete (endpoint, nbytes, 0);
	
	/* enable rx FIFO to prepare for next packet */
	writew(readw(episaddr[ep]) & ~(1 << 10), episaddr[ep]);
	readw(episaddr[ep]);
}

/* Handle TX transaction on non-ISO endpoint.
 * This function implements TRM Figure 14-29.
 * The ep argument is a physical endpoint number for a non-ISO IN endpoint
 * in the range 16 to 30.
 */
static void mb9bfxxx_udc_epn_tx(int ep)
{
	struct usb_endpoint_instance *endpoint =
		mb9bfxxx_find_ep (ep);

	/* We need to transmit a terminating zero-length packet now if
	 * we have sent all of the data in this URB and the transfer
	 * size was an exact multiple of the packet size.
	 */
	if (endpoint->tx_urb
	    && (endpoint->last == endpoint->tx_packetSize)
	    && (endpoint->tx_urb->actual_length - endpoint->sent -
		endpoint->last == 0)) {
		/* Prepare to transmit a zero-length packet. */
		endpoint->sent += endpoint->last;
		/* write 0 bytes of data to FIFO */
		mb9bfxxx_write_noniso_tx_fifo (endpoint);
	} else if (endpoint->tx_urb
		   && endpoint->tx_urb->actual_length) {
		/* Check to see if we have more data ready to transmit
		 * now.
		 */
		if (endpoint->tx_urb
		    && endpoint->tx_urb->actual_length) {
			/* write data to FIFO */
			mb9bfxxx_write_noniso_tx_fifo (endpoint);
		}
	}
	/* enable tx FIFO to start transmission */
	writew(readw(eposaddr[ep]) & ~(1 << 10), eposaddr[ep]);
	readw(eposaddr[ep]);
}


/*
-------------------------------------------------------------------------------
*/

/* Handle general USB interrupts and dispatch according to type.
 */
static void mb9bfxxx_udc_irq(void)
{
	u8 irq_src = readb(USB_UDCS);

	if (!(irq_src & ~USB_UDCS_SOF))	/* ignore SOF interrupts ) */
		return;

	UDCDBGA ("< IRQ #%d start >- %x", udc_interrupts, irq_src);
	/*serial_printf("< IRQ #%d start >- %x\n", udc_interrupts, irq_src); */

	if (irq_src & USB_UDCS_SETP) {
		/* Device setup */
		mb9bfxxx_udc_setup (udc_device->bus->endpoint_array + 0);
	} else
		mb9bfxxx_udc_state_changed();
	UDCDBGA ("< IRQ #%d end >", udc_interrupts);
	udc_interrupts++;
}

/* EP1-5 Tx/Rx Interrupt */
static void mb9bfxxx_udc_ep_irq(int epnum)
{
	UDCDBGA ("non-ISO IRQ, IRQ_SRC %d - %x",
		 epnum, readw(USB_EP1S + (epnum - 1) * 4));

	if (readw(USB_EP1C + (epnum - 1) * 4) & 0x1000) { /* Endpoint N OUT transaction */
		UDCDBGA ("TX on ep %x", epnum);
		mb9bfxxx_udc_epn_tx(epnum);
	} else {
		UDCDBGA ("RX on ep %x", epnum);
		mb9bfxxx_udc_epn_rx(epnum);
	}
	writew(readw(eposaddr[epnum]) & ~(1 << 10), eposaddr[epnum]);
	readw(eposaddr[epnum]);
}

/*
-------------------------------------------------------------------------------
*/


/*
 * Start of public functions.
 */

/* Called to start packet transmission. */
int udc_endpoint_write(struct usb_endpoint_instance *endpoint)
{
	unsigned short epnum =
		endpoint->endpoint_address & USB_ENDPOINT_NUMBER_MASK;

	UDCDBGA ("Starting transmit on ep %x", epnum);

	if (endpoint->tx_urb) {
		/* write data to FIFO */
		mb9bfxxx_write_noniso_tx_fifo (endpoint);
		/* enable tx FIFO to start transmission */
		writew(readw(eposaddr[epnum]) & ~(1 << 10), eposaddr[epnum]);
		readw(eposaddr[epnum]);
	}

	return 0;
}

/* Start to initialize h/w stuff */
int udc_init(void)
{
	udc_device = NULL;

	UDCDBG ("starting");
	writeb(readb(UECLK_USBEN + CONFIG_FM3_UDC_CH * 4) | 0x01,
	       UECLK_USBEN + CONFIG_FM3_UDC_CH * 4);
	writew(readw(USB_UDCC) | 0x00a0, USB_UDCC);
	writew(readw(USB_UDCC) & ~0x0001, USB_UDCC);
	
	return 0;
}

/* Stall endpoint */
static void udc_stall_ep(unsigned int ep_addr)
{
	/*int ep_addr = PHYS_EP_TO_EP_ADDR(ep); */
	int ep_num = ep_addr & USB_ENDPOINT_NUMBER_MASK;

	UDCDBGA ("stall ep_addr %d", ep_addr);
	writew(readw(USB_EP0C + ep_num) | 0x0200, USB_EP0C + ep_num);
}

/*
 * udc_setup_ep - setup endpoint
 *
 * Associate a physical endpoint with endpoint_instance
 */
void udc_setup_ep(struct usb_device_instance *device,
		  unsigned int ep, struct usb_endpoint_instance *endpoint)
{
	__u16 control = 0;
	UDCDBGA ("setting up endpoint addr %x", endpoint->endpoint_address);

	if (ep == 0) {
	}
	if (endpoint && (ep < UDC_MAX_ENDPOINTS)) {
		int ep_addr = endpoint->endpoint_address;

		if (!ep_addr) {
			writew(0x40, USB_EP0C);
		} else {
			if ((ep_addr & USB_ENDPOINT_DIR_MASK) == USB_DIR_IN) {
				control |= 0x9000;
				control |= (endpoint->rcv_attributes & 3) << 13;
			} else {
				control |= 0x8000;
				control |= (endpoint->tx_attributes & 3) << 13;
			}
			control |= (ep_addr == 1)?0x100:0x40;
			writew(control, USB_EP1C + (ep_addr - 1) * 4);

		}
	}
}

/* Turn on the USB connection by enabling the pullup resistor */
void udc_connect(void)
{
	int i;

	UDCDBG ("connect, enable Pullup");
	writew(readw(USB_UDCC) & ~0x0080, USB_UDCC);
	writew(readw(USB_EP0OS) & ~0x8000, USB_EP0OS);
	writew(readw(USB_EP0IS) & ~0x8000, USB_EP0IS);
	for (i = 0; i < 4; i++)
		writew(readw(USB_EP1S + i * 4) & ~0x8000, 
		       USB_EP1S + i * 4);

	writew(readw(USB_UDCC) & ~0x0020, USB_UDCC);
}

/* Turn off the USB connection by disabling the pullup resistor */
void udc_disconnect(void)
{
	UDCDBG ("disconnect, disable Pullup");
	writew(readw(USB_UDCC) | 0x0020, USB_UDCC);
}

/* Switch on the UDC */
void udc_enable (struct usb_device_instance *device)
{
	UDCDBGA ("enable device %p, status %d", device, device->status);

	/* initialize driver state variables */
	udc_devstat = 0;

	/* Save the device structure pointer */
	udc_device = device;

	/* Setup ep0 urb */
	if (!ep0_urb) {
		ep0_urb =
			usbd_alloc_urb (udc_device,
					udc_device->bus->endpoint_array);
	} else {
		serial_printf ("udc_enable: ep0_urb already allocated %p\n",
			       ep0_urb);
	}
	mb9bfxxx_configure_endpoints(device);
}

/* Switch off the UDC */
void udc_disable(void)
{
	UDCDBG("disable UDC");
	writew(0x00a0, USB_UDCC);

	/* Free ep0 URB */
	if (ep0_urb)
		usbd_dealloc_urb(ep0_urb);
}

/**
 * udc_startup - allow udc code to do any additional startup
 */
void udc_startup_events(struct usb_device_instance *device)
{
	/* The DEVICE_INIT event puts the USB device in the state STATE_INIT. */
	usbd_device_event_irq(device, DEVICE_INIT, 0);

	/* The DEVICE_CREATE event puts the USB device in the state
	 * STATE_ATTACHED.
	 */
	usbd_device_event_irq(device, DEVICE_CREATE, 0);

	/* Some USB controller driver implementations signal
	 * DEVICE_HUB_CONFIGURED and DEVICE_RESET events here.
	 * DEVICE_HUB_CONFIGURED causes a transition to the state STATE_POWERED,
	 * and DEVICE_RESET causes a transition to the state STATE_DEFAULT.
	 */

	udc_enable (device);
}

/**
 * udc_irq - do pseudo interrupts
 */
void udc_irq(void)
{
	/* Loop while we have interrupts.
	 * If we don't do this, the input chain
	 * polling delay is likely to miss
	 * host requests.
	 */
	int handle_irq;
	int i;

	do {
		handle_irq = 0;
		if (readb(USB_UDCS)) {
			handle_irq = 1;
			mb9bfxxx_udc_irq ();
		}
		if (readw(USB_EP0IS) & 0x0400) {
			handle_irq = 1;
			mb9bfxxx_udc_ep0_rx();
		}
		if (readw(USB_EP0OS) & 0x0400) {
			handle_irq = 1;
			mb9bfxxx_udc_ep0_tx();
		}
		for (i = 0; i < 5; i++) {
			if (readw(USB_EP1S + i * 4) & 0x0400) {
				handle_irq = 1;
				mb9bfxxx_udc_ep_irq(i + 1);
			}
		}
	} while(handle_irq);
}

/* Flow control */
void udc_set_nak(int epid)
{
}

void udc_unset_nak(int epid)
{
}
