/*
 * Debug Monitor Framework
 * Author: Yasushi Tanaka
 *
 * [ YMODEM ]
 */

#include "header.h"
#include "monitor.h"
#include "logwin.h"
#include "comm.h"
#include "ymodem.h"

/*
 * j^
 */

/* j^Jnʒu(X) */
#define YMODEM_MONITOR_X                0

/* j^Jnʒu(Y) */
#define YMODEM_MONITOR_Y                1

/*
 * YMODEM萔
 */

/* SOH (t[̊Jn)(128oCg) */
#define YMODEM_SOH                      0x01

/* STX (eLXg̊Jn)(1024oCg) */
#define YMODEM_STX                      0x02

/* EOT (t@C]̏I) */
#define YMODEM_EOT                      0x04

/* ACK (mI) */
#define YMODEM_ACK                      0x06

/* NAK (ےI) */
#define YMODEM_NAK                      0x15

/* CAN (LZv) */
#define YMODEM_CAN                      0x18

/* C (MJnv) */
#define YMODEM_C                        0x43

/*
 * ^CAEgуgC
 */

/* M^CAEg */
#define YMODEM_SEND_TIMEOUT             1000

/* MgC */
#define YMODEM_SEND_RETRY               40

/* M^CAEg */
#define YMODEM_RECV_TIMEOUT             500

/* MgC */
#define YMODEM_RECV_RETRY               80

/*
 * ԑJ
 */

/*  */
#define YMODEM_STATE_NONE               0

/*
 * Mn
 */

/* M'C'M҂(1) */
#define YMODEM_STATE_WAITC1             1

/* t@C񑗐M */
#define YMODEM_STATE_SENDINFO           2

/* t@CACK҂ */
#define YMODEM_STATE_INFOACK            3

/* M'C'M҂(2) */
#define YMODEM_STATE_WAITC2             4

/* f[^ubNM */
#define YMODEM_STATE_SENDDATA           5

/* f[^ubNACK҂ */
#define YMODEM_STATE_DATAACK            6

/* EOTM */
#define YMODEM_STATE_SENDEOT            7

/* EOTɑ΂ACK҂ */
#define YMODEM_STATE_EOTACK             8

/*
 * Mn
 */

/* 'C'M(1) */
#define YMODEM_STATE_SENDC1             9

/* SOHM҂ */
#define YMODEM_STATE_WAITSOH            10

/*
 * YMODEMIuWFNg
 */
typedef struct _YMDOEM_OBJECT
{
    /*
     * OJ
     */

    /* ʐM|[g̃CfbNX */
    UINT comm_index;

    /* M */
    BOOL sending;

    /* M */
    BOOL receiving;

    /* Xe[^X */
    UINT status;

    /* t@Cl[(t@CTCY̒ǉzA0x60Ƃ) */
    char file_name[0x60];

    /* t@CTCY */
    UINT file_size;

    /* t@Cobt@ */
    BYTE file_buf[0x10000];

    /*
     * 
     */

    /* ԑJ */
    UINT state;

    /* Jn */
    DWORD start_time;

    /* gCJE^ */
    UINT retry;

    /* ubNԍ */
    UINT block_no;

    /* ubNobt@ */
    BYTE block_buf[0x400 + 5];

    /* ubNItZbg */
    UINT block_offset;

    /* obt@ItZbg */
    UINT buf_offset;

    /* CANJE^ */
    UINT cancel;

    /*
     * Ql(j^\p)
     */

    /* ACKM */
    UINT ack_cnt;

    /* NAKM */
    UINT nak_cnt;

    /* CANM */
    UINT can_cnt;

    /* ^CAEg */
    UINT timeout_cnt;

    /* gC */
    UINT retry_cnt;

    /* CRC */
    WORD crc;
} YMODEM_OBJECT;

/*
 * staticϐ
 */

/* IuWFNg */
YMODEM_OBJECT g_ymodem_object;

/*
 * static֐
 */

/* IuWFNgԑJڗv */
static void ymodem_object_trans(YMODEM_OBJECT *object, UINT state);

/*
 * YMODEM
 * CRCe[u
 */
static WORD ymodem_crc_table[0x100] = {
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7,
    0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF,
    0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6,
    0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE,
    0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485,
    0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D,
    0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4,
    0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC,

    0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823,
    0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B,
    0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12,
    0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A,
    0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41,
    0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49,
    0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70,
    0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78,

    0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F,
    0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067,
    0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E,
    0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256,
    0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D,
    0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
    0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C,
    0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634,

    0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB,
    0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3,
    0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A,
    0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92,
    0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9,
    0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1,
    0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8,
    0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0,
};

/*
 * YMODEM
 * CRCZo
 */
static WORD ymodem_crc_calc(YMODEM_OBJECT *object)
{
    UINT size;
    UINT loop;
    WORD crc;

    assert(NULL != object);

    /* SOHSTXɂāAf[^̃TCY𕪂 */
    if (YMODEM_SOH == object->block_buf[0])
    {
        /* 128oCg */
        size = 0x80;
    }
    else
    {
        /* 1024oCg */
        size = 0x400;
    }

    /* CRCl */
    crc = 0x0000;

    /* [v */
    for (loop = 0; loop < size; loop++)
    {
        /* e[ug */
        crc = (WORD)(crc << 8) ^ ymodem_crc_table[(crc >> 8) ^ object->block_buf[3 + loop]];
    }

    /* CRCԂ */
    return crc;
}

/*
 * YMODEM
 * IuWFNg
 */
static void ymodem_object_init(YMODEM_OBJECT *object, UINT comm_index)
{
    assert(NULL != object);

    /* IuWFNg */
    memset(object, 0, sizeof(YMODEM_OBJECT));

    /* ʐM|[gԍݒ */
    object->comm_index = comm_index;
}

/*
 * YMODEM
 * G[()
 */
static void ymodem_error_success(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /*  */
    object->status = YMODEM_SUCCESS;

    /* Ԃɖ߂ */
    object->state = YMODEM_STATE_NONE;

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->sending = FALSE;

        /* O */
        logwin_printf("t@CM");
    }

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->receiving = FALSE;

        /* O */
        logwin_printf("t@CM");
    }
}

/*
 * YMODEM
 * G[(|[gN[Yꂽ)
 */
static void ymodem_error_closed(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* N[Yꂽ */
    object->status = YMODEM_CLOSED;

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->sending = FALSE;

        /* O */
        logwin_printf("t@CMs(COM|[gN[Yꂽ)");
    }

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->receiving = FALSE;

        /* O */
        logwin_printf("t@CMs(COM|[gN[Yꂽ)");
    }
}

/*
 * YMODEM
 * G[(^CAEg)
 */
static void ymodem_error_timeout(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* ^CAEg */
    object->status = YMODEM_TIMEOUT;

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->sending = FALSE;

        /* O */
        logwin_printf("t@CMs(^CAEg)");
    }

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->receiving = FALSE;

        /* O */
        logwin_printf("t@CMs(^CAEg)");
    }
}

/*
 * YMODEM
 * G[(gC)
 */
static void ymodem_error_retry(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* gC */
    object->status = YMODEM_RETRY;

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->sending = FALSE;

        /* O */
        logwin_printf("t@CMs(gC񐔏)");
    }

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->receiving = FALSE;

        /* O */
        logwin_printf("t@CMs(gC񐔏)");
    }
}

/*
* YMODEM
* G[(LZ)
*/
static void ymodem_error_cancel(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* LZ */
    object->status = YMODEM_CANCEL;

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->sending = FALSE;

        /* O */
        logwin_printf("t@CMs(LZvM)");
    }

    /* M */
    if (FALSE != object->sending)
    {
        /* MłȂ */
        object->receiving = FALSE;

        /* O */
        logwin_printf("t@CMs(LZvM)");
    }
}

/*
 * YMODEM
 * Mobt@̓ǂݎ̂
 */
static void ymodem_discard_recv(YMODEM_OBJECT *object)
{
    UINT bytes;
    UINT result;
    BYTE buf;

    assert(NULL != object);

    /* MoCg̎擾 */
    bytes = comm_get_bytes(object->comm_index);

    /* bytes0ȏł胋[v */
    while (0 < bytes)
    {
        /* 1oCgȏ̎MoCĝŁA1oCgǂݎ */
        result = comm_recv(object->comm_index, &buf, sizeof(buf));
        if (0 == result)
        {
            /* ǂݎɎs */
            ymodem_error_closed(object);
            break;
        }

        /* MoCg̎擾 */
        bytes = comm_get_bytes(object->comm_index);
    }
}

/*
 * YMODEM
 * ^CAEg`FbN
 */
static BOOL ymodem_check_timeout(YMODEM_OBJECT *object, DWORD timeout)
{
    DWORD diff;

    assert(NULL != object);

    /* Ԃ𓾂 */
    diff = (DWORD)(timeGetTime() - object->start_time);
    /* Ԃp[^ȏł΃^CAEg */
    if (diff >= timeout)
    {
        /* JnԂ𑝂₷ */
        object->start_time += timeout;

        /* ^CAEg++ */
        object->timeout_cnt++;

        /* ^CAEg */
        return TRUE;
    }

    /* ^CAEgł͂Ȃ */
    return FALSE;
}

/*
 * YMODEM
 * gC`FbN
 */
static BOOL ymodem_check_retry(YMODEM_OBJECT *object, UINT retry_max)
{
    assert(NULL != object);

    /* gC++ */
    object->retry_cnt++;

    /* gCJE^CNg */
    object->retry++;

    /* gC𒴂Ă΃gCI[o[ */
    if (object->retry > retry_max)
    {
        /* gCI[o[ */
        return TRUE;
    }

    /* gC𒴂ĂȂ */
    return FALSE;
}

/*
 * YMODEM
 * ItZbg`FbN
 */
static BOOL ymodem_check_offset(YMODEM_OBJECT *object)
{
    UINT size;

    assert(NULL != object);

    /* SOHSTXɂāAf[^̃TCY𕪂 */
    if (YMODEM_SOH == object->block_buf[0])
    {
        /* 128oCgubN */
        size = 0x80 + 5;
    }
    else
    {
        /* 1024oCg */
        size = 0x400 + 5;
    }

    /* block_offsetsizeɓBĂTRUEԂ */
    assert(object->block_offset <= size);
    if (object->block_offset == size)
    {
        /* ubNItZbgI[ɓB */
        return TRUE;
    }

    /* ubNItZbg͏I[ɓBĂȂ */
    return FALSE;
}

/*
 * YMODEM
 * LZ`FbN
 */
static BOOL ymodem_check_cancel(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* LZJE^CNg */
    object->cancel++;

    /* 2AŃLZ󂯎LZ */
    if (2 <= object->cancel)
    {
        /* LZ */
        return TRUE;
    }

    /* LZłȂ */
    return FALSE;
}

/*
* YMODEM
* M̎MLN^҂
*/
static BYTE ymodem_wait_send(YMODEM_OBJECT *object)
{
    BOOL result;
    UINT len;
    UINT error;
    BYTE buf;

    /* ^CAEg`FbN */
    result = ymodem_check_timeout(object, YMODEM_SEND_TIMEOUT);
    if (FALSE != result)
    {
        /* ^CAEĝŁAgC`FbN */
        result = ymodem_check_retry(object, YMODEM_SEND_RETRY);
        if (FALSE != result)
        {
            /* ^CAEgG[ */
            ymodem_error_timeout(object);
            return 0;
        }
    }

    /* Mobt@̃f[^TCY𓾂 */
    len = comm_get_bytes(object->comm_index);
    while (len > 0)
    {
        /* 1oCgǂݍ */
        error = comm_recv(object->comm_index, &buf, sizeof(buf));
        if (0 == error)
        {
            /* |[gN[Yꂽ */
            ymodem_error_closed(object);
            return 0;
        }

        /* CANȊOłCANJE^NA */
        if (YMODEM_CAN != buf)
        {
            object->cancel = 0;
        }

        switch (buf)
        {
        /* C */
        case YMODEM_C:
            return YMODEM_C;

        /* ACK */
        case YMODEM_ACK:
            object->ack_cnt++;
            return YMODEM_ACK;

        /* NAK*/
        case YMODEM_NAK:
            object->nak_cnt++;
            return YMODEM_NAK;

        /* CAN */
        case YMODEM_CAN:
            object->can_cnt++;
            /* LZ`FbN */
            result = ymodem_check_cancel(object);
            if (FALSE != result)
            {
                /* LZG[ */
                ymodem_error_cancel(object);
                return 0;
            }

        /* ͓̑ǂݎ̂ */
        default:
            break;
        }
    }

    /* LȎMf[^Ȃ */
    return 0;
}

/*
 * YMODEM
 * M̎MLN^҂
 */
static BYTE ymodem_wait_recv(YMODEM_OBJECT *object)
{
    BOOL result;
    UINT len;
    UINT error;
    BYTE buf;

    /* ^CAEg`FbN */
    result = ymodem_check_timeout(object, YMODEM_RECV_TIMEOUT);
    if (FALSE != result)
    {
        /* ^CAEĝŁAgC`FbN */
        result = ymodem_check_retry(object, YMODEM_RECV_RETRY);
        if (FALSE != result)
        {
            /* ^CAEgG[ */
            ymodem_error_timeout(object);
            return 0;
        }
    }

    /* Mobt@̃f[^TCY𓾂 */
    len = comm_get_bytes(object->comm_index);
    while (len > 0)
    {
        /* 1oCgǂݍ */
        error = comm_recv(object->comm_index, &buf, sizeof(buf));
        if (0 == error)
        {
            /* |[gN[Yꂽ */
            ymodem_error_closed(object);
            return 0;
        }

        /* CANȊOłCANJE^NA */
        if (YMODEM_CAN != buf)
        {
            object->cancel = 0;
        }

        switch (buf)
        {
        /* SOH */
        case YMODEM_SOH:
            return YMODEM_SOH;

        /* STX */
        case YMODEM_STX:
            return YMODEM_STX;

        /* EOT */
        case YMODEM_EOT:
            return YMODEM_EOT;

        /* CAN */
        case YMODEM_CAN:
            object->can_cnt++;
            /* LZ`FbN */
            result = ymodem_check_cancel(object);
            if (FALSE != result)
            {
                /* LZG[ */
                ymodem_error_cancel(object);
                return 0;
            }

        /* ͓̑ǂݎ̂ */
        default:
            break;
        }
    }

    /* LȎMf[^Ȃ */
    return 0;
}

/*
 * YMODEM
 * ubNMJn(128oCgŒ)
 */
static void ymodem_block_sendstart(YMODEM_OBJECT *object)
{
    WORD crc;

    assert(NULL != object);

    /* SOH */
    object->block_buf[0] = YMODEM_SOH;

    /* ubNԍ */
    object->block_buf[1] = (BYTE)object->block_no;

    /* ubNԍ̕␔ */
    object->block_buf[2] = (BYTE)(0xff - object->block_buf[1]);

    /* CRCZo */
    crc = ymodem_crc_calc(object);

    /* CRCL */
    object->crc = crc;

    /* CRC */
    object->block_buf[3 + 0x80 + 0] = (BYTE)(crc >> 8);

    /* CRC */
    object->block_buf[3 + 0x80 + 1] = (BYTE)crc;

    /* ubNItZbg */
    object->block_offset = 0;
}

/*
 * YMODEM
 * ubNM
 */
static void ymodem_block_senddo(YMODEM_OBJECT *object)
{
    UINT size;
    UINT result;

    assert(object != NULL);

    /* SOHSTXɂāAf[^̃TCY𕪂 */
    if (YMODEM_SOH == object->block_buf[0])
    {
        /* 128oCgubN */
        size = 0x80 + 5;
    }
    else
    {
        /* 1024oCg */
        size = 0x400 + 5;
    }

    /* size - ubNItZbgM */
    assert((size - object->block_offset) > 0);
    result = comm_send(object->comm_index, &object->block_buf[object->block_offset], size - object->block_offset);
    if (0 == result)
    {
        /* ݂Ɏs */
        ymodem_error_closed(object);
    }
    else
    {
        /* ꕔ߂ */
        object->block_offset += result;
        assert(object->block_offset <= size);
    }
}

/*
 * YMODEM
 * M'C'M҂(1)
 * enter
 */
static void ymodem_waitc1_enter(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* gC񐔂 */
    object->retry = 0;

    /* JnԂ */
    object->start_time = timeGetTime();

    /* CANM񐔂 */
    object->cancel = 0;

    /* Mobt@̓ǂݎ̂ */
    ymodem_discard_recv(object);
}

/*
 * YMODEM
 * M'C'M҂(1)
 * do
 */
static void ymodem_waitc1_do(YMODEM_OBJECT *object)
{
    BYTE buf;

    /* M̎MLN^҂ */
    buf = ymodem_wait_send(object);

    /* 'C'łΏԑJ */
    if (YMODEM_C == buf)
    {
        ymodem_object_trans(object, YMODEM_STATE_SENDINFO);
    }
}

/*
 * YMODEM
 * M'C'M҂(1)
 * exit
 */
static void ymodem_waitc1_exit(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* gC񐔂 */
    object->retry = 0;
}

/*
 * YMODEM
 * t@C񑗐M
 * enter
 */
static void ymodem_sendinfo_enter(YMODEM_OBJECT *object)
{
    char filesize[0x80];
    size_t len;

    assert(NULL != object);

    /* ubNԍ0ɏ */
    object->block_no = 0x00;

    /* ubNobt@ */
    memset(object->block_buf, 0, sizeof(object->block_buf));

    /* t@Cl[ubNobt@+3ɃRs[ */
    strcpy_s((char *)&object->block_buf[3], _countof(object->block_buf) - 3, object->file_name);

    /* t@Cl[̒𓾂 */
    len = strlen(object->file_name);

    /* t@CTCYtH[}bg */
    sprintf_s(filesize, _countof(filesize), "%u", object->file_size);

    /* t@Cl[̌A0x00󂯂Ēǉ */
    strcpy_s((char *)&object->block_buf[3 + len + 1], _countof(object->block_buf) - 3, filesize);

    /* ubNMJn */
    ymodem_block_sendstart(object);
}

/*
 * YMODEM
 * t@C񑗐M
 * do
 */
static void ymodem_sendinfo_do(YMODEM_OBJECT *object)
{
    BOOL result;

    /* ubNM */
    ymodem_block_senddo(object);

    /* ItZbg[܂œBAԑJ */
    result = ymodem_check_offset(object);
    if (FALSE != result)
    {
        /* t@CACK҂ */
        ymodem_object_trans(object, YMODEM_STATE_INFOACK);
    }
}

/*
 * YMODEM
 * t@C񑗐M
 * exit
 */
static void ymodem_sendinfo_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * t@CACK҂
 * enter
 */
static void ymodem_infoack_enter(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* JnԂ */
    object->start_time = timeGetTime();
}

/*
 * YMODEM
 * t@CACK҂
 * do
 */
static void ymodem_infoack_do(YMODEM_OBJECT *object)
{
    BYTE buf;
    BOOL result;

    /* M̎MLN^҂ */
    buf = ymodem_wait_send(object);

    switch (buf)
    {
    /* ACK͏ԑJ */
    case YMODEM_ACK:
        ymodem_object_trans(object, YMODEM_STATE_WAITC2);
        break;

    /* NAK̓gC`FbN */
    case YMODEM_NAK:
        result = ymodem_check_retry(object, YMODEM_SEND_RETRY);
        if (FALSE != result)
        {
            /* gCG[ */
            ymodem_error_retry(object);
            return;
        }
        else
        {
            /* t@Cđ */
            ymodem_object_trans(object, YMODEM_STATE_SENDINFO);
        }
        break;

    /* ͉̑Ȃ */
    default:
        break;
    }
}

/*
 * YMODEM
 * t@CACK҂
 * exit
 */
static void ymodem_infoack_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * M'C'M҂(2)
 * enter
 */
static void ymodem_waitc2_enter(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* gC񐔂 */
    object->retry = 0;

    /* JnԂ */
    object->start_time = timeGetTime();
}

/*
 * YMODEM
 * M'C'M҂(2)
 * do
 */
static void ymodem_waitc2_do(YMODEM_OBJECT *object)
{
    BYTE buf;

    /* M̎MLN^҂ */
    buf = ymodem_wait_send(object);

    /* 'C'łΏԑJ */
    if (YMODEM_C == buf)
    {
        if (0 < object->file_size)
        {
            ymodem_object_trans(object, YMODEM_STATE_SENDDATA);
        }
        else
        {
            ymodem_object_trans(object, YMODEM_STATE_SENDEOT);
        }
    }
}

/*
 * YMODEM
 * M'C'M҂(2)
 * exit
 */
static void ymodem_waitc2_exit(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* gC񐔂 */
    object->retry = 0;

    /* ubNԍ */
    object->block_no = 0x01;

    /* obt@ItZbg */
    object->buf_offset = 0;
}

/*
 * YMODEM
 * f[^ubNM
 * enter
 */
static void ymodem_senddata_enter(YMODEM_OBJECT *object)
{
    UINT len;

    assert(NULL != object);

    /* ubNobt@ */
    memset(object->block_buf, 0, sizeof(object->block_buf));

    /* Rs[钷Zo(ő0x80) */
    assert(object->buf_offset < object->file_size);
    len = object->file_size - object->buf_offset;
    if (0x80 < len)
    {
        len = 0x80;
    }

    /* ubNItZbgf[^Rs[ */
    memcpy(&object->block_buf[3], &object->file_buf[object->buf_offset], len);

    /* ubNMJn */
    ymodem_block_sendstart(object);
}

/*
 * YMODEM
 * f[^ubNM
 * do
 */
static void ymodem_senddata_do(YMODEM_OBJECT *object)
{
    BOOL result;

    assert(NULL != object);

    /* ubNM */
    ymodem_block_senddo(object);

    /* ItZbg[܂œBԑJ */
    result = ymodem_check_offset(object);
    if (FALSE != result)
    {
        ymodem_object_trans(object, YMODEM_STATE_DATAACK);
    }
}

/*
 * YMODEM
 * f[^ubNM
 * exit
 */
static void ymodem_senddata_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * f[^ubNACK҂
 * enter
 */
static void ymodem_dataack_enter(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* JnԂ */
    object->start_time = timeGetTime();
}

/*
 * YMODEM
 * f[^ubNACK҂
 * do
 */
static void ymodem_dataack_do(YMODEM_OBJECT *object)
{
    BYTE buf;
    BOOL result;

    assert(NULL != object);

    /* M̎MLN^҂ */
    buf = ymodem_wait_send(object);

    switch (buf)
    {
        /* ACK */
    case YMODEM_ACK:
        /* ACK͎̃ubN܂EOTM */
        if ((object->buf_offset + 0x80) >= object->file_size)
        {
            /* gC񐔂 */
            object->retry = 0;

            /* EOTM */
            ymodem_object_trans(object, YMODEM_STATE_SENDEOT);
        }
        else
        {
            /* gC񐔂 */
            object->retry = 0;

            /* ubNԍi߂ */
            object->block_no = (object->block_no + 1) & 0xff;

            /* obt@ItZbgi߂ */
            object->buf_offset += 0x80;

            /* f[^ubNM */
            ymodem_object_trans(object, YMODEM_STATE_SENDDATA);
        }
        break;

        /* NAK */
    case YMODEM_NAK:
        /* NAK̓gC`FbN */
        result = ymodem_check_retry(object, YMODEM_SEND_RETRY);
        if (FALSE != result)
        {
            /* gCG[ */
            ymodem_error_retry(object);
            return;
        }
        else
        {
            /* f[^ubNđ */
            ymodem_object_trans(object, YMODEM_STATE_SENDDATA);
        }
        break;

        /* ͉̑Ȃ */
    default:
        break;
    }
}

/*
 * YMODEM
 * f[^ubNACK҂
 * exit
 */
static void ymodem_dataack_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * EOTM
 * enter
 */
static void ymodem_sendeot_enter(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * EOTM
 * do
 */
static void ymodem_sendeot_do(YMODEM_OBJECT *object)
{
    UINT result;
    BYTE buf;

    assert(NULL != object);

    /* EOTM */
    buf = YMODEM_EOT;
    result = comm_send(object->comm_index, &buf, sizeof(buf));
    if (0 == result)
    {
        /* ݂Ɏs */
        ymodem_error_closed(object);
    }
    else
    {
        /* ԑJ */
        ymodem_object_trans(object, YMODEM_STATE_EOTACK);
    }
}

/*
 * YMODEM
 * EOTM
 * exit
 */
static void ymodem_sendeot_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * EOTɑ΂ACK҂
 * enter
 */
static void ymodem_eotack_enter(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* JnԂ */
    object->start_time = timeGetTime();
}

/*
 * YMODEM
 * EOTɑ΂ACK҂
 * do
 */
static void ymodem_eotack_do(YMODEM_OBJECT *object)
{
    BYTE buf;
    BOOL result;

    assert(NULL != object);

    /* M̎MLN^҂ */
    buf = ymodem_wait_send(object);

    switch (buf)
    {
    /* ACK */
    case YMODEM_ACK:
        /* ACK͑MI */
        ymodem_error_success(object);
        break;

    /* NAK */
    case YMODEM_NAK:
        /* NAK̓gC`FbN */
        result = ymodem_check_retry(object, YMODEM_SEND_RETRY);
        if (FALSE != result)
        {
            /* gCG[ */
            ymodem_error_retry(object);
            return;
        }
        else
        {
            /* EOTđ */
            ymodem_object_trans(object, YMODEM_STATE_SENDEOT);
        }
        break;

        /* ͉̑Ȃ */
    default:
        break;
    }
}

/*
 * YMODEM
 * EOTɑ΂ACK҂
 * exit
 */
static void ymodem_eotack_exit(YMODEM_OBJECT *object)
{
    UNREFERENCED_PARAMETER(object);
}

/*
 * YMODEM
 * 'C'M(1)
 * enter
 */
static void ymodem_sendc1_enter(YMODEM_OBJECT *object)
{
}

/*
 * YMODEM
 * 'C'M(1)
 * do
 */
static void ymodem_sendc1_do(YMODEM_OBJECT *object)
{
    UINT result;
    BYTE buf;

    assert(NULL != object);

    /* CM */
    buf = YMODEM_C;
    result = comm_send(object->comm_index, &buf, sizeof(buf));
    if (0 == result)
    {
        /* ݂Ɏs */
        ymodem_error_closed(object);
    }
    else
    {
        /* ԑJ */
        ymodem_object_trans(object, YMODEM_STATE_WAITSOH);
    }
}

/*
 * YMODEM
 * 'C'M(1)
 * exit
 */
static void ymodem_sendc1_exit(YMODEM_OBJECT *object)
{
}

/*
 * YMODEM
 * SOHM҂
 * enter
 */
static void ymodem_waitsoh_enter(YMODEM_OBJECT *object)
{
    /* JnԂ */
    object->start_time = timeGetTime();
}

/*
 * YMODEM
 * SOHM҂
 * do
 */
static void ymodem_waitsoh_do(YMODEM_OBJECT *object)
{
    BYTE buf;

    /* M̎MLN^҂ */
    buf = ymodem_wait_recv(object);

    /* SOHłΏԑJ */
    if (YMODEM_SOH == buf)
    {
        ymodem_object_trans(object, YMODEM_STATE_SENDINFO);
    }
}

/*
 * YMODEM
 * SOHM҂
 * exit
 */
static void ymodem_waitsoh_exit(YMODEM_OBJECT *object)
{
}

/*
 * YMODEM
 * IuWFNgԑJڗv
 */
static void ymodem_object_trans(YMODEM_OBJECT *object, UINT state)
{
    assert(NULL != object);

    /* exit */
    switch (object->state)
    {
    /*  */
    case YMODEM_STATE_NONE:
        break;

    /* M'C'M҂(1) */
    case YMODEM_STATE_WAITC1:
        ymodem_waitc1_exit(object);
        break;

    /* t@C̑M */
    case YMODEM_STATE_SENDINFO:
        ymodem_sendinfo_exit(object);
        break;

    /* t@CACK҂ */
    case YMODEM_STATE_INFOACK:
        ymodem_infoack_exit(object);
        break;

    /* M'C'M҂(2) */
    case YMODEM_STATE_WAITC2:
        ymodem_waitc2_exit(object);
        break;

    /* f[^ubNM */
    case YMODEM_STATE_SENDDATA:
        ymodem_senddata_exit(object);
        break;

    /* f[^ubNACK҂ */
    case YMODEM_STATE_DATAACK:
        ymodem_dataack_exit(object);
        break;

    /* EOTM */
    case YMODEM_STATE_SENDEOT:
        ymodem_sendeot_exit(object);
        break;

    /* EOTɑ΂ACK҂ */
    case YMODEM_STATE_EOTACK:
        ymodem_eotack_exit(object);
        break;

    /* 'C'M(1) */
    case YMODEM_STATE_SENDC1:
        ymodem_sendc1_exit(object);
        break;

    /* SOHM҂ */
    case YMODEM_STATE_WAITSOH:
        ymodem_waitsoh_exit(object);
        break;
    }

    /* ԕύX */
    object->state = state;

    /* enter */
    switch (object->state)
    {
    /* M'C'M҂(1) */
    case YMODEM_STATE_WAITC1:
        ymodem_waitc1_enter(object);
        break;

    /* t@C̑M */
    case YMODEM_STATE_SENDINFO:
        ymodem_sendinfo_enter(object);
        break;

    /* t@CACK҂ */
    case YMODEM_STATE_INFOACK:
        ymodem_infoack_enter(object);
       break;

    /* M'C'M҂(2) */
    case YMODEM_STATE_WAITC2:
        ymodem_waitc2_enter(object);
        break;

    /* f[^ubNM */
    case YMODEM_STATE_SENDDATA:
        ymodem_senddata_enter(object);
        break;

    /* f[^ubNACK҂ */
    case YMODEM_STATE_DATAACK:
        ymodem_dataack_enter(object);
        break;

    /* EOTM */
    case YMODEM_STATE_SENDEOT:
        ymodem_sendeot_enter(object);
        break;

    /* EOTɑ΂ACK҂ */
    case YMODEM_STATE_EOTACK:
        ymodem_eotack_enter(object);
        break;

    /* 'C'M(1) */
    case YMODEM_STATE_SENDC1:
        ymodem_sendc1_enter(object);
        break;

    /* SOHM҂ */
    case YMODEM_STATE_WAITSOH:
        ymodem_waitsoh_enter(object);
        break;
    }
}

/*
 * YMODEM
 * IuWFNgACh
 */
static void ymodem_object_idle(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* M܂͎M̂ǂłȂ΁AȂ */
    if ((FALSE == object->sending) && (FALSE == object->receiving))
    {
        return;
    }

    /* do */
    assert(YMODEM_STATE_NONE != object->state);
    switch (object->state)
    {
        /* M'C'M҂(1) */
    case YMODEM_STATE_WAITC1:
        ymodem_waitc1_do(object);
        break;

        /* t@C̑M */
    case YMODEM_STATE_SENDINFO:
        ymodem_sendinfo_do(object);
        break;

        /* t@CACK҂ */
    case YMODEM_STATE_INFOACK:
        ymodem_infoack_do(object);
        break;

        /* M'C'M҂(2) */
    case YMODEM_STATE_WAITC2:
        ymodem_waitc2_do(object);
        break;

        /* f[^ubNM */
    case YMODEM_STATE_SENDDATA:
        ymodem_senddata_do(object);
        break;

        /* f[^ubNACK҂ */
    case YMODEM_STATE_DATAACK:
        ymodem_dataack_do(object);
        break;

    /* EOTM */
    case YMODEM_STATE_SENDEOT:
        ymodem_sendeot_do(object);
        break;

    /* EOTɑ΂ACK҂ */
    case YMODEM_STATE_EOTACK:
        ymodem_eotack_do(object);
        break;

    /* 'C'M(1) */
    case YMODEM_STATE_SENDC1:
        ymodem_sendc1_do(object);
        break;

    /* SOHM҂ */
    case YMODEM_STATE_WAITSOH:
        ymodem_waitsoh_do(object);
        break;
    }
}

/*
 * YMODEM
 * IuWFNgMJn
 */
static BOOL ymodem_object_send(YMODEM_OBJECT *object, const char *filename, UINT filesize)
{
    size_t namelen;
    BOOL result;

    assert(NULL != object);
    assert(NULL != filename);

    /* M܂͎Mł΃G[ */
    assert((FALSE == object->sending) && (FALSE == object->receiving));
    if ((FALSE != object->sending) || (FALSE != object->receiving))
    {
        /* M܂͎Mł */
        return FALSE;
    }

    /* t@Cl[elȏł΃G[ */
    namelen = strlen(filename);
    assert(namelen < _countof(object->file_name));
    if (namelen >= _countof(object->file_name))
    {
        /* t@Cl[ */
        return FALSE;
    }

    /* t@CTCYobt@TCY𒴂Ă΃G[ */
    assert(filesize <= sizeof(object->file_buf));
    if (filesize > sizeof(object->file_buf))
    {
        /* t@CTCY傫 */
        return FALSE;
    }

    /* ʐM|[gI[vłȂ΃G[ */
    result = comm_open(object->comm_index);
    if (FALSE == result)
    {
        return FALSE;
    }

    /* M */
    object->sending = TRUE;

    /* BUSY */
    object->status = YMODEM_BUSY;

    /* t@Cl[ */
    strcpy_s(object->file_name, _countof(object->file_name), filename);

    /* t@CTCY */
    object->file_size = filesize;

    /* ԑJ */
    ymodem_object_trans(object, YMODEM_STATE_WAITC1);

    /*  */
    return TRUE;
}

/*
 * YMODEM
 * IuWFNgMJn
 */
static BOOL ymodem_object_recv(YMODEM_OBJECT *object)
{
    BOOL result;

    assert(NULL != object);

    /* M܂͎Mł΃G[ */
    assert((FALSE == object->sending) && (FALSE == object->receiving));
    if ((FALSE != object->sending) || (FALSE != object->receiving))
    {
        /* M܂͎Mł */
        return FALSE;
    }

    /* ʐM|[gI[vłȂ΃G[ */
    result = comm_open(object->comm_index);
    if (FALSE == result)
    {
        return FALSE;
    }

    /* M */
    object->receiving = TRUE;

    /* BUSY */
    object->status = YMODEM_BUSY;

    /* Mobt@̓ǂݎ̂ */
    ymodem_discard_recv(object);

    /* gC񐔂 */
    object->retry = 0;

    /* ԑJ */
    ymodem_object_trans(object, YMODEM_STATE_SENDC1);

    /*  */
    return TRUE;
}

/*
 * YMODEM
 * IuWFNgj^
 */
static void ymodem_object_monitor(YMODEM_OBJECT *object)
{
    assert(NULL != object);

    /* ʒuƃNA*/
    monitor_locate(YMODEM_MONITOR_X, YMODEM_MONITOR_Y);
    monitor_clear(40, 15);

    /* ^Cg */
    monitor_printf("[YMODEM]\n");

    /* ʐM|[gԍ */
    monitor_printf("ʐM|[gԍ   %u\n", object->comm_index);

    /* MtO */
    if (FALSE == object->sending)
    {
        monitor_printf("M         ACh\n");
   }
    else
    {
        monitor_printf("M         M\n");
    }

    /* MtO */
    if (FALSE == object->receiving)
    {
        monitor_printf("M         ACh\n");
    }
    else
    {
        monitor_printf("M         M\n");
    }

    /* Xe[^X */
    switch (object->status)
    {
    /*  */
    case YMODEM_SUCCESS:
        monitor_printf("Xe[^X       \n");
        break;

    /* 쒆 */
    case YMODEM_BUSY:
        monitor_printf("Xe[^X       쒆\n");
        break;

    /* obt@TCY */
    case YMODEM_SIZEOVER:
        monitor_printf("Xe[^X       obt@TCY\n");
        break;

    /* ^CAEg */
    case YMODEM_TIMEOUT:
        monitor_printf("Xe[^X       ^CAEg\n");
        break;

    /* gCI[o[ */
    case YMODEM_RETRY:
        monitor_printf("Xe[^X       gCI[o[\n");
        break;

    /* LZvM */
    case YMODEM_CANCEL:
        monitor_printf("Xe[^X       LZvM\n");
        break;

    /* |[gN[Y */
    case YMODEM_CLOSED:
        monitor_printf("Xe[^X       |[gN[Y\n");
        break;
    }

    /* t@Cl[ */
    monitor_printf("t@Cl[   %s\n", object->file_name);

    /* t@CTCY */
    monitor_printf("t@CTCY   %u\n", object->file_size);

    /*  */
    switch (object->state)
    {
    /*  */
    case YMODEM_STATE_NONE:
        monitor_printf("         \n");
        break;

    /* M'C'M҂(1) */
    case YMODEM_STATE_WAITC1:
        monitor_printf("         M'C'M҂(1)\n");
        break;

    /* t@C񑗐M */
    case YMODEM_STATE_SENDINFO:
        monitor_printf("         t@C񑗐M\n");
        break;

    /* t@CACK҂ */
    case YMODEM_STATE_INFOACK:
        monitor_printf("         t@CACK҂\n");
        break;

    /* M'C'M҂(2) */
    case YMODEM_STATE_WAITC2:
        monitor_printf("         M'C'M҂(2)\n");
        break;

    /* f[^ubNM */
    case YMODEM_STATE_SENDDATA:
        monitor_printf("         f[^ubNM\n");
        break;

    /* f[^ubNACK҂ */
    case YMODEM_STATE_DATAACK:
        monitor_printf("         f[^ubNACK҂\n");
        break;

    /* EOTM */
    case YMODEM_STATE_SENDEOT:
        monitor_printf("         EOTM\n");
        break;

    /* EOTɑ΂ACK҂ */
    case YMODEM_STATE_EOTACK:
        monitor_printf("         EOTɑ΂ACK҂\n");
        break;

    /* 'C'M(1) */
    case YMODEM_STATE_SENDC1:
        monitor_printf("         'C'M(1)\n");
        break;

    /* SOHM҂ */
    case YMODEM_STATE_WAITSOH:
        monitor_printf("         SOHM҂\n");
        break;
    }

    /* ubNԍ */
    monitor_printf("ubNԍ     0x%02X\n", object->block_no);

    /* obt@ItZbg */
    monitor_printf("ItZbg       0x%04X\n", object->buf_offset);

    /* ACKM */
    monitor_printf("ACKM    %u\n", object->ack_cnt);

    /* NAKM */
    monitor_printf("NAKM    %u\n", object->nak_cnt);

    /* CANM */
    monitor_printf("CANM    %u\n", object->can_cnt);

    /* ^CAEg */
    monitor_printf("^CAEg %u\n", object->timeout_cnt);

    /* gC */
    monitor_printf("gC     %u\n", object->retry_cnt);
}

/*
 * YMODEM
 * 
 */
void ymodem_init(UINT comm_index)
{
    /* IuWFNg */
    ymodem_object_init(&g_ymodem_object, comm_index);
}

/*
 * YMODEM
 * ACh
 */
void ymodem_idle(void)
{
    /* IuWFNgACh */
    ymodem_object_idle(&g_ymodem_object);

    /* j^ */
    ymodem_object_monitor(&g_ymodem_object);
}

/*
 * YMODEM
 * MJn
 */
BOOL ymodem_send(const char* filename, UINT filesize)
{
    assert(NULL != filename);

    /* IuWFNgMJn */
    return ymodem_object_send(&g_ymodem_object, filename, filesize);
}

/*
 * YMODEM
 * MJn
 */
BOOL ymodem_recv(void)
{
    /* IuWFNgMJn */
    return ymodem_object_recv(&g_ymodem_object);
}

/*
 * YMODEM
 * Xe[^X擾
 */
UINT ymodem_get_status(void)
{
    return g_ymodem_object.status;
}

/*
 * YMODEM
 * t@Cl[擾
 */
const char* ymodem_get_filename(void)
{
    return NULL;
}

/*
 * YMODEM
 * t@CTCY擾
 */
UINT ymodem_get_filesize(void)
{
    return g_ymodem_object.file_size;
}

/*
 * YMODEM
 * obt@|C^擾
 */
BYTE* ymodem_get_filebuf(void)
{
    return g_ymodem_object.file_buf;
}
