﻿// TcpServer.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// マルチスレッド TCP サーバ。
/// </summary>
/// @since 2015/8/12
using System;
using System.Collections.ObjectModel;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;

/// <summary>
/// ソケットを使った C# ネットワークライブラリ。
/// </summary>
namespace SocketNet
{
	/// <summary>
	/// マルチスレッド TCP サーバ。
	/// </summary>
	/// <remarks>
	/// <para>
	/// このサーバは、TCPメッセージを非同期的に受信します。
	/// <see cref="Start"/> メソッドで待ち受けを開始します。
	/// 接続が確立すると <see cref="Connected"/> イベントが発行されます。
	/// 接続が切断されると <see cref="Disconnected"/> イベントが発行されます。
	/// メッセージを完全に受信すると <see cref="DataReceived"/> イベントが発行されます。
	/// </para>
	/// <para>
	/// 受信するメッセージについては <see cref="TcpConnection"/> クラスの説明を参照してください。
	/// </para>
	/// </remarks>
	public sealed class TcpServer: IDisposable
	{
		/// <summary>
		/// ペンディングできる接続の最大数。
		/// </summary>
		public static readonly int MaxPendingConnections = 3;

		private readonly PacketSpec PacketSpec;

		private TcpListener tcpListener;
		private List<TcpConnection> clientConnections;
		private List<TcpConnection> connectionsToClose;
		private volatile bool isShuttingDown;
		private volatile bool acceptingConnections;

		private bool disposed = false;

		/// <summary>
		/// 接続が確立した時に発行されます。
		/// </summary>
		public event EventHandler<TcpConnectionEventArgs> Connected;

		/// <summary>
		/// 接続が切断された時に発行されます。
		/// </summary>
		public event EventHandler<TcpConnectionEventArgs> Disconnected;

		/// <summary>
		/// データを受信した時に発行されます。
		/// </summary>
		public event EventHandler<TcpDataReceivedEventArgs> DataReceived;

		/// <summary>
		/// Tcpサーバにバインドされている IP アドレスを取得します。
		/// </summary>
		public IPAddress IPAddress { get; private set; }

		/// <summary>
		/// Tcpサーバにバインドされているポート番号を取得します。
		/// </summary>
		public int Port { get; private set; }

		/// <summary>
		/// サーバの状態を取得します。
		/// </summary>
		public bool IsRunning
		{
			get
			{
				return acceptingConnections;
			}
		}

		/// <summary>
		/// アクティブな接続の数を取得します。
		/// </summary>
		public int ActiveConnectionCount
		{
			get
			{
				return clientConnections.Count;
			}
		}

		/// <summary>
		/// アクティブな接続のリストを取得します。
		/// </summary>
		public ReadOnlyCollection<TcpConnection> ActiveConnections
		{
			get
			{
				lock (clientConnections)
				{
					return clientConnections.AsReadOnly();
				}
			}
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="port">バインドするポート。</param>
		/// <param name="packet">受信するパケットの各部の長さの設定を格納するオブジェクト。</param>
		/// <remarks>
		/// ループバックアドレスを使用します。
		/// </remarks>
		public TcpServer(int port, PacketSpec packet)
			: this(IPAddress.Loopback, port, packet)
		{
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="ipAddress">バインドするIPアドレス。</param>
		/// <param name="port">バインドするポート。</param>
		/// <param name="packetSpec">受信するパケットの各部の長さの設定を格納するオブジェクト。</param>
		public TcpServer(IPAddress ipAddress, int port, PacketSpec packetSpec)
		{
			IPAddress = ipAddress;
			Port = port;
			PacketSpec = packetSpec;
			clientConnections = new List<TcpConnection>();
			connectionsToClose = new List<TcpConnection>();
			isShuttingDown = false;
		}

		/// <summary>
		/// オブジェクトに関連付けられたリソースを解放します。
		/// </summary>
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		/// <summary>
		/// オブジェクトに関連付けられたリソースを解放します。
		/// </summary>
		/// <param name="disposing">
		/// メソッドの呼び出し元が Dispose() メソッドか (値は <c>true</c>)、
		/// それともファイナライザーか (値は <c>false</c>) を示します。
		/// </param>
		private void Dispose(bool disposing)
		{
			if (disposed)
			{
				return;
			}

			if (disposing)
			{
				if (isShuttingDown == false)
				{
					Stop();
				}

				lock (clientConnections)
				{
					foreach (TcpConnection connection in clientConnections)
					{
						connection.Dispose();
					}

					clientConnections.Clear();
					clientConnections = null;
				}

				lock (connectionsToClose)
				{
					connectionsToClose.Clear();
					connectionsToClose = null;
				}
			}

			disposed = true;
		}

		/// <summary>
		/// TCPサーバをスタートします。
		/// </summary>
		/// <remarks>非同期で呼び出します。</remarks>
		/// <returns>メソッドに関連付けられた Task オブジェクト。</returns>
		public async Task Start()
		{
			tcpListener = null;

			try
			{
				isShuttingDown = false;
				acceptingConnections = true;

				tcpListener = new TcpListener(IPAddress, Port);
				tcpListener.Start(MaxPendingConnections);

				while (true)
				{
					Socket socket = await tcpListener.AcceptSocketAsync();
					if (socket == null)
					{
						break;
					}

					await Task.Run(() =>
						{
							TcpConnection connection = new TcpConnection(socket, PacketSpec);
							connection.Disconnected += new EventHandler<TcpConnectionEventArgs>(OnDisconnected);
							connection.DataReceived += new EventHandler<TcpDataReceivedEventArgs>(OnDataReceived);

							lock (clientConnections)
							{
								clientConnections.Add(connection);
							}

							connection.ReceiveDataAsync();

							OnConnected(new TcpConnectionEventArgs(connection));
						});
				}
			}
			catch (ObjectDisposedException)
			{
				// Supress exception
			}
			finally
			{
				lock (clientConnections)
				{
					lock (connectionsToClose)
					{
						foreach (TcpConnection connection in clientConnections)
						{
							connectionsToClose.Add(connection);
						}
					}

					CloseMarkedConnections();
				}

				acceptingConnections = false;
				isShuttingDown = true;
			}
		}

		/// <summary>
		/// Tcpサーバを停止します。
		/// </summary>
		public void Stop()
		{
			tcpListener.Stop();
		}

		/// <summary>
		/// アクティブな接続をクローズします。
		/// </summary>
		/// <param name="connection">クローズする接続。</param>
		public void CloseConnection(TcpConnection connection)
		{
			try
			{
				connection.Dispose();
			}
			catch
			{
				// Igore any exceptions
			}
			finally
			{
				lock (clientConnections)
				{
					clientConnections.Remove(connection);
				}
			}
		}

		/// <summary>
		/// 接続イベントを発行します。
		/// </summary>
		/// <param name="e">イベントデータを格納した <see cref="TcpConnectionEventArgs"/> オブジェクト。</param>
		private void OnConnected(TcpConnectionEventArgs e)
		{
			if (Connected != null)
			{
				Connected(this, e);
			}
		}

		/// <summary>
		/// 切断イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納する <see cref="TcpConnectionEventArgs"/> オブジェクト。</param>
		private void OnDisconnected(object sender, TcpConnectionEventArgs e)
		{
			if (Disconnected != null)
			{
				CloseConnection(e.Connection);
				Disconnected(this, e);
			}
		}

		/// <summary>
		/// データ受信イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納する <see cref="TcpDataReceivedEventArgs"/> オブジェクト。</param>
		private void OnDataReceived(object sender, TcpDataReceivedEventArgs e)
		{
			if (DataReceived != null)
			{
				DataReceived(this, e);
			}
		}

		/// <summary>
		/// マークされた接続をクローズします。
		/// </summary>
		private void CloseMarkedConnections()
		{
			lock (connectionsToClose)
			{
				foreach (TcpConnection connection in connectionsToClose)
				{
					CloseConnection(connection);
				}

				connectionsToClose.Clear();
			}
		}
	}
}

