﻿// Packet.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// パケットの定数を提供するクラス。
/// </summary>
/// @since 2015.8.14
using System;

namespace SocketNet
{
	/// <summary>
	/// パケットのヘッダ長、フッタ長、ペイロード長を取得するためのプロパティ・メソッドを実装するクラス。
	/// </summary>
	/// <remarks>
	/// 各長さを変更する場合は、このクラスから継承してプロパティ・メソッドをオーバライドします。
	/// </remarks>
	public class PacketSpec
	{
		/// <summary>
		/// ヘッダ長を取得します。
		/// </summary>
		public virtual int HeaderLength { get; private set; }

		/// <summary>
		/// フッタ長を取得します。
		/// </summary>
		public virtual int FooterLength { get; private set; }

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="headerLength">ヘッダ長。</param>
		/// <param name="footerLength">フッタ長。</param>
		public PacketSpec(int headerLength = 4, int footerLength = 0)
		{
			HeaderLength = headerLength;
			FooterLength = footerLength;
		}

		/// <summary>
		/// 指定されたヘッダからペイロード長を取得します。
		/// </summary>
		/// <returns>ペイロード長。</returns>
		/// <param name="header">ヘッダ。</param>
		public virtual int ObtainPayloadLength(byte[] header)
		{
			int payloadLength = 0;

			// header は必ず HeaderLength の長さで送られてくるためチェックする必要はない。
			// if (header.Length < HeaderLength)
			// {
			//     return 0;
			// }
			for (int i = 0; i < HeaderLength; i++)
			{
				payloadLength = (payloadLength << 8) | header[i];
			}

			return payloadLength;
		}
	}
}

