﻿// MainWindow.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// GTK# ベースの簡単なチャットクライアントのウィンドウを実装。
/// </summary>
/// @since 2015.8.14
using System;
using System.Net;
using System.Net.Sockets;
using Gtk;
using SocketNet;
using SampleChatClient;

/// <summary>
/// SocketNet ライブラリのサンプルプログラム (GTK# ベースの簡単なチャットクライアント)。
/// </summary>
public partial class MainWindow: Gtk.Window
{
	/// <summary>
	/// チャットサーバに接続するためのクライアントを取得します。
	/// </summary>
	/// <value>The client.</value>
	public ChatClient Client { get; private set; }

	/// <summary>
	/// コンストラクタ。
	/// </summary>
	public MainWindow () : base(Gtk.WindowType.Toplevel)
	{
		Build();
	}

	/// <summary>
	/// ウィンドウのデリートイベントを処理します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="a">イベントデータを格納したオブジェクト。</param>
	protected void OnDeleteEvent(object sender, DeleteEventArgs a)
	{
		Application.Quit();
		a.RetVal = true;
	}

	/// <summary>
	/// 接続ボタンがクリックされたときに、接続処理を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnConnectButtonClicked(object sender, EventArgs e)
	{
		try
		{
			IPAddress serverIPAddress = IPAddress.Parse(serverAddressEntry.Text);
			Client = new ChatClient(serverIPAddress, 50001);
			Client.ChatMessageReceived += OnChatMessageReceived;
			Client.Connect();
			connectButton.Sensitive = false;
			disconnectButton.Sensitive = true;
			serverAddressEntry.Sensitive = false;
			messageEntry.Sensitive = true;
			Client.BeginReceiveMessage();
		}
		catch (FormatException ex)
		{
			messageTextview.Buffer.Text = ex.Message + "\n";
		}
		catch (SocketException ex)
		{
			messageTextview.Buffer.Text = ex.Message + "\n";
		}
	}

	/// <summary>
	/// 切断ボタンがクリックされたときに、切断処理を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnDisconnectButtonClicked(object sender, EventArgs e)
	{
		Client.Close();
		Client = null;
		connectButton.Sensitive = true;
		disconnectButton.Sensitive = false;
		serverAddressEntry.Sensitive = true;
		sendButton.Sensitive = false;
		messageEntry.Sensitive = false;
	}

	/// <summary>
	/// メッセージ入力エリアが変更されたときに、送信ボタンの有効化・無効化を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnMessageEntryChanged(object sender, EventArgs e)
	{
		sendButton.Sensitive = messageEntry.Text.Length > 0;
	}

	/// <summary>
	/// チャットメッセージを受信したときに、テキストビューにメッセージを追加します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	private void OnChatMessageReceived(object sender, ChatMessageReceivedEventArgs e)
	{
		messageTextview.Buffer.Text += e.ChatMessage.Message + "\n";
	}

	/// <summary>
	/// 送信ボタンをクリックされたときに、メッセージ入力エリアの文字列をサーバに送信します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnSendButtonClicked(object sender, EventArgs e)
	{
		var message = new ChatMessage(messageEntry.Text);
		Client.Send(message);
	}

	/// <summary>
	/// メッセージ入力エリアでリターンキーが押されたときに、送信ボタンをクリックする動作を実行します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnMessageEntryActivated(object sender, EventArgs e)
	{
		sendButton.Click();
	}

	/// <summary>
	/// サーバアドレス入力エリアが変更されたときに、接続ボタンの有効・無効を切り替えます。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnServerAddressEntryChanged(object sender, EventArgs e)
	{
		connectButton.Sensitive = serverAddressEntry.Text.Length > 0;
	}
}
