﻿// ChatServer.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// SocketNet ライブラリのサンプルの簡単なチャットサーバ。
/// </summary>
/// @since 2015.8.14
using System;
using System.Collections.ObjectModel;
using System.Collections.Generic;
using System.Net;
using SocketNet;
using SampleChatClient;

/// <summary>
/// SocketNet ライブラリのサンプルプログラム (コンソールベースの簡単なチャットサーバ)。
/// </summary>
namespace SampleChatServer
{
	/// <summary>
	/// <see cref="SocketNet.TcpServer"/> クラスを利用したチャットサーバ。
	/// </summary>
	public class ChatServer
	{
		private TcpServer tcpServer;

		/// <summary>
		/// 接続が確立した時に発行されます。
		/// </summary>
		public event EventHandler<TcpConnectionEventArgs> Connected;
		
		/// <summary>
		/// 接続が切断された時に発行されます。
		/// </summary>
		public event EventHandler<TcpConnectionEventArgs> Disconnected;

		/// <summary>
		/// チャットメッセージを受信した時に発行されます。
		/// </summary>
		public event EventHandler<ChatMessageReceivedEventArgs> ChatMessageReceived;

		/// <summary>
		/// アクティブな接続のリストを取得します。
		/// </summary>
		public ReadOnlyCollection<TcpConnection> ActiveConnections { get { return tcpServer.ActiveConnections; } }

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="ipAddress">待ち受けするIPアドレス。</param>
		/// <param name="port">待ち受けするポート番号。</param>
		public ChatServer (IPAddress ipAddress, int port)
		{
			tcpServer = new TcpServer(ipAddress, port, new ChatMessage());
			tcpServer.Connected += OnConnected;
			tcpServer.Disconnected += OnDisconnected;
			tcpServer.DataReceived += OnDataReceived;
		}

		/// <summary>
		/// チャットサーバを開始します。
		/// </summary>
		public void Start()
		{
			tcpServer.Start();
		}

		/// <summary>
		/// チャットサーバを停止します。
		/// </summary>
		public void Stop()
		{
			tcpServer.Stop();
		}

		/// <summary>
		/// 接続イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納したオブジェクト。</param>
		private void OnConnected(object sender, TcpConnectionEventArgs e)
		{
			if (Connected != null)
			{
				Connected(this, e);
			}
		}

		/// <summary>
		/// 切断イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納する <see cref="TcpConnectionEventArgs"/> オブジェクト。</param>
		private void OnDisconnected(object sender, TcpConnectionEventArgs e)
		{
			if (Disconnected != null)
			{
				Disconnected(this, e);
			}
		}

		/// <summary>
		/// チャットメッセージ受信イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		private void OnDataReceived(object sender, TcpDataReceivedEventArgs e)
		{
			if (ChatMessageReceived != null)
			{
				ChatMessage chatMessage = ChatMessage.FromByteArray(e.Data);
				chatMessage.Connection = e.Connection;
				ChatMessageReceived(this, new ChatMessageReceivedEventArgs(chatMessage));
			}
		}
	}
}

