/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.twitter;

import android.content.Context;
import android.content.SharedPreferences;
import android.test.InstrumentationTestCase;

import twitter4j.DirectMessage;
import twitter4j.Status;
import twitter4j.User;
import twitter4j.internal.http.HttpResponseCode;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException.StatusCode;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;
import com.sonyericsson.eventstream.twitterplugin.utils.MockTwitter;
import com.sonyericsson.eventstream.twitterplugin.utils.MockTwitterFactory;
import com.sonyericsson.eventstream.twitterplugin.utils.ReflectionUtilities;

import junit.framework.Assert;

import java.util.LinkedList;
import java.util.List;

/**
 *
 * @author 23049952
 *
 */
public class UTwitterCommunicationTests extends InstrumentationTestCase {

    Context mContext;

    TwitterCommunication mTwitterCommunication;

    MockTwitter mTwitter;

    MockTwitterFactory mTwitterFactory;

    @Override
    protected void setUp() throws Exception {
        mContext = getInstrumentation().getTargetContext();
        mTwitterCommunication = new TwitterCommunication(mContext);
        mTwitter = new MockTwitter();
        mTwitterFactory = new MockTwitterFactory();

        try {
            ReflectionUtilities.getField(mTwitterCommunication, "mTwitterFactory").set(mTwitterCommunication,
                    mTwitterFactory);
        } catch (Exception e) {
            Assert.fail("Could not change the factory field");
        }

        try {
            ReflectionUtilities.getField(mTwitterCommunication, "mTwitter").set(mTwitterCommunication,
                    mTwitter);
        } catch (Exception e) {
            Assert.fail("Could not change the twitter field");
        }

        // Clear all values
        SharedPreferences preferences = mContext.getSharedPreferences(
                PreferencesHelper.CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.clear();
                editor.commit();
            }
        }

        mTwitter.performThrowExceptionOnCall(false, 0);
        mTwitter.returnNullOnCall(false);

        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        mTwitterCommunication = null;
        mTwitter = null;
        mTwitterFactory = null;

        super.tearDown();
    }

    /**
    * Test status update with valid content
    */
   public void testUpdateValidStatus() {
       boolean exceptionThrown = false;
       mTwitter.setMethodCallCount("updateStatus");
        try {
            mTwitterCommunication.updateStatus("Status");
        } catch (Exception e) {
            exceptionThrown = true;
        }
        assertEquals("Exceptions should not be thrown", false, exceptionThrown);
        assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test status update and check that it catch and throws the correct exception
    */
   public void testUpdateStatusThrowException() {
       boolean exceptionThrown = false;
       mTwitter.performThrowExceptionOnCall(true, HttpResponseCode.BAD_REQUEST);
       try {
           mTwitterCommunication.updateStatus("Status");
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exception should be thrown", true, exceptionThrown);
   }

   /**
    * Test that it can handle null return values
    */
   public void testUpdateStatusNullInparam() {
       boolean exceptionThrown = false;
       mTwitter.returnNullOnCall(true);
       try {
           mTwitterCommunication.updateStatus(null);
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
   }

   /**
    * Test get status with valid content
    */
   public void testGetOwnUser() {
       boolean exceptionThrown = false;
       mTwitter.setMethodCallCount("showUser");
        try {
            User ownUser = mTwitterCommunication.getOwnUser();
            assertNotNull(ownUser);
        } catch (Exception e) {
            exceptionThrown = true;
        }
        assertEquals("Exceptions should not be thrown", false, exceptionThrown);
        assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test get status and check that it catch and throws the correct exception
    */
   public void testGetOwnUserThrowException() {
       boolean exceptionThrown = false;
       mTwitter.performThrowExceptionOnCall(true, HttpResponseCode.BAD_REQUEST);
       try {
           mTwitterCommunication.getOwnUser();
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exception should be thrown", true, exceptionThrown);
   }

   /**
    * Test that it can handle null return values
    */
   public void testGetOwnUserNullInparam() {
       boolean exceptionThrown = false;
       mTwitter.returnNullOnCall(true);
       try {
           mTwitterCommunication.getOwnUser();
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should be thrown", true, exceptionThrown);
   }


    /**
     * Test to set a valid paging since id
     */
    public void testSetPaging() {
        boolean exceptionThrown = false;

        List<User> users = mTwitter.generateSomeUsers(1);
        List<Status> statuses = mTwitter.generateStatuses(users, 10);
        List<DirectMessage> messages = mTwitter.generateMessages(users, 10);

        try {
            mTwitterCommunication.setPagingSinceForStatus(statuses);
            mTwitterCommunication.setPagingSinceForMessages(messages);
        } catch (Exception e) {
            exceptionThrown = true;
        }
        assertEquals("Exceptions should not be thrown", false, exceptionThrown);
    }

    /**
     * Test to set a invalid paging since id, null params.
     */
    public void testSetPagingEmptyInparam() {
        boolean exceptionThrown = false;

        List<Status> statuses = new LinkedList<Status>();
        List<DirectMessage> messages = new LinkedList<DirectMessage>();

        try {
            mTwitterCommunication.setPagingSinceForStatus(statuses);
            mTwitterCommunication.setPagingSinceForMessages(messages);
        } catch (Exception e) {
            exceptionThrown = true;
        }
        assertEquals("Exceptions should not be thrown", false, exceptionThrown);
    }

   /**
    * Test to set a invalid paging since id, empty inparam lists.
    */
   public void testSetPagingNullInparam() {
       boolean exceptionThrown = false;

       try {
           mTwitterCommunication.setPagingSinceForStatus(null);
           mTwitterCommunication.setPagingSinceForMessages(null);
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
   }

   /**
    * Test that it is possible to get a valid home time line content
    */
   public void testGetValidHomeTimeline() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
       boolean exceptionThrown = false;
       List<Status> statuses = null;
       mTwitter.setMethodCallCount("getHomeTimeline");
       try {
           statuses = mTwitterCommunication.getHomeTimeline();
           assertNotNull(statuses);
           assertTrue("Should contain content", statuses.size() > 0);
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that the home time line do not contain any own statuses.
    */
   public void testGetValidHomeTimelineFilterOwnStatuses() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
       boolean exceptionThrown = false;
       List<Status> statuses = null;
       mTwitter.setMethodCallCount("getHomeTimeline");
       try {
           statuses = mTwitterCommunication.getHomeTimeline();
           assertNotNull(statuses);
           assertTrue("Should contain content", statuses.size() > 0);
           User ownUser = mTwitterCommunication.getOwnUser();
           for (Status status : statuses) {
               assertTrue(status.getUser().getId() != ownUser.getId());
           }
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it handles home time line request with internal error
    */
   public void testGetHomeTimelineThrow() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
       mTwitter.performThrowExceptionOnCall(true, HttpResponseCode.BAD_REQUEST);
       boolean exceptionThrown = false;

       try {
           mTwitterCommunication.getHomeTimeline();
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should be thrown", true, exceptionThrown);
   }

   /**
    * Test that it handles home time line request with internal null
    */
   public void testGetHomeTimelineNullParams() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
       mTwitter.returnNullOnCall(true);
       boolean exceptionThrown = false;
       List<Status> statuses = null;

       try {
           statuses = mTwitterCommunication.getHomeTimeline();
           assertNotNull(statuses);
           assertEquals("Size should be zero", 0, statuses.size());
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
   }

   /**
    * Test that it is possible to get a valid DirectMessages content
    */
   public void testGetValidDirectMessages() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, MockTwitter.DEFAULT_PAGING_ID);
       boolean exceptionThrown = false;
       List<DirectMessage> messages = null;
       mTwitter.setMethodCallCount("getDirectMessages");
       try {
           messages = mTwitterCommunication.getDirectMessages();
           assertNotNull(messages);
           assertTrue("Should contain content", messages.size() > 0);
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it handles DirectMessages request with internal error
    */
   public void testGetDirectMessagesThrow() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, MockTwitter.DEFAULT_PAGING_ID);
       mTwitter.performThrowExceptionOnCall(true, HttpResponseCode.BAD_REQUEST);
       boolean exceptionThrown = false;

       try {
           mTwitterCommunication.getDirectMessages();
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should be thrown", true, exceptionThrown);
   }

   /**
    * Test that it handles DirectMessages request with internal null
    */
   public void testGetDirectMessagesNullParams() {
       PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, MockTwitter.DEFAULT_PAGING_ID);
       mTwitter.returnNullOnCall(true);
       boolean exceptionThrown = false;
       List<DirectMessage> messages = null;

       try {
           messages = mTwitterCommunication.getDirectMessages();
           assertNotNull(messages);
           assertEquals("Size should be zero", 0, messages.size());
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
   }

   /**
    * Test that it is possible to get a valid FriendsList content,
    * 0 users expected
    */
   public void testGetValidFriendsListEmpty() {
       boolean exceptionThrown = false;
       List<User> users = null;
       mTwitter.generateUsers(0);
       mTwitter.setMethodCallCount("getFriendsStatuses");

       try {
           users = mTwitterCommunication.getFriendsList();
           assertNotNull(users);
           assertTrue("Should contain content", users.size() == 0);
       } catch (Exception e) {
           e.printStackTrace();
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it is possible to get a valid FriendsList content,
    * 3 users expected
    */
   public void testGetValidFriendsList() {
       boolean exceptionThrown = false;
       List<User> users = null;
       mTwitter.generateUsers(3);
       mTwitter.setMethodCallCount("getFriendsStatuses");

       try {
           users = mTwitterCommunication.getFriendsList();
           assertNotNull(users);
           assertTrue("Should contain content", users.size() == 3);
       } catch (Exception e) {
           e.printStackTrace();
           exceptionThrown = true;
       }
       assertFalse(exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it is possible to get a valid FriendsList content,
    * 100 users expected
    */
   public void testGetValidFriendsList100() {
       boolean exceptionThrown = false;
       List<User> users = null;
       mTwitter.generateUsers(100);
       mTwitter.setMethodCallCount("getFriendsStatuses");

       try {
           users = mTwitterCommunication.getFriendsList();
           assertNotNull(users);
           assertTrue("Should contain content", users.size() == 100);
       } catch (Exception e) {e.printStackTrace();
           exceptionThrown = true;
       }
       assertFalse(exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it is possible to get a valid FriendsList content,
    * 999 users expected
    */
   public void testGetValidFriendsListMany() {
       boolean exceptionThrown = false;
       List<User> users = null;
       mTwitter.generateUsers(999);
       mTwitter.setMethodCallCount("getFriendsStatuses");

       try {
           users = mTwitterCommunication.getFriendsList();
           assertNotNull(users);
           assertTrue("Should contain content", users.size() == 999);
       } catch (Exception e) {e.printStackTrace();
           exceptionThrown = true;
       }
       assertFalse(exceptionThrown);
       assertEquals("method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test that it handles FriendsList request with internal error
    */
   public void testGetFriendsListThrow() {
       mTwitter.performThrowExceptionOnCall(true, HttpResponseCode.BAD_REQUEST);
       boolean exceptionThrown = false;

       try {
           mTwitterCommunication.getFriendsList();
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should be thrown", true, exceptionThrown);
   }

   /**
    * Test that it handles FriendsList request with internal null
    */
   public void testGetFriendsListNullParams() {
       mTwitter.returnNullOnCall(true);
       boolean twitterExceptionThrown = false;
       boolean nullPointerExceptionThrown = false;
       List<User> users = null;

       try {
           users = mTwitterCommunication.getFriendsList();
           assertNull(users);
       } catch (TwitterPluginException e) {
           twitterExceptionThrown = true;
       } catch (NullPointerException e){
           nullPointerExceptionThrown = true;
       }
       assertFalse(twitterExceptionThrown);
       assertTrue(nullPointerExceptionThrown);
   }

   /**
    * Test authentication using mocked twitter content returning different values
    */
   public void testAuthenticateTwitter() {
       mTwitter.setMethodCallCount("verifyCredentials");
       boolean exceptionThrown = false;

       // Set twitter as null and set factory instance
       mTwitterFactory.setMockTwitterInstance(mTwitter);

       try {
           ReflectionUtilities.getField(mTwitterCommunication, "mTwitter").set(mTwitterCommunication,
                   null);
       } catch (Exception e) {
           Assert.fail("Could not change the factory field");
       }

       try {
           boolean result = mTwitterCommunication.authenticateTwitterAccount("username", "password");
           assertTrue("Should be authenticated", result);
           assertEquals("Token should exist", true, PreferencesHelper.hasToken(mContext));
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);

       // Check that internal getAuthenticatedTwitter is called when mTwitter is null
       exceptionThrown = false;
       try {
           PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
           mTwitterCommunication.getHomeTimeline();
       } catch (Exception e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown", false, exceptionThrown);

       // Check that verify is called twice,
       assertEquals("method should be called", 2, mTwitter.getMethodCallCount());
   }

   /**
    * Test authentication using mocked twitter content returning different values
    */
   public void testAuthenticateTwitterReturnsMinusOne() {
       boolean exceptionThrown = true;

       // Set twitter as null and set factory instance
       mTwitterFactory.setMockTwitterInstance(mTwitter);
       mTwitterFactory.setThrowsExceptionWithStatusCodeMinusOne();

       try {
           ReflectionUtilities.getField(mTwitterCommunication, "mTwitter").set(mTwitterCommunication,
                   null);
       } catch (Exception e) {
           Assert.fail("Could not change the factory field");
       }

       try {
           mTwitterCommunication.authenticateTwitterAccount("username", "password");
           exceptionThrown = false;
       } catch (TwitterPluginException e) {
           assertEquals("Expected SERVER_UNAVAILABLE when status code is -1", StatusCode.SERVER_UNAVAILABLE, e.getStatusCode());
       }
       assertEquals("Expected an exceptions to be thrown", true, exceptionThrown);
   }

   /**
    * Test authentication using mocked twitter content returning different values
    */
   public void testInvalidAuthenticateTwitter() {
       mTwitter.setMethodCallCount("verifyCredentials");
       boolean exceptionThrown = false;

       // Set twitter as null and set factory instance
       mTwitterFactory.setMockTwitterInstance(null);

       try {
           ReflectionUtilities.getField(mTwitterCommunication, "mTwitter").set(mTwitterCommunication,
                   null);
       } catch (Exception e) {
           Assert.fail("Could not change the factory field");
       }

       try {
           boolean result = mTwitterCommunication.authenticateTwitterAccount(null, "password");
           assertFalse(result);
           result = mTwitterCommunication.authenticateTwitterAccount("", "password");
           assertFalse(result);
           result = mTwitterCommunication.authenticateTwitterAccount("username", null);
           assertFalse(result);
            result = mTwitterCommunication.authenticateTwitterAccount("username", "");
           assertFalse(result);

           assertEquals("Token should not exist", false, PreferencesHelper.hasToken(mContext));
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should not be thrown in authenticateTwitterAccount", false, exceptionThrown);

       // Check that internal getAuthenticatedTwitter is called when mTwitter is null
       exceptionThrown = false;
       try {
           PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, MockTwitter.DEFAULT_PAGING_ID);
           mTwitterCommunication.getHomeTimeline();
       } catch (TwitterPluginException e) {
           exceptionThrown = true;
       }
       assertEquals("Exceptions should be thrown in getHomeTimeline", true, exceptionThrown);

       // Check that verify is called twice,
       assertEquals("method should be called", 0, mTwitter.getMethodCallCount());
   }

   /**
    * Test if twitter communication calls shutdown and perform the clear content
    */
   public void testLogoutTwitter() {
       PreferencesHelper.setToken(mContext, "key", "secret");
       mTwitter.setMethodCallCount("shutdown");
       mTwitterCommunication.logout();
       assertEquals("Shutdown method should be called", 1, mTwitter.getMethodCallCount());
   }

   /**
    * Test if twitter communication calls shutdown and perform the clear content
    */
   public void testShutdownTwitter() {
       PreferencesHelper.setToken(mContext, "key", "secret");
       mTwitter.setMethodCallCount("shutdown");
       mTwitterCommunication.cleanup();
       assertEquals("Shutdown method should be called", 1, mTwitter.getMethodCallCount());
   }
}
