/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin.util;

import android.util.Config;
import android.util.Log;
import com.sonyericsson.eventstream.facebookplugin.Facebook;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import org.apache.http.client.ClientProtocolException;

/**
 * Class for mocking the facebook communication class
 * Allows us to simulate any server/http response to any
 * type of request
 */
public class MockCommunication extends Facebook.Communication {

    private ArrayList<Data> mServerResults = new ArrayList<Data>();

    private int mIndex = 0;

    private boolean mEnableDefault;

    private static final int defaultCode = 200;

    private static final String defaultPhrase = "OK";

    private static final String defaultContent = "{ \"data\": [] }";

    /**
     *  Container of data associated with a (server-)request
     */
    private class Data {
        public int mHttpResponseCode = 200;
        public String mHttpPhrase = "";
        public String mContent = "";
        public Exception mException = null;

        private Data(int httpCode, String httpPhrase, String content, Exception exception) {
            mHttpResponseCode = httpCode;
            mHttpPhrase = httpPhrase;
            mContent = content;
            mException = exception;
        }
    }

    public void allowDefaultResponse(boolean enable) {
        mEnableDefault = enable;
    }

    public MockCommunication(int code, String phrase, String content) {
        mServerResults.add(new Data(code, phrase, content, null));
    }

    public MockCommunication(Exception exception) {
        mServerResults.add(new Data(-1, null, null, exception));
    }

    public MockCommunication(int[] codes, String[] phrases, String[] content, Exception[] exception) {
        for (int index = 0; index < codes.length; index++) {
            if (exception == null) {
                mServerResults.add(new Data(codes[index], phrases[index], content[index], null));
            } else {
                mServerResults.add(new Data(codes[index], phrases[index], content[index], exception[index]));
            }
        }
    }

    public MockCommunication(int[] codes, String[] phrases, String[] content) {
        this(codes, phrases, content, null);
    }

    @Override
    public CommunicationResult executePost(String uri) throws ClientProtocolException, IOException {
        try {
            URL myURL = new URL(uri);
            myURL.toURI();    //will throw exception if a malformed uri is used
        } catch (URISyntaxException ex) {
            if (Config.DEBUG) {
                Log.d("MockCommunication", "executePost URISyntaxException = " + ex.toString());
            }
            //We can't throw URISyntaxException so let's throw ClientProtocolException instead
            ClientProtocolException throwThis = new ClientProtocolException("Malformed URI = " + uri);
            throw throwThis;
        }
        return execute(uri);
    }

    @Override
    public CommunicationResult executeGet(String uri) throws ClientProtocolException, IOException {
        try {
            URL myURL = new URL(uri);
            myURL.toURI();    //will throw exception if a malformed uri is used
        } catch (URISyntaxException ex) {
            if (Config.DEBUG) {
                Log.d("MockCommunication", "executeGet URISyntaxException = " + ex.toString());
            }
            //We can't throw URISyntaxException so let's throw ClientProtocolException instead
            ClientProtocolException throwThis = new ClientProtocolException("Malformed URI = " + uri);
            throw throwThis;
        }
        return execute(uri);
    }

    private CommunicationResult execute(String uri) throws ClientProtocolException, IOException {
        if (mEnableDefault && mIndex >= mServerResults.size()) {
            return new CommunicationResult(defaultCode, defaultPhrase, defaultContent);
        }
        Data data = mServerResults.get(mIndex);

        // Increase index...
        mIndex++;

        // Act upon data...
        if (data.mException != null) {
            if (data.mException instanceof ClientProtocolException) {
                throw (ClientProtocolException) data.mException;
            } else if (data.mException instanceof IOException) {
                throw (IOException) data.mException;
            } else {
                return null;
            }
        } else {
            return new CommunicationResult(data.mHttpResponseCode, data.mHttpPhrase, data.mContent);
        }
    }

    @Override
    public void closeConnection() {
        // Do nothing!
    }

    @Override
    public void shutdown() {
        // Do nothing!
    }
}