/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.facebookplugin;

import android.app.Application;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Intent;
import android.util.Log;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;

import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY_PARAMETER;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Erik Hellman <erik.hellman@sonyericsson.com>
 */
public class FacebookPluginApplication extends Application {

    public enum State {
        NOT_CONFIGURED,
        AUTHENTICATION_IN_PROGRESS,
        AUTHENTICATION_FAILED,
        AUTHENTICATED,
        AUTHENTICATION_BAD_CREDENTIALS,
        STATUS_UPDATE_FAILED
    };

    private static final String LOG_TAG = "FacebookPluginApplication";

    private static final int UPDATE_NOTIFICATION = 1;

    private static final int AUTH_FAILED_NOTIFICATION = 2;

    private static final int STATUS_UPDATE_FAILED = 3;

    private NotificationHandler mNotificationHandler;

    private CleanUpHandler mCleanupHandler;

    private List<StateListener> mStateListener;

    /** The current state */
    private State mState = State.NOT_CONFIGURED;

    /**
     * Interface for the state listener
     */
    public interface StateListener {
        /**
         * Indicate that the current state has changed
         *
         * @param oldState The old state
         * @param newState The new changed state
         */
        void onStateChange(final FacebookNotification newState);
    }

    @Override
    public void onCreate() {
        super.onCreate();

        Facebook facebook = null;

        mNotificationHandler = new NotificationHandler();
        mCleanupHandler = new CleanUpHandler();
        facebook = FacebookFactory.getFacebook(getApplicationContext());
        facebook.setServiceStateListener(mNotificationHandler);
        facebook.setServiceStateListener(mCleanupHandler);

        synchronized (this) {
            mStateListener = Collections.synchronizedList(new ArrayList<StateListener>());
        }

        if (facebook.isLoggedIn()) {
            setState(State.AUTHENTICATED);
        }
    }

    @Override
    public void onTerminate() {
        super.onTerminate();

        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());

        facebook.removeServiceStateListener(mNotificationHandler);
        facebook.removeServiceStateListener(mCleanupHandler);

        FacebookFactory.terminate(false);
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();

        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());
        facebook.close();
    }

    public synchronized void setState(State newState) {
        FacebookNotification state = new FacebookNotification(newState, null);

        setState(state);
    }

    public synchronized void setState(FacebookNotification notification) {
        mState = notification.getState();

        if (mStateListener != null) {
            if (mStateListener.isEmpty()) {
                if (notification.getState() == State.AUTHENTICATION_FAILED
                        || notification.getState() == State.AUTHENTICATION_BAD_CREDENTIALS
                        || notification.getState() == State.STATUS_UPDATE_FAILED) {
                    fireNotification(notification.getState());
                }
            } else {
                for (StateListener listener : mStateListener) {
                    listener.onStateChange(notification);
                }
            }
        }
    }

    public synchronized State getState() {
        return mState;
    }

    public synchronized void addStateListener(StateListener listener) {
        if (listener != null) {
            mStateListener.add(listener);
        }
    }

    public synchronized void removeStateListener(StateListener listener) {
        if (listener != null) {
            mStateListener.remove(listener);
        }
    }

    public void cancelNotifications() {
        NotificationManager nm = (NotificationManager)getSystemService(NOTIFICATION_SERVICE);
        nm.cancelAll();
    }

    /**
     * Clear all data associated with facebook.
     */
    public void clearFacebookData() {
        // Clear data from event stream framework
        new Database(getApplicationContext()).clearData();

        // Clear "local"/internal data
        new Settings(getApplicationContext()).removeSettings();
    }

    public class NotificationHandler implements Facebook.ServiceStateChangeListener {

        public void onServiceStateChanged(Facebook.ServiceState oldState,
                Facebook.ServiceState newState) {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Plugin application onservice state change:" + oldState + ":" + newState);
            }

            if (newState == Facebook.ServiceState.SERVER_OPERATION_IN_PROGRESS) {
                NotificationManager nm = (NotificationManager)getSystemService(NOTIFICATION_SERVICE);
                Notification notification = new Notification(
                        R.drawable.notif_data_fetch_ongoing_icn, null, System.currentTimeMillis());
                PendingIntent pendingIntent = PendingIntent.getActivity(
                        FacebookPluginApplication.this, 0, null, Intent.FLAG_ACTIVITY_NEW_TASK);
                notification.setLatestEventInfo(FacebookPluginApplication.this, getResources()
                        .getText(R.string.ts_facebook_update_notification_title), getResources()
                        .getText(R.string.ts_facebook_update_notification_message), pendingIntent);
                nm.notify(UPDATE_NOTIFICATION, notification);
            } else if (oldState == Facebook.ServiceState.SERVER_OPERATION_IN_PROGRESS
                    && newState == Facebook.ServiceState.PASSIVE) {
                NotificationManager nm = (NotificationManager)getSystemService(NOTIFICATION_SERVICE);
                nm.cancel(UPDATE_NOTIFICATION);
            }
        }
    }

    /**
     * Do we, due to the 30 day inactivity requirement, need to delete all
     * facebook data?
     *
     * This callback will be called when the service changes state. When before
     * communicating with the server, the app needs to clean up data if there
     * hasn't been any communication towards the server for 30 days.
     */
    public class CleanUpHandler implements Facebook.ServiceStateChangeListener {
        /**
         * Thirty (30) days in milliseconds
         */
        private static final long ACCESS_30_DAYS = 30 * 24 * 60 * 60 * 1000L;

        @Override
        public void onServiceStateChanged(Facebook.ServiceState oldState,
                Facebook.ServiceState newState) {
            // If we haven't communicated with the server for the last 30 days,
            // we have to
            // delete all facebook data
            if (newState == Facebook.ServiceState.AUTHENTICATION_IN_PROGRESS
                    || newState == Facebook.ServiceState.SERVER_OPERATION_IN_PROGRESS) {
                final long currentTime = System.currentTimeMillis();

                if (isCleanupNeeded(currentTime)) {
                    clearFacebookData();
                }

                storeLastCommuncationTime(currentTime);
            }
        }

        /**
         * Store the current time (in milliseconds since epoch).
         */
        private void storeLastCommuncationTime(final long currentTime) {
            new Settings(getApplicationContext()).setLastCommunicationTime(currentTime);
        }

        /**
         * Get the last time we communicated with the server
         *
         * @return the time, in milliseconds since epoch
         */
        private long getLastCommunicationTime() {
            return new Settings(getApplicationContext()).getLastCommunicationTime();
        }

        /**
         * Do we, due to the 30 day inactivity requirement, need to remove all
         * facebook data?
         *
         * @return true if we have exceeded 30 days of inactivity, false otherwise
         */
        protected boolean isCleanupNeeded(final long currentTime) {
            final long lastAccess = getLastCommunicationTime();
            final boolean result = (lastAccess != 0 &&
                                    lastAccess <= currentTime &&
                                    ((lastAccess + ACCESS_30_DAYS) <= currentTime));

            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Facebook clean up; lastAccess:" + lastAccess + " currentTime:"
                        + currentTime + " result:" + result);
            }

            return result;
        }
    }

    public synchronized void fireNotification(State state) {
        NotificationManager nm = (NotificationManager)getSystemService(NOTIFICATION_SERVICE);
        Notification notif = new Notification(R.drawable.notif_data_fetch_failed_icn, null, System
                .currentTimeMillis());

        if (state == State.AUTHENTICATION_FAILED || state == State.AUTHENTICATION_BAD_CREDENTIALS) {
            Intent intent = new Intent();
            intent.setComponent(new ComponentName(this, FacebookPluginConfig.class));
            PendingIntent pendingIntent = PendingIntent.getActivity(this, 0, intent,
                    Intent.FLAG_ACTIVITY_NEW_TASK);
            notif.setLatestEventInfo(this, getResources().getText(
                    R.string.ts_facebook_auth_failed_notification_title), getResources().getText(
                    R.string.ts_facebook_auth_failed_notification_message), pendingIntent);
            nm.notify(AUTH_FAILED_NOTIFICATION, notif);
            // Clear all data and change state to logged out
            Intent logoutIntent = new Intent(Constants.LOGOUT_INTENT);

            logoutIntent.setClass(getApplicationContext(), FacebookService.class);
            logoutIntent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
            startService(logoutIntent);
        } else if (state == State.STATUS_UPDATE_FAILED) {
            PendingIntent pendingIntent = PendingIntent.getActivity(this, 0, null, 0);
            notif.setLatestEventInfo(this, getResources().getString(
                    R.string.ts_facebook_update_failed_notification_title), getResources()
                    .getString(R.string.ts_facebook_connection_failed), pendingIntent);
            nm.notify(STATUS_UPDATE_FAILED, notif);
        }
    }

    public synchronized void clearNotification() {
        NotificationManager nm = (NotificationManager)getSystemService(NOTIFICATION_SERVICE);

        nm.cancel(STATUS_UPDATE_FAILED);
        nm.cancel(AUTH_FAILED_NOTIFICATION);
    }
}
