/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.view;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants;
import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.InternalIntentExtraParams;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication.State;
import com.sonyericsson.eventstream.twitterplugin.utility.NetworkHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.NotificationHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.DialogInterface.OnCancelListener;
import android.content.DialogInterface.OnClickListener;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.widget.EditText;
import android.widget.TextView;

public class ConfigurationActivity extends Activity implements TwitterPluginApplication.Listener,
 DialogInterface.OnCancelListener, DialogInterface.OnClickListener {

    AlertDialog mLoginDialog;

    AlertDialog mDisclaimerDialog;

    ProgressDialog mProgressDialog;

    static final int DIALOG_ERROR_INVALID_ACCOUNT_ID = 1;
    static final int DIALOG_ERROR_MISSING_USERNAME_ID = 2;
    static final int DIALOG_ERROR_MISSING_PASSWORD_ID = 3;
    static final int DIALOG_ERROR_NETWORK_FAILED_ID = 4;
    static final int DIALOG_LOGOUT_ID = 5;

    /** constant for savedInstanceState bundle */
    private static final String BUNDLE_LOGIN_USERNAME = "login_username";

    private String mCachedUsername;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_NO_TITLE);

        // Check if the user hasn't accepted the disclaimer
        if (!PreferencesHelper.hasAcceptedDisclaimer(this)) {
            setState(State.ACTIVITY_SHOW_DISCLAIMER);
        }

        if (savedInstanceState != null &&
                getState() == State.SERVICE_NOT_AUTHENTICATED) {
            // restore saved username
            mCachedUsername = savedInstanceState.getString(BUNDLE_LOGIN_USERNAME);
            EditText usernameField = ((EditText)mLoginDialog.findViewById(R.id.login_username));
            if (usernameField != null && mCachedUsername != null) {
                usernameField.setText(mCachedUsername);
            }
        }
    }

    @Override
    protected void onResume() {
        super.onResume();

        ((TwitterPluginApplication) getApplication()).setConfigurationActivityVisible(true);
        ((TwitterPluginApplication) getApplication()).addListener(this);

        switch(getState()) {
            case ACTIVITY_SHOW_DISCLAIMER:
                setupDisclaimerDialog();
                break;

            case SERVICE_NOT_CONFIGURED:
                setState(State.SERVICE_NOT_AUTHENTICATED);
                break;

            case SERVICE_AUTHENTICATION_FAILED:
                setState(State.SERVICE_NOT_AUTHENTICATED);
                break;

            case SERVICE_AUTHENTICATION_IN_PROGRESS:
                showProgressDialog();
                break;

            case SERVICE_AUTHENTICATION_SUCCESS:
                setState(State.SERVICE_AUTHENTICATED);
                break;

            case SERVICE_AUTHENTICATED:
                // If activity is started and already authenticated perform logout
                showDialog(DIALOG_LOGOUT_ID);
                break;

            case SERVICE_NOT_AUTHENTICATED:
                setupLoginDialog();
                break;

            default:
                break;
        }
    }

    /**
     * This is override from the system, however removed the override
     * since the pc tests complained.
     */
    protected Dialog onCreateDialog(int id, Bundle args) {
        switch (id) {
            case DIALOG_ERROR_INVALID_ACCOUNT_ID:
                return createErrorDialog(R.string.ts_twitter_login_account_failed_txt);

            case DIALOG_ERROR_MISSING_PASSWORD_ID:
                return createErrorDialog(R.string.ts_twitter_login_no_password_txt);

            case DIALOG_ERROR_MISSING_USERNAME_ID:
                return createErrorDialog(R.string.ts_twitter_login_no_user_name_txt);

            case DIALOG_ERROR_NETWORK_FAILED_ID:
                return createErrorDialog(R.string.ts_twitter_login_network_failed_txt);

            case DIALOG_LOGOUT_ID:
                return createLogoutDialog();

            default:
                throw new IllegalArgumentException("unknown dialog");
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        if (mLoginDialog != null) {
            mLoginDialog.dismiss();
        }
        mLoginDialog = null;

        if (mDisclaimerDialog != null) {
            mDisclaimerDialog.dismiss();
        }
        mDisclaimerDialog = null;
    }

    @Override
    protected void onPause() {
        super.onPause();
        ((TwitterPluginApplication) getApplication()).setConfigurationActivityVisible(false);
        ((TwitterPluginApplication) getApplication()).removeListener(this);
        if (mProgressDialog != null) {
            mProgressDialog.dismiss();
        }
        mProgressDialog = null;
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        if (getState() == State.SERVICE_NOT_AUTHENTICATED && mLoginDialog != null) {
            // save username
            mCachedUsername = ((EditText)mLoginDialog.findViewById(R.id.login_username)).getText().toString();
            if (mCachedUsername != null && mCachedUsername.length() > 0) {
                outState.putString(BUNDLE_LOGIN_USERNAME, mCachedUsername);
            }
        }
    }

    public void onStateChange(TwitterPluginApplication.State oldState, final TwitterPluginApplication.State newState) {
        Runnable runnable = new Runnable() {
            public void run() {
                if (!isFinishing()) {
                    switch(newState) {
                        case SERVICE_NOT_AUTHENTICATED:
                            hideProgressDialog();
                            setupLoginDialog();
                            break;

                        case SERVICE_AUTHENTICATION_IN_PROGRESS:
                            showProgressDialog();
                            break;

                        case SERVICE_AUTHENTICATION_FAILED:
                            setState(State.ACTIVITY_INVALID_ACCOUNT);
                            break;

                        case SERVICE_AUTHENTICATION_SUCCESS:
                            setState(State.SERVICE_AUTHENTICATED);
                            break;

                        case SERVICE_AUTHENTICATED:
                            hideProgressDialog();
                            finish();
                            break;

                        case ACTIVITY_INVALID_ACCOUNT:
                            hideProgressDialog();
                            showDialog(DIALOG_ERROR_INVALID_ACCOUNT_ID);
                            break;

                        case ACTIVITY_MISSING_USERNAME:
                            hideProgressDialog();
                            showDialog(DIALOG_ERROR_MISSING_USERNAME_ID);
                            break;

                        case ACTIVITY_MISSING_PASSWORD:
                            hideProgressDialog();
                            showDialog(DIALOG_ERROR_MISSING_PASSWORD_ID);
                            break;

                        case ACTIVITY_NETWORK_FAILED:
                            hideProgressDialog();
                            showDialog(DIALOG_ERROR_NETWORK_FAILED_ID);
                            break;

                        case ACTIVITY_CREATE_ACCOUNT:
                            Intent intent = new Intent(Intent.ACTION_VIEW,
                                    Uri.parse(TwitterConf.CREATE_ACCOUNT_URL));
                            // launch activity
                            if (intent != null) {
                                try {
                                    startActivity(intent);
                                } catch (ActivityNotFoundException e) {
                                    Log.e(Config.LOG_TAG, "Failed to launch register account activity");
                                }
                            }
                            setState(State.SERVICE_NOT_CONFIGURED);
                            break;
                    }
                }
            }
        };
        runOnUiThread(runnable);
    }

    private void showProgressDialog() {
        if (mProgressDialog == null)  {
            mProgressDialog = new ProgressDialog(this);
            mProgressDialog.setMessage(getResources().getText(
                    R.string.ts_twitter_authenticating_txt));
            mProgressDialog.setCancelable(false);
        }
        mProgressDialog.show();
    }

    private void hideProgressDialog() {
        if (mProgressDialog != null)  {
            mProgressDialog.hide();
        }
    }

    private void setupLoginDialog() {
        NotificationHelper.removeFromStatusBar(this);

        if (mLoginDialog == null ||
            (mLoginDialog != null && !mLoginDialog.isShowing())) {
            mLoginDialog = createLoginDialog();
            mLoginDialog.show();
        }
        // set cached username and reset the password
        EditText editText = ((EditText)mLoginDialog.findViewById(R.id.login_username));
        editText.setText(mCachedUsername);
        editText = ((EditText)mLoginDialog.findViewById(R.id.login_password));
        editText.setText("");
    }

    private void setupDisclaimerDialog() {
        if (mDisclaimerDialog == null ||
            (mDisclaimerDialog != null && !mDisclaimerDialog.isShowing())) {
            mDisclaimerDialog = createDisclaimerDialog();
            mDisclaimerDialog.show();
        }
    }

    private AlertDialog createDisclaimerDialog() {
        AlertDialog.Builder disclaimerDialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        String serviceName = getResources().getString(R.string.ts_twitter_name_txt);
        String disclaimer = getResources().getString(R.string.ts_twitter_disclamer_message_txt,
                serviceName, serviceName);

        View disclaimerView = getLayoutInflater().inflate(R.layout.disclaimer, null);
        TextView message = (TextView) disclaimerView.findViewById(R.id.disclaimerText);
        message.setText(disclaimer);

        disclaimerDialog.setTitle(R.string.ts_twitter_disclamer_title_txt);
        disclaimerDialog.setIcon(R.drawable.twitter_popup_title_icn);
        disclaimerDialog.setView(disclaimerView);
        disclaimerDialog.setPositiveButton(R.string.gui_accept_terms_txt, new OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                setResult(RESULT_OK);
                setState(State.SERVICE_NOT_AUTHENTICATED);
            }
        });
        disclaimerDialog.setNegativeButton(R.string.gui_cancel_txt, new OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                setResult(RESULT_OK);
                finish();
            }
        });
        disclaimerDialog.setOnCancelListener(new OnCancelListener() {
            public void onCancel(DialogInterface dialog) {
                setResult(RESULT_OK);
                finish();
            }
        });
        return disclaimerDialog.create();
    }

    private AlertDialog createLoginDialog() {
        AlertDialog.Builder loginDialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        LayoutInflater inflater = LayoutInflater.from(this);
        View loginView = inflater.inflate(R.layout.login, null);
        loginDialog.setView(loginView);
        loginDialog.setTitle(R.string.ts_twitter_login_title_txt);
        loginDialog.setIcon(R.drawable.twitter_popup_title_icn);
        loginDialog.setPositiveButton(R.string.gui_sign_in_txt, this);
        loginDialog.setNegativeButton(R.string.gui_cancel_txt, this);
        loginDialog.setOnCancelListener(this);
        return loginDialog.create();
    }

    private Dialog createErrorDialog(int message) {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        dialog.setIcon(android.R.drawable.ic_dialog_alert);
        dialog.setTitle(R.string.ts_twitter_login_authentication_failed_title_txt);
        dialog.setMessage(message);
        dialog.setNegativeButton(R.string.gui_ok_txt, new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialogInterface, int whichButton) {
                setState(State.SERVICE_NOT_AUTHENTICATED);
            }
        });
        dialog.setOnCancelListener(this);
        return dialog.create();
    }

    private Dialog createLogoutDialog() {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        dialog.setTitle(R.string.ts_twitter_logout_dialog_title_txt);
        dialog.setMessage(R.string.ts_twitter_logout_dialog_text_txt);
        dialog.setIcon(android.R.drawable.ic_dialog_info);
        dialog.setPositiveButton(R.string.gui_ok_txt, new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialogInterface, int whichButton) {
                Intent logoutIntent = new Intent(PluginConstants.TWITTER_LOGOUT);
                startService(logoutIntent);
                finish();
            }
        });
        dialog.setNegativeButton(R.string.gui_cancel_txt, new OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                setResult(RESULT_OK);
                finish();
            }
        });
        dialog.setOnCancelListener(new OnCancelListener() {
            public void onCancel(DialogInterface dialog) {
                setResult(RESULT_OK);
                finish();
            }
        });
        return dialog.create();
    }

    private String validateUserName() {
        mCachedUsername = ((EditText)mLoginDialog.findViewById(R.id.login_username)).getText().toString();
        if (mCachedUsername.length() == 0) {
            setState(State.ACTIVITY_MISSING_USERNAME);
            return null;
        }
        return mCachedUsername;
    }

    private String validatePassword() {
        String password = ((EditText)mLoginDialog.findViewById(R.id.login_password)).getText().toString();
        if (password.length() == 0) {
            setState(State.ACTIVITY_MISSING_PASSWORD);
            return null;
        }
        return password;
    }

    private void callLogin(Context context) {
        String username = validateUserName();
        if (username == null) {
            return;
        }
        String password = validatePassword();
        if (password == null) {
            return;
        }
        if (!NetworkHelper.isNetworkAvailable(context)) {
            setState(State.ACTIVITY_NETWORK_FAILED);
            return;
        }
        Intent loginIntent = new Intent(PluginConstants.TWITTER_AUTHENTICATE);
        loginIntent.putExtra(InternalIntentExtraParams.USERNAME, username);
        loginIntent.putExtra(InternalIntentExtraParams.PASSWORD, password);

        context.startService(loginIntent);
    }

    public void onClick(DialogInterface dialogInterface, int whichButton) {
        switch(whichButton) {
            case DialogInterface.BUTTON_POSITIVE:
                callLogin(this);
                break;
            case DialogInterface.BUTTON_NEGATIVE:
                callCancel();
                break;
        }
    }

    public void onCancel(DialogInterface dialog) {
        callCancel();
    }

    public void callCreateAccount(View view) {
        setState(State.ACTIVITY_CREATE_ACCOUNT);
    }

    private void callCancel() {
        setState(State.SERVICE_NOT_AUTHENTICATED);
        finish();
    }

    private void setState(State state) {
        ((TwitterPluginApplication) getApplication()).setState(state);
    }

    private State getState() {
        return ((TwitterPluginApplication) getApplication()).getState();
    }
}
