/*
 * Copyright (C) 2010 Sony Ericsson Mobile Communications AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 *
 */

package com.sonyericsson.eventstream.rssplugin.db;

import com.sonyericsson.eventstream.rssplugin.PluginConstants.Config;

import android.content.ContentProvider;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;


/**
 * RssFeedProvider.
 */
public class RssFeedProvider extends ContentProvider {

    /** maximum number of channels. */
    public static final int MAX_CHANNEL_ROWS = 30;

    /** maximum number of items. */
    public static final int MAX_ITEM_ROWS = 500;

    /** channel table name */
    public static final String CHANNEL_TABLE_NAME = "channels";

    /** channel id column name */
    public static final String CHANNEL_ID_COLUMN = "_id";

    /** channel url column name */
    public static final String CHANNEL_URL_COLUMN = "url";

    /** item table name */
    public static final String ITEM_TABLE_NAME = "items";

    /** item id column name */
    public static final String ITEM_ID_COLUMN = "_id";

    /** channel id column name */
    public static final String ITEM_CHANNEL_ID_COLUMN = "channel_id";

    /** item url column name */
    public static final String ITEM_URL_COLUMN = "url";

    private static final String DATABASE_NAME = "rssfeed.db";
    private static final int DATABASE_VERSION = 1;
    private static final String AUTHORITY = "com.sonyericsson.eventstream.rssplugin.db";
    private static final String CONTENT_URI_BASE = "content://" + AUTHORITY + "/";
    public static final Uri CHANNEL_CONTENT_URI = Uri.parse(CONTENT_URI_BASE + CHANNEL_TABLE_NAME);
    public static final Uri ITEM_CONTENT_URI = Uri.parse(CONTENT_URI_BASE + ITEM_TABLE_NAME);

    private SQLiteOpenHelper mOpenHelper;

    /**
     * DatabaseHelper.
     */
    private static class DatabaseHelper extends SQLiteOpenHelper {

        DatabaseHelper(Context context) {
            super(context, DATABASE_NAME, null, DATABASE_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            db.execSQL("CREATE TABLE  IF NOT EXISTS " + CHANNEL_TABLE_NAME + " ("
                    + CHANNEL_ID_COLUMN + " INTEGER PRIMARY KEY,"
                    + CHANNEL_URL_COLUMN + " TEXT UNIQUE"
                    + ");");
            db.execSQL("CREATE TABLE  IF NOT EXISTS " + ITEM_TABLE_NAME + " ("
                    + ITEM_ID_COLUMN + " INTEGER PRIMARY KEY,"
                    + ITEM_CHANNEL_ID_COLUMN + " INTEGER,"
                    + ITEM_URL_COLUMN + " TEXT UNIQUE"
                    + ");");
        }

        @Override
        public void onUpgrade(
                SQLiteDatabase db,
                int oldVersion, int newVersion) {
            db.execSQL("DROP TABLE IF EXISTS " + CHANNEL_TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + ITEM_TABLE_NAME);
            onCreate(db);
        }
    }

    @Override
    public boolean onCreate() {
        mOpenHelper = new DatabaseHelper(getContext());
        return true;
    }

    @Override
    public Cursor query(Uri uri,
            String[] projection, String selection,
            String[] selectionArgs, String sortOrder) {
        SQLiteDatabase db = mOpenHelper.getReadableDatabase();
        Cursor cursor = null;
        String table = uri.getPathSegments().get(0);
        if (CHANNEL_TABLE_NAME.equals(table)) {
            cursor = db.query(CHANNEL_TABLE_NAME, projection, selection, selectionArgs,
            null, null, sortOrder);
        } else if (ITEM_TABLE_NAME.equals(table)) {
            cursor = db.query(ITEM_TABLE_NAME, projection, selection, selectionArgs,
            null, null, sortOrder);
        }

        if (cursor != null) {
            cursor.setNotificationUri(getContext().getContentResolver(), uri);
        }
        return cursor;
    }

    @Override
    public String getType(Uri uri) {
        return null;
    }

    @Override
    public Uri insert(Uri uri, ContentValues initialValues) {
        ContentValues values;
        if (null != initialValues) {
            values = new ContentValues(initialValues);
        } else {
            values = new ContentValues();
        }

        Uri returnUri = null;
        String table = uri.getPathSegments().get(0);
        if (CHANNEL_TABLE_NAME.equals(table)) {
            returnUri = insertChannel(uri, values);
        } else if (ITEM_TABLE_NAME.equals(table)) {
            returnUri = insertItem(uri, values);
        }
        return returnUri;
    }

    @Override
    public int delete(Uri uri,
            String where, String[] whereArgs) {
        SQLiteDatabase db = mOpenHelper.getWritableDatabase();

        int count = 0;
        String table = uri.getPathSegments().get(0);
        if (CHANNEL_TABLE_NAME.equals(table)) {
            count = db.delete(CHANNEL_TABLE_NAME, where, whereArgs);
        } else if (ITEM_TABLE_NAME.equals(table)) {
            count = db.delete(ITEM_TABLE_NAME, where, whereArgs);
        }

        getContext().getContentResolver().notifyChange(uri, null);
        return count;
    }

    @Override
    public int update(Uri uri,
            ContentValues values,
            String where, String[] whereArgs) {
        SQLiteDatabase db = mOpenHelper.getWritableDatabase();

        int count = 0;
        String table = uri.getPathSegments().get(0);
        if (CHANNEL_TABLE_NAME.equals(table)) {
            count = db.update(CHANNEL_TABLE_NAME, values,
                    CHANNEL_ID_COLUMN + "=" + uri.getPathSegments().get(1)
                    + (!TextUtils.isEmpty(where)
                            ? " AND (" + where + ")" : ""), whereArgs);
        } else if (ITEM_TABLE_NAME.equals(table)) {
            count = db.update(ITEM_TABLE_NAME, values,
                    ITEM_ID_COLUMN + "=" + uri.getPathSegments().get(1)
                    + (!TextUtils.isEmpty(where)
                            ? " AND (" + where + ")" : ""), whereArgs);
        }

        getContext().getContentResolver().notifyChange(uri, null);
        return count;
    }

    private Uri insertChannel(Uri uri, ContentValues values) {
        Uri channelUri = null;
        Cursor cursor = query(uri, new String[]{CHANNEL_ID_COLUMN},
                null, null, null);
        int count = cursor.getCount();
        cursor.close();
        if (count < MAX_CHANNEL_ROWS) {
            SQLiteDatabase db = mOpenHelper.getWritableDatabase();
            long rowId = db.insert(CHANNEL_TABLE_NAME, null, values);
            if (0 < rowId) {
                db.update(CHANNEL_TABLE_NAME, values,
                        CHANNEL_ID_COLUMN + "=" + rowId, null);

                channelUri = ContentUris.withAppendedId(
                        CHANNEL_CONTENT_URI, rowId);
                getContext().getContentResolver().notifyChange(channelUri, null);
            }
        } else {
            if(Config.DEBUG) {
                Log.d(Config.LOG_TAG, "channel number limit exceeded.");
            }
        }
        return channelUri;
    }

    private Uri insertItem(Uri uri, ContentValues values) {
        Uri itemUri = null;
        Cursor cursor = query(uri, new String[]{ITEM_ID_COLUMN},
                null, null, null);
        int count = cursor.getCount();
        cursor.close();
        if (count < MAX_ITEM_ROWS) {
            SQLiteDatabase db = mOpenHelper.getWritableDatabase();
            long rowId = db.insert(ITEM_TABLE_NAME, null, values);
            if (0 < rowId) {
                db.update(ITEM_TABLE_NAME, values,
                        ITEM_ID_COLUMN + "=" + rowId, null);

                itemUri = ContentUris.withAppendedId(
                        ITEM_CONTENT_URI, rowId);
                getContext().getContentResolver().notifyChange(itemUri, null);
            }
        } else {
            if(Config.DEBUG) {
                Log.d(Config.LOG_TAG, "item number limit exceeded.");
            }
        }
        return itemUri;
    }
}
