/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.service;

import android.app.IntentService;
import android.content.Intent;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.os.Environment;
import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.routine_work.simple_battery_logger.db.BatteryHistoryDBHelper;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.utils.Log;
import org.routine_work.simple_battery_logger.util.PreferenceUtils;

/**
 *
 * @author sawai
 */
public class ExportOneDayDataCsvFileService extends IntentService
	implements DBConstants
{

	public static final String EXTRA_DATE = "date"; // yyyy-MM-dd
	private static final String LOG_TAG = "simple-battery-logger";
	private BatteryHistoryDBHelper dbHelper;

	public ExportOneDayDataCsvFileService(String name)
	{
		super(name);
	}

	public ExportOneDayDataCsvFileService()
	{
		this("ExportOneDayDataCsvFileService");
	}

	@Override
	public void onCreate()
	{
		super.onCreate();
		dbHelper = new BatteryHistoryDBHelper(this);
	}

	@Override
	public void onDestroy()
	{
		super.onDestroy();

		if (dbHelper != null)
			dbHelper.close();
	}

	@Override
	protected void onHandleIntent(Intent intent)
	{
		Log.i(LOG_TAG, "ExportOneDayDataCsvFileService  : Start at " + new Date());

		// init param
		String dateString = intent.getStringExtra(EXTRA_DATE); // yyyy-MM-dd
		if (dateString == null)
		{
			SimpleDateFormat dateFormat = new SimpleDateFormat(DBConstants.DATE_FORMAT);
			dateString = dateFormat.format(new Date());
		}

		// export
		boolean result = exportOneDayDataToCSVFile(dateString);

		// send broadcast intent
		Intent broadcastIntent = new Intent();
		broadcastIntent.setAction(ExportOneDayDataCsvFileCompletedReceiver.EXPORT_ONE_DAY_DATA_CSV_FILE_COMPLETED_ACTION);
		broadcastIntent.putExtra(ExportOneDayDataCsvFileCompletedReceiver.EXTRA_RESULT, result);
		sendBroadcast(broadcastIntent);

		Log.i(LOG_TAG, "ExportOneDayDataCsvFileService : End at " + new Date());
	}

	private boolean exportOneDayDataToCSVFile(String dateString)
	{
		boolean result = false;
		Log.v(LOG_TAG, "Hello");

		String externalStorageState = Environment.getExternalStorageState();
		if (externalStorageState.equalsIgnoreCase(Environment.MEDIA_MOUNTED))
		{
			String csvExportDirectoryName = PreferenceUtils.getCsvExportDirectoryName(this);
			Log.i(LOG_TAG, "csvExportDirectoryName => " + csvExportDirectoryName);

			String csvSortOrder = PreferenceUtils.getCsvSortOrder(this);
			Log.i(LOG_TAG, "csvSortOrder => " + csvSortOrder);

			// init exportDir
			File externalStorageDirectory = Environment.getExternalStorageDirectory();
			File exportDir = new File(externalStorageDirectory, csvExportDirectoryName);
			if (exportDir.exists() == false)
			{
				boolean rc = exportDir.mkdirs();
				Log.d(LOG_TAG, exportDir + "mkdirs() : rc => " + rc);
			}

			// init input db
			SQLiteDatabase db = dbHelper.getReadableDatabase();
			CsvFileExporter csvFileExporter = new CsvFileExporter(this);
			try
			{
				// export one day data
				try
				{
					String fileName = dateString + CsvFileExporter.CSV_FILE_SUFFIX;
					File outputFile = new File(exportDir, fileName);
					Cursor cursor = dbHelper.getOneDayBatteryHistoryCursor(db, dateString, csvSortOrder);
					try
					{
						csvFileExporter.export(outputFile, cursor);
						result = true;
					}
					finally
					{
						cursor.close();
					}
				}
				catch (ParseException ex)
				{
					Log.e(LOG_TAG, "Failed in exportCSV. "
						+ ": exportDir => " + exportDir
						+ ", dateString => " + dateString, ex);
				}
				catch (IOException ex)
				{
					Log.e(LOG_TAG, "Failed in exportCSV. "
						+ ": exportDir => " + exportDir
						+ ", dateString => " + dateString, ex);
				}
			}
			finally
			{
				// quit input db
				db.close();
			}
		}
		else
		{
			Log.i(LOG_TAG, "ExportOneDayDataCsvFileService : CSV export is canceled. SD card is not mounted.");
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}
}
