// -*- C++ -*-
/*!
 * @file PeriodicExecutionContext.h
 * @brief PeriodicExecutionContext class
 * @date $Date: 2008-01-14 07:53:05 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: PeriodicExecutionContext.h 1806 2010-01-27 06:18:28Z fsi-katami $
 *
 */

#ifndef RTC_PERIODICEXECUTIONCONTEXT_H
#define RTC_PERIODICEXECUTIONCONTEXT_H

#include <coil/Task.h>
#include <coil/Mutex.h>
#include <coil/Condition.h>
#include <vector>
#include <iostream>

#include <rtm/RTC.h>
#include <rtm/idl/RTCSkel.h>
#include <rtm/idl/OpenRTMSkel.h>
#include <rtm/Manager.h>
#include <rtm/StateMachine.h>
#include <rtm/ExecutionContextBase.h>

// ACE
#define NUM_OF_LIFECYCLESTATE 4

#ifdef WIN32
#pragma warning( disable : 4290 )
#endif

namespace RTC
{
  /*!
   * @if jp
   * @class PeriodicExecutionContext
   * @brief PeriodicExecutionContext 饹
   *
   * Periodic Sampled Data Processing(¹)ExecutionContext饹
   *
   * @since 0.4.0
   *
   * @else
   * @class PeriodicExecutionContext
   * @brief PeriodicExecutionContext class
   *
   * Periodic Sampled Data Processing (for the execution cycles)
   * ExecutionContext class
   *
   * @since 0.4.0
   *
   * @endif
   */
  class PeriodicExecutionContext
    : public virtual ExecutionContextBase,
      public coil::Task
  {
  public:
    /*!
     * @if jp
     * @brief ǥեȥ󥹥ȥ饯
     *
     * ǥեȥ󥹥ȥ饯
     * ץե˰ʲιܤꤹ롣
     *  - kind : PERIODIC
     *  - rate : 0.0
     *
     * @else
     * @brief Default Constructor
     *
     * Default Constructor
     * Set the following items to profile.
     *  - kind : PERIODIC
     *  - rate : 0.0
     *
     * @endif
     */
    PeriodicExecutionContext();
    
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     * ꤵ줿ͤץեꤹ롣
     *
     * @param owner  Executioncontext  owner
     * @param rate ư(Hz)(ǥե:1000)
     *
     * @else
     * @brief Constructor
     *
     * Constructor
     * Set the configuration value to profile.
     *
     * @param owner The owner of this Executioncontext
     * @param rate Execution cycle(Hz)(The default value:1000)
     *
     * @endif
     */
    PeriodicExecutionContext(OpenRTM::DataFlowComponent_ptr owner,
			     double rate = 1000.0);
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    virtual ~PeriodicExecutionContext(void);
    
    /*!
     * @if jp
     * @brief CORBA ֥ȻȤμ
     *
     * ܥ֥Ȥ ExecutioncontextService ȤƤ CORBA ֥Ȼ
     * 롣
     *
     * @return CORBA ֥Ȼ
     *
     * @else
     * @brief Get the reference to the CORBA object
     *
     * Get the reference to the CORBA object as ExecutioncontextService
     * of this object.
     *
     * @return The reference to CORBA object
     *
     * @endif
     */
    virtual ExecutionContextService_ptr getObjRef(void) {return m_ref;}
    
    /*!
     * @if jp
     * @brief ExecutionContextѥƥӥƥåɤ
     *
     * Executioncontext Ѥƥӥƥåɤư롣
     *  ACE_Task ӥ饹᥽åɤΥС饤ɡ
     *
     * @param args ̾0
     *
     * @return ¹Է
     *
     * @else
     *
     * @brief Generate internal activity thread for ExecutionContext
     *
     * Generate internal activity thread and run. 
     * This is ACE_Task class method's override.
     *
     * @param args Usually give 0
     *
     * @return The generation result
     *
     * @endif
     */     
    virtual int open(void *args);
    
    /*!
     * @if jp
     * @brief ExecutionContext ѤΥåɼ¹Դؿ
     *
     * ExecutionContext ѤΥåɼ¹Դؿ
     * Ͽ줿ݡͥȤνƤӽФ
     *
     * @return ¹Է
     *
     * @else
     * @brief Thread execution function for ExecutionContext
     *
     * Thread execution function for ExecutionContext.
     * Invoke the registered components operation.
     *
     * @return The execution result
     *
     * @endif
     */     
    virtual int svc(void);
    
    /*!
     * @if jp
     * @brief ExecutionContext ѤΥåɼ¹Դؿ
     *
     * ExecutionContext ѤΥåɽλ˸ƤФ롣
     * ݡͥȥ֥Ȥ󥢥ƥֲޥ͡ؤΤԤ
     *  ACE_Task ӥ饹᥽åɤΥС饤ɡ
     *
     * @param flags λե饰
     *
     * @return λ
     *
     * @else
     *
     * @brief Thread execution function for ExecutionContext
     *
     * This function is invoked when activity thread for ExecutionContext exits.
     * Deactivate the component object and notify it to manager.
     * This is ACE_Task class method's override.
     *
     * @param flags Flag of the close
     *
     * @return The close result
     *
     * @endif
     */     
    virtual int close(unsigned long flags);
    
    //============================================================
    // ExecutionContext
    //============================================================
    /*!
     * @if jp
     * @brief ExecutionContext ¹Ծֳǧؿ
     *
     *  ExecutionContext  Runnning ֤ξ true ֤
     * Executioncontext  Running δ֡ Executioncontext ˻äƤ
     * ƤΥƥRTݡͥȤExecutionContext μ¹Լ˱
     * ¹Ԥ롣
     *
     * @return ֳǧؿ(ư:true:false)
     *
     * @else
     *
     * @brief Check for ExecutionContext running state
     *
     * This operation shall return true if the context is in the Running state.
     * While the context is Running, all Active RTCs participating
     * in the context shall be executed according to the contexts execution
     * kind.
     *
     * @return Check state function (Running:trueStopping:false)
     *
     * @endif
     */
    virtual CORBA::Boolean is_running(void)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionContext μ¹Ԥ򳫻
     *
     * ExecutionContext μ¹Ծ֤ Runnning Ȥ뤿ΥꥯȤȯԤ롣
     * ExecutionContext ξ֤ܤ ComponentAction::on_startup 
     * ƤӽФ롣
     * äƤRTݡͥȤޤ ExecutionContext 򳫻
     * 뤳ȤϤǤʤ
     * ExecutionContext ʣ󳫻/ߤ򷫤֤ȤǤ롣
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Start the ExecutionContext
     *
     * Request that the context enter the Running state. 
     * Once the state transition occurs, the ComponentAction::on_startup 
     * operation will be invoked.
     * An execution context may not be started until the RT-Components that
     * participate in it have been initialized.
     * An execution context may be started and stopped multiple times.
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t start(void)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionContext μ¹Ԥ
     *
     * ExecutionContext ξ֤ Stopped Ȥ뤿ΥꥯȤȯԤ롣
     * ܤȯϡComponentAction::on_shutdown ƤӽФ롣
     * äƤRTݡͥȤλ ExecutionContext ߤ
     * ɬפ롣
     * ExecutionContext ʣ󳫻/ߤ򷫤֤ȤǤ롣
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Stop the ExecutionContext
     *
     * Request that the context enter the Stopped state. 
     * Once the transition occurs, the ComponentAction::on_shutdown operation
     * will be invoked.
     * An execution context must be stopped before the RT components that
     * participate in it are finalized.
     * An execution context may be started and stopped multiple times.
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t stop(void)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionContext μ¹Լ(Hz)
     *
     * Active ֤ˤRTݡͥȤ¹Ԥ(ñ:Hz)롣
     *
     * @return (ñ:Hz)
     *
     * @else
     *
     * @brief Get execution rate(Hz) of ExecutionContext
     *
     * This operation shall return the rate (in hertz) at which its Active
     * participating RTCs are being invoked.
     *
     * @return Execution cycle(Unit:Hz)
     *
     * @endif
     */
    virtual CORBA::Double get_rate(void)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionContext μ¹Լ(Hz)ꤹ
     *
     * Active ֤ˤRTݡͥȤ¹Ԥ(ñ:Hz)ꤹ롣
     * ¹ԼѹϡDataFlowComponentAction  on_rate_changed ˤä
     * RTݡͥȤã롣
     *
     * @param rate (ñ:Hz)
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Set execution rate(Hz) of ExecutionContext
     *
     * This operation shall set the rate (in hertz) at which this contexts 
     * Active participating RTCs are being called.
     * If the execution kind of the context is PERIODIC, a rate change shall
     * result in the invocation of on_rate_changed on any RTCs realizing
     * DataFlowComponentAction that are registered with any RTCs participating
     * in the context.
     *
     * @param rate Execution cycle(Unit:Hz)
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t  set_rate(CORBA::Double rate)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief RTݡͥȤ򥢥ƥֲ
     *
     * Inactive ֤ˤRTݡͥȤActive ܤ
     * ƥֲ롣
     * ƤФ줿̡on_activate ƤӽФ롣
     * ꤷRTݡͥȤüԥꥹȤ˴ޤޤʤϡBAD_PARAMETER 
     * ֤롣
     * ꤷRTݡͥȤξ֤ Inactive ʳξϡ
     *  PRECONDITION_NOT_MET ֤롣
     *
     * @param comp ƥֲоRTݡͥ
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Activate an RT-component
     *
     * The given participant RTC is Inactive and is therefore not being invoked
     * according to the execution contexts execution kind. This operation
     * shall cause the RTC to transition to the Active state such that it may
     * subsequently be invoked in this execution context.
     * The callback on_activate shall be called as a result of calling this
     * operation. This operation shall not return until the callback has
     * returned, and shall result in an error if the callback does.
     *
     * @param comp The target RT-Component for activation
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t activate_component(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief RTݡͥȤ󥢥ƥֲ
     *
     * Inactive ֤ˤRTݡͥȤ󥢥ƥֲ
     * Inactive ܤ롣
     * ƤФ줿̡on_deactivate ƤӽФ롣
     * ꤷRTݡͥȤüԥꥹȤ˴ޤޤʤϡBAD_PARAMETER 
     * ֤롣
     * ꤷRTݡͥȤξ֤ Active ʳξϡ
     * PRECONDITION_NOT_MET ֤롣
     *
     * @param comp 󥢥ƥֲоRTݡͥ
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Deactivate an RT-component
     *
     * The given RTC is Active in the execution context. Cause it to transition 
     * to the Inactive state such that it will not be subsequently invoked from
     * the context unless and until it is activated again.
     * The callback on_deactivate shall be called as a result of calling this
     * operation. This operation shall not return until the callback has 
     * returned, and shall result in an error if the callback does.
     *
     * @param comp The target RT-Component for deactivate
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t deactivate_component(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief RTݡͥȤꥻåȤ
     *
     * Error ֤RTݡͥȤߤ롣
     * ƤФ줿̡on_reset ƤӽФ롣
     * ꤷRTݡͥȤüԥꥹȤ˴ޤޤʤϡBAD_PARAMETER
     * ֤롣
     * ꤷRTݡͥȤξ֤ Error ʳξϡPRECONDITION_NOT_MET
     * ֤롣
     *
     * @param comp ꥻåоRTݡͥ
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Reset the RT-component
     *
     * Attempt to recover the RTC when it is in Error.
     * The ComponentAction::on_reset callback shall be invoked. This operation
     * shall not return until the callback has returned, and shall result in an
     * error if the callback does. If possible, the RTC developer should
     * implement that callback such that the RTC may be returned to a valid
     * state.
     *
     * @param comp The target RT-Component for reset
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t reset_component(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief RTݡͥȤξ֤
     *
     * ꤷRTݡͥȤξ(LifeCycleState)롣
     * ꤷRTݡͥȤüԥꥹȤ˴ޤޤʤϡUNKNOWN_STATE 
     * ֤롣
     *
     * @param comp ּоRTݡͥ
     *
     * @return ߤξ(LifeCycleState)
     *
     * @else
     *
     * @brief Get RT-component's state
     *
     * This operation shall report the LifeCycleState of the given participant
     * RTC.
     * UNKNOWN_STATE will be returned, if the given RT-Component is not inclued
     * in the participant list.
     *
     * @param comp The target RT-Component to get the state
     *
     * @return The current state of the target RT-Component(LifeCycleState)
     *
     * @endif
     */
    virtual LifeCycleState get_component_state(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionKind 
     *
     *  ExecutionContext  ExecutionKind 
     *
     * @return ExecutionKind
     *
     * @else
     *
     * @brief Get the ExecutionKind
     *
     * This operation shall report the execution kind of the execution context.
     *
     * @return ExecutionKind
     *
     * @endif
     */
    virtual ExecutionKind get_kind(void)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief RTݡͥȤɲä
     *
     * ꤷRTݡͥȤ򻲲üԥꥹȤɲä롣
     * ɲä줿RTݡͥȤ attach_context ƤФ졢Inactive ֤
     * 롣
     * ꤵ줿RTݡͥȤnullξϡBAD_PARAMETER ֤롣
     * ꤵ줿RTݡͥȤ DataFlowComponent ʳξϡ
     * BAD_PARAMETER ֤롣
     *
     * @param comp ɲоRTݡͥ
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Add an RT-component
     *
     * The operation causes the given RTC to begin participating in the
     * execution context.
     * The newly added RTC will receive a call to 
     * LightweightRTComponent::attach_context and then enter the Inactive state.
     * BAD_PARAMETER will be invoked, if the given RT-Component is null or
     * if the given RT-Component is other than DataFlowComponent.
     *
     * @param comp The target RT-Component for add
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t add_component(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);

    /*!
     * @if jp
     * @brief ݡͥȤХɤ롣
     *
     * ݡͥȤХɤ롣
     *
     * @param rtc RTݡͥ
     * @return ReturnCode_t Υ꥿󥳡
     * @else
     * @brief Bind the component.
     *
     * Bind the component.
     *
     * @param rtc RT-Component's instances
     * @return The return code of ReturnCode_t type
     * @endif
     */
    virtual RTC::ReturnCode_t bindComponent(RTObject_impl* rtc);
    
    /*!
     * @if jp
     * @brief RTݡͥȤ򻲲üԥꥹȤ
     *
     * ꤷRTݡͥȤ򻲲üԥꥹȤ롣
     * 줿RTݡͥȤ detach_context ƤФ롣
     * ꤵ줿RTݡͥȤüԥꥹȤϿƤʤϡ
     * BAD_PARAMETER ֤롣
     *
     * @param comp оRTݡͥ
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Remove the RT-Component from participant list
     *
     * This operation causes a participant RTC to stop participating in the
     * execution context.
     * The removed RTC will receive a call to
     * LightweightRTComponent::detach_context.
     * BAD_PARAMETER will be returned, if the given RT-Component is not 
     * participating in the participant list.
     *
     * @param comp The target RT-Component for delete
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t remove_component(LightweightRTObject_ptr comp)
      throw (CORBA::SystemException);
    
    /*!
     * @if jp
     * @brief ExecutionContextProfile 
     *
     *  ExecutionContext Υץե롣
     *
     * @return ExecutionContextProfile
     *
     * @else
     *
     * @brief Get the ExecutionContextProfile
     *
     * This operation provides a profile descriptor for the execution 
     * context.
     *
     * @return ExecutionContextProfile
     *
     * @endif
     */
    virtual ExecutionContextProfile* get_profile(void)
      throw (CORBA::SystemException);
    
  protected:
    //============================================================
    // DFPBase
    //============================================================
    typedef LifeCycleState ExecContextState;
    /***
	enum ExecContextState
	{
	INACTIVE_STATE,
	ACTIVE_STATE,
	ERROR_STATE,
	};
    */
    typedef RTC_Utils::StateHolder<ExecContextState> ECStates;
    
    /*!
     * @if jp
     * @class DFPBase
     * @brief DFPBase 饹
     *
     * üԥꥹȤϿ줿 DataFlowParticipant 뤿ݥ饹
     *
     * @since 0.4.0
     *
     * @else
     * @class DFPBase
     * @brief DFPBase class
     *
     * The abstract class to manage DataFlowParticipant registered in 
     * tha participant list.
     *
     * @since 0.4.0
     *
     * @endif
     */
    class DFPBase
    {
    public:
      
      /*!
       * @if jp
       * @brief 󥹥ȥ饯
       *
       * 󥹥ȥ饯
       *
       * @param id ° ExecutionContext ID
       *
       * @else
       * @brief Constructor
       *
       * Constructor
       *
       * @param id ID of participating ExecutionContext
       *
       * @endif
       */
      DFPBase(RTC::ExecutionContextHandle_t id)
	: ec_id(id), m_sm(NUM_OF_LIFECYCLESTATE)
      {
	m_sm.setListener(this);
	m_sm.setEntryAction (ACTIVE_STATE, &DFPBase::on_activated);
	m_sm.setDoAction    (ACTIVE_STATE, &DFPBase::on_execute);
	m_sm.setPostDoAction(ACTIVE_STATE, &DFPBase::on_state_update);
	m_sm.setExitAction  (ACTIVE_STATE, &DFPBase::on_deactivated);
	m_sm.setEntryAction (ERROR_STATE,  &DFPBase::on_aborting);
	m_sm.setDoAction    (ERROR_STATE,  &DFPBase::on_error);
	m_sm.setExitAction  (ERROR_STATE,  &DFPBase::on_reset);
	
	ECStates st;
	st.prev = INACTIVE_STATE;
	st.curr = INACTIVE_STATE;
	st.next = INACTIVE_STATE;
	m_sm.setStartState(st);
	m_sm.goTo(INACTIVE_STATE);
      }	
      
      /*!
       * @if jp
       * @brief ǥȥ饯
       *
       * ǥȥ饯
       *
       * @else
       * @brief Destructor
       *
       * Destructor
       *
       * @endif
       */
      virtual ~DFPBase(void){}
      
      /*!
       * @if jp
       * @brief ExecutionContext ¹Գϻ˸ƤФ貾۴ؿ
       *
       * äƤ ExecutionContext ¹Ԥ򳫻Ϥ(Running֤ܻ)
       * ˸ƤФ貾۴ؿ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when ExecutionContext starts
       *
       * Pure virtual function to be invoked when given execution context, in
       * which the RTC is participating, has transited from Stopped to Running.
       *
       * @endif
       */
      virtual void on_startup(void) = 0;
      
      /*!
       * @if jp
       * @brief ExecutionContext ߻˸ƤФ貾۴ؿ
       *
       * äƤ ExecutionContext ¹Ԥߤ(Stopped֤ܻ)
       * ˸ƤФ貾۴ؿ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when ExecutionContext stops
       *
       * Pure virtual function to be invoked when given execution context, in
       * which the RTC is participating, has transited from Running to Stopped.
       *
       * @endif
       */
      virtual void on_shutdown(void) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȤƥֲ줿˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤƥֲ줿
       * (Active֤ܻ)˸ƤФ貾۴ؿ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when RT-Component is 
       *        activated
       *
       * Pure virtual function to be invoked when the RTC has been activated
       * in the given execution context.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_activated(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȤ󥢥ƥֲ줿˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤ󥢥ƥֲ줿
       * (Deactive֤ܻ)˸ƤФ貾۴ؿ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when RT-Component is 
       *        deactivated
       *
       * Pure virtual function to be invoked when the RTC has been deactivated
       * in the given execution context.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_deactivated(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȤǥ顼ȯ˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤ˥顼ȯ(Error֤ܻ)
       * ˸ƤФ貾۴ؿ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when RT-Component occurs 
       *        error
       *
       * Pure virtual function to be invoked when the RTC is transiting from
       * the Active state to the Error state in some execution context.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_aborting(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȤ顼֤λ˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤ顼֤ˤ֡on_execute 
       * on_state_update ؤäŪ˸ƤӽФ貾۴ؿ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked while RT-Component
       * is in the error state
       *
       * If the RTC is in the Error state relative to some execution context
       * when it would otherwise be invoked from that context.
       * This operation shall be invoked in sorted order at the rate of the
       * context instead of DataFlowComponentAction::on_execute and 
       * on_state_update.The RTC is transitioning from the Active state to 
       * the Error state in some execution context.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_error(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȤꥻåȤ˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤꥻåȤݤ˸ƤФ貾۴ؿ
       * δؿ˽λȡRTC Inactive ֤롣
       * δؿ˽λʤäϡ Error ֤αޤ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when RT-Component resets.
       *
       * The RTC is in the Error state. An attempt is being made to recover it 
       * such that it can return to the Inactive state.
       * If the RTC was successfully recovered and can safely return to 
       * the Inactive state, this method shall complete with ReturnCode_t::OK.
       * Any other result shall indicate that the RTC should remain in the 
       * Error state.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_reset(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȼ¹ԻŪ˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤ Active ֤ǤȤȤˡ
       * ExecutionContext  Running ֤ξˡꤵ줿ưŪ
       * ƤӽФ貾۴ؿ
       * Two-Pass Execution κǽμ¹ԤǸƤФ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be periodically invoked while 
       *        RT-Component is running
       *
       * This operation will be invoked periodically at the rate of the given
       * execution context as long as the following conditions hold:
       *  - The RTC is Active.
       *  - The given execution context is Running.
       * This callback occurs during the first execution pass.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_execute(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief RTݡͥȼ¹ԻŪ˸ƤФ貾۴ؿ
       *
       * оݤRTݡͥȤ Active ֤ǤȤȤˡ
       * ExecutionContext  Running ֤ξˡꤵ줿ưŪ
       * ƤӽФ貾۴ؿ
       * Two-Pass Execution Σܤμ¹ԤǸƤФ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       *
       * @brief Pure virtual function to be periodically invoked while 
       *        RT-Component is running
       *
       * This operation will be invoked periodically at the rate of the given
       * execution context as long as the following conditions hold:
       *  - The RTC is Active.
       *  - The given execution context is Running.
       * This callback occurs during the second execution pass.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      virtual void on_state_update(const ECStates& st) = 0;
      
      /*!
       * @if jp
       * @brief ExecutionContext μ¹Լѹ˸ƤФ貾۴ؿ
       *
       * äƤ ExecutionContext μ¹ԼѹȤʤäˡ
       * ѹã뤿˸ƤӽФ貾۴ؿ
       *
       * @else
       *
       * @brief Pure virtual function to be invoked when when the execution 
       *        cycles of ExecutionContext is changed.
       *
       * This operation is a notification that the rate of the indicated
       * execution context has changed.
       *
       * @endif
       */
      virtual void on_rate_changed(void) = 0;
      
      /*!
       * @if jp
       * @brief ܤ¹Ԥ
       *
       * оRTݡͥȤξܤ¹Ԥ롣
       *
       * @return 
       *
       * @else
       * @brief Get the worker to execute the state transition
       *
       * Get the worker that executes the state transition of the target
       * component to manage.
       *
       * @return The worker
       *
       * @endif
       */
      virtual void worker(void) {return m_sm.worker();}
      
      /*!
       * @if jp
       * @brief ߤξ֤
       *
       * оRTݡͥȤθߤξ֤롣
       *
       * @return ߾
       *
       * @else
       * @brief Get the current state of the target component
       *
       * Get the current state of the target component to manage
       *
       * @return The current state of the target RT-Component
       *
       * @endif
       */
      virtual ExecContextState get_state(void){ return m_sm.getState();}
      
      /*!
       * @if jp
       * @brief äƤ ExecutionContext  ID
       * @else
       * @brief ID of participating ExecutionContext
       * @endif
       */
      ExecutionContextHandle_t ec_id;
      
      /*!
       * @if jp
       * @brief оRTݡͥȤΥơȥޥ
       * @else
       * @brief The state machine of the target RT-Component to manage
       * @endif
       */
      RTC_Utils::StateMachine<ExecContextState, DFPBase> m_sm;
    };
    
    //============================================================
    // DFP
    //============================================================
    /*!
     * @if jp
     * @class DFP
     * @brief DFP 饹
     *
     * üԥꥹȤϿ줿 DataFlowParticipant δؿư뤿
     * ƥץ졼ȥ饹
     *
     * @param Object оݥݡͥȤη
     *
     * @since 0.4.0
     *
     * @else
     * @class DFP
     * @brief DFP class
     *
     * Template class to invoke DataFlowParticipant registered
     * in the participant list.
     *
     * @param Object Type of the target component to manage
     *
     * @since 0.4.0
     *
     * @endif
     */
    template <class Object>
    class DFP
      : public DFPBase
    {
    public:
      /*!
       * @if jp
       * @brief ǥեȥ󥹥ȥ饯
       *
       * ǥեȥ󥹥ȥ饯
       *
       * @param obj оݥݡͥ
       * @param id ° ExecutionContext ID
       *
       * @else
       * @brief Default constructor
       *
       * Default constructor
       *
       * @param obj The target component to manage
       * @param id ID of participating ExecutionContext
       *
       * @endif
       */
      DFP(Object obj, ExecutionContextHandle_t id)
	: DFPBase(id), m_obj(obj), m_active(true)
      {
      }
      
      /*!
       * @if jp
       * @brief ExecutionContext ¹Գϻ˸ƤФؿ
       *
       * äƤ ExecutionContext ¹Ԥ򳫻Ϥ(Running֤ܻ)
       * ˡоݥݡͥȤ on_startup ƤӤ
       *
       * @else
       * @brief Function to be invoked when ExecutionContext starts
       *
       * When the given ExecutionContext transits from Stopped to Running,
       * on_startup of the participation component will be invoked.
       *
       * @endif
       */
      void on_startup(void)
      {
	m_obj->on_startup(ec_id);
      }
      
      /*!
       * @if jp
       * @brief ExecutionContext ߻˸ƤФؿ
       *
       * äƤ ExecutionContext ¹Ԥߤ(Stopped֤ܻ)
       * ˡоݥݡͥȤ on_shutdown ƤӤ
       *
       * @else
       * @brief Function to be invoked when ExecutionContext stops
       *
       * When the given ExecutionContext transits from Running to Stopped,
       * on_shutdown of the participation component will be invoked.
       *
       * @endif
       */
      void on_shutdown(void)
      {
	m_obj->on_shutdown(ec_id);
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȤƥֲ줿˸ƤФؿ
       *
       * оݤRTݡͥȤƥֲ줿(Active֤ܻ)
       * ˡоݥݡͥȤ on_activated ƤӤ
       * оݥݡͥȤΥƥֲԤˤϡơȥޥ
       *  Error ֤ܤ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked when RT-Component was activated
       *
       * When the given ExecutionContext transits to the Active state,
       * on_activated of the participation component will be invoked.
       * If it fails, the state machine transits to the Errot state.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_activated(const ECStates& st)
      {
	if (m_obj->on_activated(ec_id) != RTC::RTC_OK)
	  {
	    m_sm.goTo(ERROR_STATE);
	    return;
	  }
	return;
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȤ󥢥ƥֲ줿˸ƤФؿ
       *
       * оݤRTݡͥȤ󥢥ƥֲ줿
       * (Deactive֤ܻ)ˡоݥݡͥȤ on_deactivated 
       * ƤӤ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked when RT-Component was deactivated
       *
       * When the given ExecutionContext transits the Deactivate state,
       * on_deactivated of the participation component will be invoked.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_deactivated(const ECStates& st)
      {
	m_obj->on_deactivated(ec_id);
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȤǥ顼ȯ˸ƤФؿ
       *
       * оݤRTݡͥȤ˥顼ȯ(Error֤ܻ)
       * ˴оݥݡͥȤ on_aborting ƤӤ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked when RT-Component occured error
       *
       * When the given ExecutionContext transits the Error state,
       * on_aborting of the participation component will be invoked.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_aborting(const ECStates& st)
      {
	m_obj->on_aborting(ec_id);
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȤ顼֤λ˸ƤФؿ
       *
       * оݤRTݡͥȤ顼֤ˤ֡
       * оݥݡͥȤ on_aborting Ū˸ƤӤ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked while RT-Component is in the error state
       *
       * While the given RT-Component is in the Error state,
       * its on_aborting will be periodically invoked.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_error(const ECStates& st)
      {
	m_obj->on_error(ec_id);
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȤꥻåȤ˸ƤФؿ
       *
       * оݤRTݡͥȤꥻåȤݤˡоݥݡͥ
       *  on_reset ƤӤ
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked when RT-Component is reset.
       *
       * When the target RT-Component is reset,
       * invoke on_reset of the target component to manage.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_reset(const ECStates& st)
      {
	if (m_obj->on_reset(ec_id) != RTC::RTC_OK)
	  {
	    m_sm.goTo(ERROR_STATE);
	    return;
	  }
	return;
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȼ¹ԻŪ˸ƤФؿ
       *
       * оݤRTݡͥȤ Active ֤ǤȤȤˡ
       * ExecutionContext  Running ֤ξˡꤵ줿ưŪ
       * оݥݡͥȤ on_execute ƤӤ
       * ؿμ¹Ԥ˼Ԥ(ͤ RTC_OK ʳ)оݥݡͥȤ
       * ֤ Error ֤ܤ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be periodically invoked while RT-Component is running
       *
       * If the given RT-Component is in the Active state and ExecutionContext
       * is in the Running state, on_execute of the given component will be
       * invoked periodically at the specified execution cycle.
       * If it fails (the return value is other than RTC_OK), its state transits 
       * to the Errot state.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_execute(const ECStates& st)
      {
	if (m_obj->on_execute(ec_id) != RTC::RTC_OK)
	  {
	    m_sm.goTo(ERROR_STATE);
	    return;
	  }  
	return;
      }
      
      /*!
       * @if jp
       * @brief RTݡͥȼ¹ԻŪ˸ƤФؿ
       *
       * оݤRTݡͥȤ Active ֤ǤȤȤˡ
       * ExecutionContext  Running ֤ξˡꤵ줿ưŪ
       * оݥݡͥȤ on_state_update ƤӤ
       * ؿμ¹Ԥ˼Ԥ(ͤ RTC_OK ʳ)оݥݡͥȤ
       * ֤ Error ֤ܤ롣
       *
       * @param st оRTݡͥȤθߤξ
       *
       * @else
       * @brief Function to be invoked periodically while RT-Component executes
       *
       * When the target RT-Component to manage is in the Active state and
       * ExecutionContext is the Running, invoke on_state_update of the target
       * component to manage periodically in specified execution cycle.
       * If it fails (the return value is other than RTC_OK), its state transits 
       * to the Errot state.
       *
       * @param st The current state of the target RT-Component
       *
       * @endif
       */
      void on_state_update(const ECStates& st)
      {
	if (m_obj->on_state_update(ec_id) != RTC::RTC_OK)
	  {
	    m_sm.goTo(ERROR_STATE);
	    return;
	  }
	return;
      }
      
      /*!
       * @if jp
       * @brief ExecutionContext μ¹Լѹ˸ƤФؿ
       *
       * äƤ ExecutionContext μ¹ԼѹȤʤäˡ
       * оݥݡͥȤ on_rate_changed ƤӤ
       *
       * @else
       * @brief Function to be invoked when the execution cycles of
       *        ExecutionContext is changed
       *
       * When the execution cycle of the participating ExecutionContext is
       * changed, invoke on_rate_changed of the target component will be invoked.
       *
       * @endif
       */
      void on_rate_changed(void)
      {
	m_obj->on_rate_changed(ec_id);
      }
      
      /*!
       * @if jp
       * @brief оݥݡͥ
       * @else
       * @brief The target component to manage
       * @endif
       */
      Object m_obj;
      
      /*!
       * @if jp
       * @brief оݥݡͥȤư֥ե饰
       * @else
       * @brief State flag of the target component to manage
       * @endif
       */
      bool m_active;
    };
    
    /*!
     * @if jp
     * @brief ݡͥȴѹ¤
     * @else
     * @brief The structure for the component management
     * @endif
     */
    struct Comp
    {
      Comp(LightweightRTObject_ptr ref, OpenRTM::DataFlowComponent_ptr dfp,
	   ExecutionContextHandle_t id)
	: _ref(LightweightRTObject::_duplicate(ref)),
	  _sm(OpenRTM::DataFlowComponent::_duplicate(dfp), id)
      {
      }
      ~Comp(void)
      {
      }
      Comp(const Comp& comp)
	: _ref(comp._ref), _sm(comp._sm.m_obj, comp._sm.ec_id)
      {
      }
      Comp& operator=(const Comp& comp)
      {
	_ref = comp._ref;
	_sm.m_obj = comp._sm.m_obj;
	_sm.ec_id = comp._sm.ec_id;
	return *this;
      }
      LightweightRTObject_var _ref;
      DFP<OpenRTM::DataFlowComponent_var> _sm;
    };

    /*!
     * @if jp
     * @brief ݡͥȸѥե󥯥
     * @else
     * @brief Functor to find the component
     * @endif
     */
    struct find_comp
    {
      LightweightRTObject_var m_comp;
      find_comp(LightweightRTObject_ptr comp) : m_comp(LightweightRTObject::_duplicate(comp)) {}
      bool operator()(Comp& comp)
      {
	return comp._ref->_is_equivalent(m_comp);
      }
    };
    
    /*!
     * @if jp
     * @brief on_startup ưѥե󥯥
     * @else
     * @brief Functor to invoke on_startup
     * @endif
     */
    struct invoke_on_startup
    {
      void operator()(Comp& comp)
      {
	comp._sm.on_startup();
      }
    };
    
    /*!
     * @if jp
     * @brief on_shutdown ưѥե󥯥
     * @else
     * @brief Functor to invoke on_shutdown
     * @endif
     */
    struct invoke_on_shutdown
    {
      void operator()(Comp& comp)
      {
	comp._sm.on_shutdown();
      }
    };
    
    /*!
     * @if jp
     * @brief on_rate_changed ưѥե󥯥
     * @else
     * @brief Functor to invoke on_rate_changed
     * @endif
     */
    struct invoke_on_rate_changed
    {
      void operator()(Comp& comp)
      {
	comp._sm.on_rate_changed();
      }
    };
    
    /*!
     * @if jp
     * @brief ¹ѥե󥯥
     * @else
     * @brief Functor to invoke worker
     * @endif
     */
    struct invoke_worker
    {
      void operator()(Comp& comp)
      {
	comp._sm.worker();
      }
    };
    
    /*!
     * @if jp
     * @brief ݡͥȤλüԥꥹ
     * @else
     * @brief List of the participating component
     * @endif
     */
    std::vector<Comp> m_comps;
    typedef std::vector<Comp>::iterator CompItr;
    
    /*!
     * @if jp
     * @brief ȥ꡼
     * @else
     * @brief Logger stream
     * @endif
     */
    Logger rtclog;

    /*!
     * @if jp
     * @brief ExecutionContext μ¹Ծ
     * true: running, false: stopped
     * @else
     * @brief The running state of ExecutionContext
     * true: running, false: stopped
     * @endif
     */
    bool m_running;

    /*!
     * @if jp
     * @brief ExecutionContext Υåɼ¹ԥե饰
     * @else
     * @brief The thread running flag of ExecutionContext
     * @endif
     */
    bool m_svc;

    struct Worker
    {
      Worker() : cond_(mutex_), running_(false) {};
      coil::Mutex mutex_;
      coil::Condition<coil::Mutex> cond_;
      bool running_;
    };
    /*!
     * @if jp
     * @brief svnѤξѿ 
     * @else
     * @brief A condition variable for external triggered worker
     * @endif
     */
    // A condition variable for external triggered worker
    Worker m_worker;
    
    /*!
     * @if jp
     * @brief ExecutionContextProfile
     * @else
     * @brief ExecutionContextProfile
     * @endif
     */
    ExecutionContextProfile m_profile;
    
    /*!
     * @if jp
     * @brief ExecutionContext μ¹Լ
     * @else
     * @brief Execution cycle of ExecutionContext
     * @endif
     */
    coil::TimeValue m_period;
    
    /*!
     * @if jp
     * @brief ExecutionContextService ֥Ȥؤλ
     * @else
     * @brief Reference to ExecutionContextService object
     * @endif
     */
    ExecutionContextService_var m_ref;
    
    /*!
     * @if jp
     * @brief ExecutionContext ¨¹(wait̵¹)ե饰
     * @else
     * @brief Flag of ExecutionContext to run immediately
     *        (to run without waiting)
     * @endif
     */
    bool m_nowait;
  }; // class PeriodicExecutionContext
}; // namespace RTC

#ifdef WIN32
#pragma warning( default : 4290 )
#endif


extern "C"
{
  void PeriodicExecutionContextInit(RTC::Manager* manager);
};

#endif // RTC_PERIODICEXECUTIONCONTEXT_H
