// -*- C++ -*-
/*!
 * @file OutPortBase.h
 * @brief InPortBase base class
 * @date $Date: 2007-12-31 03:08:05 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2003-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: OutPortBase.h 1754 2010-01-22 07:18:53Z fsi-katami $
 *
 */

#ifndef RTC_OUTPORTBASE_H
#define RTC_OUTPORTBASE_H

#include <vector>
#include <string>

#include <coil/Properties.h>
#include <coil/stringutil.h>

#include <rtm/PortBase.h>
#include <rtm/InPortConsumer.h>
#include <rtm/OutPortProvider.h>
#include <rtm/ConnectorBase.h>
#include <rtm/CdrBufferBase.h>
#include <rtm/SystemLogger.h>
#include <rtm/ConnectorListener.h>

namespace RTC
{
  class PublisherBase;
  class ConnectorBase;
  class OutPortConnector;

  /*!
   * @if jp
   *
   * @class OutPortBase
   *
   * @brief OutPort 쥯饹
   * 
   * OutPort δ쥯饹
   *
   *
   *
   * Properties: port.outport
   * ץѥƥ
   *
   * - port.outport
   * - port.outport.[name]
   * ConnectorProfile.properties ξ
   * - dataport.outport
   *
   * ʲ˻ꤷΤϤ롣
   * (port.outport.[name]ͥ褵)
   * ˡΥץѥƥ³ ConnectorProfile ˤ
   * Ϥ礬ꡢξ ConnectorProfile ͥ褵롣
   *
   * - input.throughput.profile: enable
   * - input.throughput.update_rate: count [n/count]
   * - input.throughput.total_bytes: [bytes]
   * - input.throughput.total_count: [n]
   * - input.throughput.max_size: [bytes]
   * - input.throughput.min_size: [bytes]
   * - input.throughput.avg_size: [bytes]
   * - input.throughput.byte_sec: [bytes/sec]
   *
   * - output.throughput.profile: enable
   * - output.throughput.update_rate: count [n/count]
   * - output.throughput.total_bytes: [bytes]
   * - output.throughput.total_count:[n]
   * - output.throughput.max_size: [bytes]
   * - output.throughput.min_size: [bytes]
   * - output.throughput.avg_size: [bytes]
   * - output.throughput.max_sendtime: [sec]
   * - output.throughput.min_sendtime: [sec]
   * - output.throughput.avg_sendtime: [sec]
   * - output.throughput.byte_sec: [bytes/sec]
   *
   * dataport.dataflow_type
   * dataport.interface_type
   * dataport.subscription_type
   *
   * [buffer]
   *
   * - buffer.type:
   *     ѲǽʥХåեΥ
   *     ConnectorProfile ξѤХåեΥ
   *     ̵ξϥǥեȤ ringbuffer Ѥ롣
   *     ex. ringbuffer, shmbuffer, doublebuffer, etc.
   *     Consumer, Publisher ΥפˤäƤΥХåե
   *     ׵᤹뤬뤿Ρξϻ̵Ȥʤ롣
   *
   * - buffer.length:
   *     ХåեĹ
   *
   * - buffer.write.full_policy:
   *     񤭤뤫ɤΥݥꥷ
   *     overwrite (), do_nothing (⤷ʤ), block (֥å)
   *     block ꤷ硢 timeout ͤꤹСָ
   *     񤭹ԲǽǤХॢȤ롣
   *
   * - buffer.write.timeout:
   *     ॢȻ֤ [sec] ǻꤹ롣
   *     1 sec -> 1.0, 1 ms -> 0.001, ॢȤʤ -> 0.0
   *
   * - buffer.read.empty_policy:
   *     ХåեΤȤɤ߽Фݥꥷ
   *     last (Ǹ), do_nothing (⤷ʤ), block (֥å)
   *     block ꤷ硢 timeout ͤꤹСָ
   *     ɤ߽ФԲǽǤХॢȤ롣
   *
   * - buffer.read.timeout:
   *     ॢȻ [sec] ǻꤹ롣
   *     1sec -> 1.0, 1ms -> 0.001, ॢȤʤ -> 0.0
   *
   * - ¾Хåե˸ͭʥץ
   *
   *
   * [publihser]
   *
   * - publisher.types:
   *      Ѳǽ Publisher Υ
   *      new, periodic, flush, etc..
   *
   * - publisher.push.policy:
   *      InPortإǡݥꥷ
   *      all: ХåեˤޤäƤ뤹٤
   *      fifo: ХåեFIFOȤߤʤ
   *      skip: Ťǡְ
   *      new: ˿ǡΤߤ
   *
   * - publisher.push.skip_rate:
   *      push.policy=skip ΤȤΥåΨ
   *      n: nˤҤȤ
   *
   * - publisher.periodic.rate:
   *
   * - publisher.thread.type:
   *       Publisher ΥåɤΥ
   * - publisher.thread.measurement.exec_time: yes/no
   * - publisher.thread.measurement.exec_count: number
   * - publisher.thread.measurement.period_time: yes/no
   * - publisher.thread.measurement.period_count: number
   *
   * [interface]
   *
   * - interface.types:
   *     OutPort interfaceΥ
   *     ex. corba_cdr, corba_any, raw_tcp ʤɥ޶ڤǻꡣ
   *     ꤷʤѲǽʤ٤ƤΥץХѤ
   *
   *
   *
   *   
   * OutPort ¦ connect() ǤϰʲΥ󥹤ǽԤ롣
   *
   * 1. OutPort ˴Ϣ connector ӥå
   *
   * 2. InPort˴Ϣ connector μ
   *  - ConnectorProfile::properties["dataport.corba_any.inport_ref"]
   *    OutPortAny Υ֥ȥե󥹤ꤵƤ硢
   *    ե󥹤Consumer֥Ȥ˥åȤ롣
   *    ե󥹤åȤƤʤ̵뤷Ʒ³
   *    (OutPortconnect() ƤӽФΥȥݥȤξϡ
   *    InPortΥ֥ȥե󥹤ϥåȤƤʤϤǤ롣)
   * 3. PortBase::connect() 򥳡
   *    Port³δܽԤ롣
   * 4. 嵭2.InPortΥե󥹤ǤʤСInPort
   *    Ϣ connector 롣
   *
   * 5. ConnectorProfile::properties Ϳ줿󤫤顢
   *    OutPort¦νԤ
   *
   * - [dataport.interface_type]
   * -- CORBA_Any ξ: 
   *    InPortAny ̤ƥǡ򴹤롣
   *    ConnectorProfile::properties["dataport.corba_any.inport_ref"]
   *    InPortAny Υ֥ȥե󥹤򥻥åȤ롣
   * -- RawTCP ξ: Raw TCP socket ̤ƥǡ򴹤롣
   *    ConnectorProfile::properties["dataport.raw_tcp.server_addr"]
   *    InPort¦ΥХɥ쥹򥻥åȤ롣
   *
   * - [dataport.dataflow_type]
   * -- Pushξ: Subscriber롣SubscriberΥפϡ
   *    dataport.subscription_type ꤵƤ롣
   * -- Pullξ: InPort¦ǡPullǼ뤿ᡢ
   *    ä˲⤹ɬפ̵
   *
   * - [dataport.subscription_type]
   * -- Onceξ: SubscriberOnce롣
   * -- Newξ: SubscriberNew롣
   * -- Periodicξ: SubscriberPeriodic롣
   *
   * - [dataport.publisher.push_rate]
   * -- dataport.subscription_type=Periodicξꤹ롣
   *
   * 6. 嵭νΤĤǤ⥨顼ǤС顼꥿󤹤롣
   *    ˽Ԥ줿RTC::RTC_OKǥ꥿󤹤롣
   *
   * @since 0.2.0
   *
   * @else
   *
   * @class OutPortBase
   *
   * @brief Output base class.
   *
   * The base class of OutPort<T> which are implementations of OutPort
   *
   * Form a kind of Observer pattern with OutPortBase and PublisherBase.
   * attach(), detach(), notify() of OutPortBase and
   * push() of PublisherBase are methods associated with the Observer pattern.
   *
   * @since 0.2.0
   *
   * @endif
   */
  class OutPortBase
    : public PortBase, public DataPortStatus
  {
  public:
    DATAPORTSTATUS_ENUM

    typedef std::vector<OutPortConnector*> ConnectorList;

    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @param name ݡ̾
     *
     * @else
     * @brief Constructor
     *
     * Constructor
     *
     * @param name Port name
     *
     * @endif
     */
    OutPortBase(const char* name, const char* data_type);
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     * Ͽ줿Ƥ Publisher 롣
     *
     * @else
     *
     * @brief Destructor
     *
     * Destructor
     * Detach all subscribed Publishers.
     *
     * @endif
     */
    virtual ~OutPortBase(void);
    
    /*!
     * @if jp
     * @brief ץѥƥν
     *
     * OutPortΥץѥƥ
     *
     * @else
     *
     * @brief Initializing properties
     *
     * This operation initializes outport's properties
     *
     * @endif
     */
    void init(coil::Properties& prop);

    /*!
     * @if jp
     *
     * @brief ǡ񤭹
     *
     * ݡȤإǡ񤭹ࡣ
     * Хɤ줿ѿꤵ줿ͤݡȤ˽񤭹ࡣ
     *
     * @return 񤭹߽(񤭹:true񤭹߼:false)
     *
     * @else
     *
     * @brief Write data
     *
     * Write data to the port.
     * Write the value, which was set to the bound variable, to the port.
     *
     * @return Writing result (Successful:true, Failed:false)
     *
     * @endif
     */
    virtual bool write() = 0;


    /*!
     * @if jp
     * @brief OutPort̾Τμ
     *
     * OutPort̾Τ롣
     *
     * @return ݡ̾
     *
     * @else
     *
     * @brief Retrieve OutPort name
     *
     * Retrieve OutPort name
     *
     * @return the port name
     *
     * @endif
     */
    //    const char* name() const;

    /*!
     * @if jp
     * @brief ץѥƥ
     *
     * OutPortΥץѥƥ롣
     *
     * @return ץѥƥ
     *
     * @else
     *
     * @brief Get properties
     *
     * Getting properties of this OutPort
     *
     * @return OutPort's properties
     *
     * @endif
     */
    coil::Properties& properties();

    /*!
     * @if jp
     * @brief Connector 
     *
     * ߽ͭƤ륳ͥ롣
     *
     * @return connector Υꥹ
     *
     * @else
     *
     * @brief Connector list
     *
     * This operation returns connector list
     *
     * @return connector list
     *
     * @endif
     */
    const std::vector<OutPortConnector*>& connectors();

    /*!
     * @if jp
     * @brief ConnectorProfile 
     *
     * ߽ͭƤ륳ͥProfile롣
     *
     * @return ConnectorProfile Υꥹ
     *
     * @else
     *
     * @brief ConnectorProfile list
     *
     * This operation returns ConnectorProfile list
     *
     * @return connector list
     *
     * @endif
     */
    ConnectorInfoList getConnectorProfiles();

    /*!
     * @if jp
     * @brief ConnectorId 
     *
     * ߽ͭƤ륳ͥID롣
     *
     * @return ConnectorId Υꥹ
     *
     * @else
     *
     * @brief ConnectorId list
     *
     * This operation returns ConnectorId list
     *
     * @return connector list
     *
     * @endif
     */
    coil::vstring getConnectorIds();

    /*!
     * @if jp
     * @brief Connector̾
     *
     * ߽ͭƤ륳ͥ̾롣
     *
     * @return Connector̾Υꥹ
     *
     * @else
     *
     * @brief Connector name list
     *
     * This operation returns Connector name list
     *
     * @return connector name list
     *
     * @endif
     */
    coil::vstring getConnectorNames();

    /*!
     * @if jp
     * @brief ConnectorProfileIDǼ
     *
     * ߽ͭƤ륳ͥIDǼ롣
     *
     * @param id Connector ID
     * @return ͥؤΥݥ
     *
     * @else
     *
     * @brief Getting ConnectorProfile by ID
     *
     * This operation returns Connector specified by ID.
     *
     * @param id Connector ID
     * @return A pointer to connector
     *
     * @endif
     */
    OutPortConnector* getConnectorById(const char* id);

     /*!
     * @if jp
     * @brief ConnectorProfile̾Ǽ
     *
     * ߽ͭƤ륳ͥ̾Ǽ롣
     *
     * @param name Connector name
     * @return ͥؤΥݥ
     *
     * @else
     *
     * @brief Getting Connector by name
     *
     * This operation returns Connector specified by name.
     *
     * @param id Connector ID
     * @return A pointer to connector
     *
     * @endif
     */
   OutPortConnector* getConnectorByName(const char* name);

    /*!
     * @if jp
     * @brief ConnectorProfileIDǼ
     *
     * ߽ͭƤ륳ͥIDǼ롣
     *
     * @param id Connector ID
     * @param prof ConnectorProfile
     * @return false ꤷIDʤ
     *
     * @else
     *
     * @brief Getting ConnectorProfile by name
     *
     * This operation returns ConnectorProfile specified by name
     *
     * @param id Connector ID
     * @param prof ConnectorProfile
     * @return falsespecified ID does not exist
     *
     * @endif
     */
    bool getConnectorProfileById(const char* id,
                                 ConnectorInfo& prof);

    /*!
     * @if jp
     * @brief ConnectorProfile̾Ǽ
     *
     * ߽ͭƤ륳ͥ̾Ǽ롣
     *
     * @param name Connector name
     * @param prof ConnectorProfile
     * @return false ꤷ̾ʤ
     *
     * @else
     *
     * @brief Getting ConnectorProfile by name
     *
     * This operation returns ConnectorProfile specified by name
     *
     * @param id Connector ID
     * @param prof ConnectorProfile
     * @return false specified name does not exist
     *
     * @endif
     */
    bool getConnectorProfileByName(const char* name,
                                   ConnectorInfo& prof);

    
    /*!
     * @if jp
     *
     * @brief InPort activates 
     *
     * InPort activate 롣
     *
     * @else
     *
     * @brief Activate all Port interfaces
     *
     * This operation activate all interfaces that is registered in the
     * ports.
     *
     * @endif
     */
    virtual void activateInterfaces();

    /*!
     * @if jp
     *
     * @brief Ƥ Port Υ󥿡ե deactivates 
     *
     * Port ϿƤƤΥ󥿡ե deactivate 롣
     *
     * @else
     *
     * @brief Deactivate all Port interfaces
     *
     * This operation deactivate all interfaces that is registered in the
     * ports.
     *
     * @endif
     */
    virtual void deactivateInterfaces();


    /*!
     * @if jp
     * @brief ConnectorDataListener ꥹʤɲä
     *
     * Хåե񤭹ߤޤɤ߽Ф٥Ȥ˴ϢƼꥹʤꤹ롣
     *
     * ǤꥹʤΥפȥХå٥Ȥϰʲ̤
     *
     * - ON_BUFFER_WRITE:          Хåե񤭹߻
     * - ON_BUFFER_FULL:           Хåեե
     * - ON_BUFFER_WRITE_TIMEOUT:  Хåե񤭹ߥॢȻ
     * - ON_BUFFER_OVERWRITE:      Хåե񤭻
     * - ON_BUFFER_READ:           Хåեɤ߽Ф
     * - ON_SEND:                  InProtؤ
     * - ON_RECEIVED:              InProtؤλ
     * - ON_SEND_ERTIMEOUT:        OutPort¦ॢȻ
     * - ON_SEND_ERERROR:          OutPort¦顼
     * - ON_RECEIVER_FULL:         InProt¦Хåեե
     * - ON_RECEIVER_TIMEOUT:      InProt¦ХåեॢȻ
     * - ON_RECEIVER_ERROR:        InProt¦顼
     *
     * ꥹʤ ConnectorDataListener ѾʲΥ˥
     * operator() Ƥɬפ롣
     *
     * ConnectorDataListener::
     *         operator()(const ConnectorProfile&, const cdrStream&)
     *
     * ǥեȤǤϡδؿͿꥹʥ֥Ȥνͭ
     * OutPort˰ܤꡢOutPortλ⤷ϡ
     * removeConnectorDataListener() ˤ˼ưŪ˲Τ롣
     * ꥹʥ֥ȤνͭƤӽФ¦ǰݻϡ3
     *  false ꤷưŪʲΤ뤳ȤǤ롣
     *
     * @param listener_type ꥹʥ
     * @param listener ꥹʥ֥ȤؤΥݥ
     * @param autoclean ꥹʥ֥ȤμưŪΤԤɤΥե饰
     *
     * @else
     * @brief Adding BufferDataListener type listener
     *
     * This operation adds certain listeners related to buffer writing and
     * reading events.
     * The following listener types are available.
     *
     * - ON_BUFFER_WRITE:          At the time of buffer write
     * - ON_BUFFER_FULL:           At the time of buffer full
     * - ON_BUFFER_WRITE_TIMEOUT:  At the time of buffer write timeout
     * - ON_BUFFER_OVERWRITE:      At the time of buffer overwrite
     * - ON_BUFFER_READ:           At the time of buffer read
     * - ON_SEND:                  At the time of sending to InPort
     * - ON_RECEIVED:              At the time of finishing sending to InPort
     * - ON_SENDER_TIMEOUT:        At the time of timeout of OutPort
     * - ON_SENDER_ERROR:          At the time of error of OutPort
     * - ON_RECEIVER_FULL:         At the time of bufferfull of InPort
     * - ON_RECEIVER_TIMEOUT:      At the time of timeout of InPort
     * - ON_RECEIVER_ERROR:        At the time of error of InPort
     *
     * Listeners should have the following function operator().
     *
     * ConnectorDataListener::
     *         operator()(const ConnectorProfile&, const cdrStream&)
     *
     * The ownership of the given listener object is transferred to
     * this OutPort object in default.  The given listener object will
     * be destroied automatically in the OutPort's dtor or if the
     * listener is deleted by removeConnectorDataListener() function.
     * If you want to keep ownership of the listener object, give
     * "false" value to 3rd argument to inhibit automatic destruction.
     *
     * @param listener_type A listener type
     * @param listener A pointer to a listener object
     * @param autoclean A flag for automatic listener destruction
     *
     * @endif
     */
    void addConnectorDataListener(ConnectorDataListenerType listener_type,
                                  ConnectorDataListener* listener,
                                  bool autoclean = true);


    /*!
     * @if jp
     * @brief ConnectorDataListener ꥹʤ
     *
     * ꤷƼꥹʤ롣
     * 
     * @param listener_type ꥹʥ
     * @param listener ꥹʥ֥ȤؤΥݥ
     *
     * @else
     * @brief Removing BufferDataListener type listener
     *
     * This operation removes a specified listener.
     *     
     * @param listener_type A listener type
     * @param listener A pointer to a listener object
     *
     * @endif
     */
    void removeConnectorDataListener(ConnectorDataListenerType listener_type,
                                     ConnectorDataListener* listener);
    

    /*!
     * @if jp
     * @brief ConnectorListener ꥹʤɲä
     *
     * Хåե񤭹ߤޤɤ߽Ф٥Ȥ˴ϢƼꥹʤꤹ롣
     *
     * ǤꥹʤΥפ
     *
     * - ON_BUFFER_EMPTY:       Хåեξ
     * - ON_BUFFER_READTIMEOUT: ХåեǥॢȤ
     *
     * ꥹʤϰʲΥ˥ operator() Ƥɬפ롣
     *
     * ConnectorListener::operator()(const ConnectorProfile&)
     *
     * ǥեȤǤϡδؿͿꥹʥ֥Ȥνͭ
     * OutPort˰ܤꡢOutPortλ⤷ϡ
     * removeConnectorListener() ˤ˼ưŪ˲Τ롣
     * ꥹʥ֥ȤνͭƤӽФ¦ǰݻϡ3
     *  false ꤷưŪʲΤ뤳ȤǤ롣
     *
     * @param listener_type ꥹʥ
     * @param listener ꥹʥ֥ȤؤΥݥ
     * @param autoclean ꥹʥ֥ȤμưŪΤԤɤΥե饰
     *
     * @else
     * @brief Adding ConnectorListener type listener
     *
     * This operation adds certain listeners related to buffer writing and
     * reading events.
     * The following listener types are available.
     *
     * - ON_BUFFER_EMPTY:       At the time of buffer empty
     * - ON_BUFFER_READTIMEOUT: At the time of buffer read timeout
     *
     * Listeners should have the following function operator().
     *
     * ConnectorListener::operator()(const ConnectorProfile&)
     *  
     * The ownership of the given listener object is transferred to
     * this OutPort object in default.  The given listener object will
     * be destroied automatically in the OutPort's dtor or if the
     * listener is deleted by removeConnectorListener() function.
     * If you want to keep ownership of the listener object, give
     * "false" value to 3rd argument to inhibit automatic destruction.
     *
     * @param listener_type A listener type
     * @param listener A pointer to a listener object
     * @param autoclean A flag for automatic listener destruction
     *
     * @endif
     */
    void addConnectorListener(ConnectorListenerType callback_type,
                              ConnectorListener* listener,
                              bool autoclean = true);

    /*!
     * @if jp
     * @brief ConnectorDataListener ꥹʤ
     *
     * ꤷƼꥹʤ롣
     * 
     * @param listener_type ꥹʥ
     * @param listener ꥹʥ֥ȤؤΥݥ
     *
     * @else
     * @brief Removing BufferDataListener type listener
     *
     * This operation removes a specified listener.
     *     
     * @param listener_type A listener type
     * @param listener A pointer to a listener object
     *
     * @endif
     */
    void removeConnectorListener(ConnectorListenerType callback_type,
                                 ConnectorListener* listener);

    /*!
     * @if jp
     * @brief endian ֤
     *
     * endian bool֤ͤ
     *
     * @return m_littleEndian littleξtruebigξfalse ֤
     *
     * @else
     * @brief
     *
     * return it whether endian setting.
     *
     *@return Return true in the case of "little", false in "big" than it.
     *
     * @endif
     */
    bool isLittleEndian();

    /*!
     * @if jp
     *
     * @brief [CORBA interface] Port ³Ԥ
     *
     * Ϳ줿 ConnectoionProfile ξ˴ŤPort֤³Ω
     * 롣δؿϼ˥ץꥱץġ뤫Ƥӽ
     * ȤȤƤ롣
     * 
     * @param connector_profile ConnectorProfile
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief [CORBA interface] Connect the Port
     *
     * This operation establishes connection according to the given
     * ConnectionProfile inforamtion. This function is premised on
     * calling from mainly application program or tools.
     *
     * @param connector_profile The ConnectorProfile.
     * @return ReturnCode_t The return code of ReturnCode_t type.
     *
     * @endif
     */
    virtual ReturnCode_t
    connect(ConnectorProfile& connector_profile)
      throw (CORBA::SystemException);


  protected:
    /*!
     * @if jp
     * @brief OutPortԤ
     *
     * propertiesξ˴ŤOutPortγƼԤ
     *
     * @else
     *
     * @brief Configureing outport
     *
     * This operation configures the outport based on the properties.
     *
     * @endif
     */
    void configure();

    /*!
     * @if jp
     *
     * @brief Interface 
     *
     * Υڥ졼ϡnotify_connect() 󥹤λϤ˥
     * 貾۴ؿǤ롣
     * notify_connect() Ǥϡ
     *
     * - publishInterfaces()
     * - connectNext()
     * - subscribeInterfaces()
     * - updateConnectorProfile()
     *
     * ν protected ؿ뤵³Ԥ롣
     * <br>
     * Υڥ졼ϡ connector_id ФƤ³
     * ¸ connector_id ФƤϹŬڤ˹Ԥɬפ롣
     *
     * @param connector_profile ³˴ؤץե
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Publish interface information
     *
     * This operation is pure virutal function that would be called at the
     * beginning of the notify_connect() process sequence.
     * In the notify_connect(), the following methods would be called in order.
     *
     * - publishInterfaces()
     * - connectNext()
     * - subscribeInterfaces()
     * - updateConnectorProfile() 
     *
     * This operation should create the new connection for the new
     * connector_id, and should update the connection for the existing
     * connection_id.
     *
     * @param connector_profile The connection profile information
     *
     * @return The return code of ReturnCode_t type.
     *
     * @endif
     */
    virtual ReturnCode_t
    publishInterfaces(ConnectorProfile& connector_profile);
    
    /*! @if jp
     *
     * @brief Interface ³
     *
     * Υڥ졼ϡnotify_connect() 󥹤֤˥
     * 貾۴ؿǤ롣
     * notify_connect() Ǥϡ
     *
     * - publishInterfaces()
     * - connectNext()
     * - subscribeInterfaces()
     * - updateConnectorProfile()
     *
     * ν protected ؿ뤵³Ԥ롣
     *
     * @param connector_profile ³˴ؤץե
     *
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Subscribe to the interface
     *
     * This operation is pure virutal function that would be called at the
     * middle of the notify_connect() process sequence.
     * In the notify_connect(), the following methods would be called in order.
     *
     * - publishInterfaces()
     * - connectNext()
     * - subscribeInterfaces()
     * - updateConnectorProfile()
     *
     * @param connector_profile The connection profile information
     *
     * @return The return code of ReturnCode_t type.
     *
     * @endif
     */
    virtual ReturnCode_t
    subscribeInterfaces(const ConnectorProfile& connector_profile);
    
    /*!
     * @if jp
     *
     * @brief Interface ³
     *
     * Υڥ졼ϡnotify_disconnect() 󥹤ν˥
     * 貾۴ؿǤ롣
     * notify_disconnect() Ǥϡ
     * - disconnectNext()
     * - unsubscribeInterfaces()
     * - eraseConnectorProfile()
     * ν protected ؿ뤵³Ԥ롣
     *
     * @param connector_profile ³˴ؤץե
     *
     * @else
     *
     * @brief Disconnect the interface connection
     *
     * This operation is pure virutal function that would be called at the
     * end of the notify_disconnect() process sequence.
     * In the notify_disconnect(), the following methods would be called.
     * - disconnectNext()
     * - unsubscribeInterfaces()
     * - eraseConnectorProfile() 
     *
     * @param connector_profile The profile information associated with 
     *                          the connection
     *
     * @endif
     */
    virtual void
    unsubscribeInterfaces(const ConnectorProfile& connector_profile);

    /*!
     * @if jp
     * @brief OutPort provider ν
     * @else
     * @brief OutPort provider initialization
     * @endif
     */
    void initProviders();

    /*!
     * @if jp
     * @brief InPort consumer ν
     * @else
     * @brief InPort consumer initialization
     * @endif
     */
    void initConsumers();

    bool checkEndian(const coil::Properties& prop, bool& littleEndian);

    /*!
     * @if jp
     * @brief OutPort provider 
     * @else
     * @brief OutPort provider creation
     * @endif
     */
    OutPortProvider* createProvider(ConnectorProfile& cprof,
                                    coil::Properties& prop);
    /*!
     * @if jp
     * @brief InPort consumer 
     * @else
     * @brief InPort consumer creation
     * @endif
     */
    InPortConsumer* createConsumer(const ConnectorProfile& cprof,
                                   coil::Properties& prop);
    
    /*!
     * @if jp
     * @brief OutPortPushConnector 
     * @else
     * @brief OutPortPushConnector creation
     * @endif
     */
    OutPortConnector* createConnector(const ConnectorProfile& cprof,
                                      coil::Properties& prop,
                                      InPortConsumer* consumer);
    /*!
     * @if jp
     * @brief OutPortPullConnector 
     * @else
     * @brief OutPortPullConnector creation
     * @endif
     */
    OutPortConnector* createConnector(const ConnectorProfile& cprof,
                                      coil::Properties& prop,
                                      OutPortProvider* provider);

  protected:
    /*!
     * @if jp
     * @brief ץѥƥ
     * @else
     * @brief Properties
     * @endif
     */
    coil::Properties m_properties;
    /*!
     * @if jp
     * @brief ³ꥹ
     * @else
     * @brief Connection list
     * @endif
     */
    std::vector<OutPortConnector*> m_connectors;
    /*!
     * @if jp
     * @brief Ѳǽprovider
     * @else
     * @brief Available providers
     * @endif
     */
    coil::vstring m_providerTypes;
    /*!
     * @if jp
     * @brief Ѳǽconsumer
     * @else
     * @brief Available consumers
     * @endif
     */
    coil::vstring m_consumerTypes;
    /*!
     * @if jp
     * @brief ³ǥ 
     * @else
     * @brief Connected Endian
     * @endif
     */
    bool m_littleEndian;
    /*!
     * @if jp
     * @brief ConnectorDataListener ꥹ
     * @else
     * @brief ConnectorDataListener listener
     * @endif
     */
    ConnectorListeners m_listeners;
   
    /*!
     * @if jp
     * @brief provider 뤿 Functor
     * @else
     * @brief Functor to delete providers
     * @endif
     */
    struct provider_cleanup;

    /*!
     * @if jp
     * @brief connector 뤿 Functor
     * @else
     * @brief Functor to delete connectors
     * @endif
     */
    struct connector_cleanup;
  };
}; // End of namespace RTC
#endif // RTC_RTCOUTPORTBASE_H
