// -*- C++ -*-
/*!
 * @file  CorbaPort.h
 * @brief CorbaPort class
 * @date  $Date: 2007-12-31 03:08:02 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2009
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: CorbaPort.h 1753 2010-01-22 06:57:12Z fsi-katami $
 *
 */

#ifndef RTC_CORBAPORT_H
#define RTC_CORBAPORT_H

#include <rtm/PortBase.h>
#include <rtm/CorbaConsumer.h>
#include <rtm/NVUtil.h>
#include <list>

/*!
 * @if jp
 * @namespace RTC
 *
 * @brief RTݡͥ
 *
 * @else
 *
 * @namespace RTC
 *
 * @brief RT-Component
 *
 * @endif
 */
namespace RTC
{
  /*!
   * @if jp
   * @class CorbaPort
   * @brief RT ݡͥ CORBA provider/consumer  Port
   *
   * CorbaPort  RT ݡͥȤˤơ桼 CORBA ֥
   * ȥӥӥ󥷥塼ޤ󶡤 Port Ǥ롣
   *
   * <p>
   * RT ݡͥȤϡPort 𤷤ƥ桼 CORBA ӥ
   * 󶡤뤳ȤǤ RT Service (Provider) ȸƤ֡ޤ
   * ¾ RT ݡͥȤΥӥѤ뤿 CORBA ֥
   * ȤΥץ졼ۥ󶡤뤳ȤǤ RT Service
   * Consumer ȸƤ֡
   * <p>
   * CorbaPort Ǥդο Provider  Consumer 뤳Ȥ
   * Port ƱΤ³ݤб Provider  Consumer Ŭڤ
   * Ϣդ뤳ȤǤ롣
   * <p>
   * CorbaPort ̾ʲΤ褦Ѥ롣
   *
   * <pre>
   * RTC::CorbaPort m_port0; // Port 
   *
   * MyService_impl m_mysvc0; //  Port 󶡤 Serivce Provider
   * RTC::CorbaConsumer<YourService> m_cons0; //  Port  Consumer
   *
   * // Service Provider  Port Ͽ
   * m_port0.registerProvider("MyService0", "Generic", m_mysvc0);
   * // Service Consumer  Port Ͽ
   * m_port0.registerConsumer("YourService0", "Generic", m_cons0 );
   *
   * // connect Ԥ줿
   *
   * m_cons0->your_service_function(); // YourService δؿ򥳡
   *
   * // connect 줿 ̤ΥݡͥȤˤ
   * m_cons1->my_service_function(); // MyService δؿ򥳡
   * </pre>
   *
   * Τ褦ˡ󶡤 Service Provider  registerProvider() 
   * Ͽ뤳Ȥˤꡢ¾ΥݡͥȤѲǽˤ¾Ѥ
   *  Service Consumer  registerConsumer() Ͽ뤳Ȥˤ¾
   * ΥݡͥȤ Service 򥳥ݡͥѲǽˤ뤳
   * Ǥ롣
   *
   * PortInterfaceProfile  Port ˽°ץХ⤷ϥ󥷥塼
   * ޥ󥿡եˤĤƤξ򵭽Ҥ뤿ΥץեǤ롣
   * ³Ԥġϡξ˴Ť ConnectorProfile Ŭ
   * ³ Port ΤǤդΰĤФư
   * ConnectorProfile Ϳ Port::connect() ƤӽФɬפ롣
   *
   * ʤPortInterfaceProfile Υ󥹥̾ "*" üʥ󥹥
   * ɽ
   *
   * PROVIDEDʤץХΥ󥹥̾ "*" ξϡ³
   * ϻǤϥ󥹥󥹤¸ߤ󥷥塼ޤ׵˱ưŪ
   * 󥹥󥹤륿פΥץХǤ뤳Ȥɽ
   * ơ³ϻǤϥ󥹥̾¸ߤʤ³
   * Υ󥿡եץˤơץХ
   * 󥹥󥹤бһҤ ConnectorProfile Ŭꤹ
   * ΤȤ롣
   *
   * REQUIREDʤ󥷥塼ޤΥ󥹥̾ "*" ξϡ
   * Υ󥷥塼ޤʣΥץХ³ǽʥפΥ󥷥塼ޤ
   * 뤳Ȥ򼨤(̤)
   *
   * ʲϡPort֤Υ󥿡ե³뤿 ConnectorProfile 
   * ޥåԥ󥰤򵭽Ҥ뤿Υ롼򼨤
   *
   * Port°륤󥿡եλҤΥեޥåȤʲΤ褦
   * 롣󥿡ե˴ؤץѥƥʲξ
   *
   * - RTC󥹥̾:              rtc_iname
   * - ݡ̾:                       port_name
   * - 󥿡ե:           if_polarity
   * - 󥿡ե̾:           if_tname
   * - 󥿡ե󥹥̾: if_iname
   * 
   * 󥿡եλҤʲʸ̾ΤǻꤹΤȤ롣
   *
   * <rtc_iname>.port.<port_name>.<if_polarity>.<if_tname>.<if_iname>
   *
   * PROVIDED()ʤץХΥ󥿥եΥץѥƥ
   * ξ硢
   *
   * - rtc_iname   = MyComp0
   * - port_name   = myservice
   * - if_polarity = provided
   * - if_tname    = echo_interface
   * - if_iname    = echo_interface2
   *
   * 󥿡եҤ
   *
   * MyComp0.port.myservice.provided.echo_interface.echo_interface2
   *
   * Τ褦˵Ҥ롣ޤƱͤREQUIRED(׵)ʤ󥷥塼
   * ޤΥ󥿡եΥץѥƥʲξ硢
   *
   * - rtc_iname   = YourComp0
   * - port_name   = yourservice
   * - if_polarity = required
   * - if_tname    = hoge_interface
   * - if_iname    = hoge_interface1
   *
   * 󥿡եҤϡ
   * 
   * YourComp0.port.myservice.required.hoge_interface.hoge_inteface1
   *
   * Τ褦˵Ҥ뤳ȤǤ롣
   * 
   * ʤưŪ󥿡եΥ󥹥󥹤Τüʥ
   * פΥ󥹥̾һ
   *
   * - <type_name>*: ưŪ󥹥̾һ
   * - <type_name>+: 󥯥󥿥󥹥̾һ
   *
   * 롣ưŪ󥿡եȤϡ³˥󥹥󥹤
   * 륿פΥ󥿡եǤ롣(̤)
   *
   * 󥷥塼ޤ׵᤹ץХ󥿡եһҤưŪ
   * 󥹥̾һ "<type_name>*" ꤵ줿硢ץХ
   * 󥹥󥹤1Ŀ롣"<type_name>*" εһҤˤץ
   * ׵᤹ n ĤΥ󥷥塼ޤ¸ߤ硢餫׵
   * (ڥ졼󥳡)1 ĤΥץХˤطۤ
   * ()
   *
   * <pre>
   * consumer0 ]---<
   * consumer1 ]---<  O----[ provider0
   * consumer2 ]---<
   * </pre>
   *  
   * Ф󥷥塼ޤ׵᤹ץХ󥿡եһ
   * ˥󥯥󥿥󥹥̾һ "<type_name>+" 
   * 줿硢һ "<type_name>+" οץХΥ󥹥
   * ưŪ롣ʤ"<type_name>+" εһҤˤץ
   * ׵᤹ n ĤΥ󥷥塼ޤ¸ߤ硢n ĤΥץХ
   * 줾׵ʲΤ褦ʴطۤ롣
   *
   * <pre>
   * consumer0 ]---<  O----[ provider0
   * consumer1 ]---<  O----[ provider1
   * consumer2 ]---<  O----[ provider2
   * </pre>
   *
   *
   * ³˺ݤơġ ConnectorProfile::properties Ŭڤʥ
   * 󥿡եޥåԥ󥰻򵭽Ҥ뤳ȤǡߤΥץХ/
   * 󥷥塼ޥ󥿡եͳ³뤳ȤǤ롣
   * ³˴ؤ RTC ˡۤʤ륤󥹥󥹤Ǥʤ顢ƱΥ
   * ̾¸ߤ硢󥿡եһҤΰݾڤǤ
   * Τǡˡˤ³ݾڤʤ
   *
   * ǥ󥿡եһҤñΤ <if_desc0>,
   * <if_desc1>, ...  Ȥ롣ޤConnectorProfile::properties 
   * NVList key  value  key: value Τ褦˵ҤΤȤ롣
   *
   * ޡ2ĤΥݡͥȤΥӥݡȤ³ͤ롣
   * 줾ΥݡͥȤΥӥݡȤʲξ硢
   *
   * - rtc_iname: MyComp0        <br>
   *   port_name: mycomp_service <br>
   *   interfaces:
   *   - if_polarity: provided   <br>
   *     if_iname: echo0         <br>
   *     if_tname: Echo
   *   - if_polarity: required   <br>
   *     if_iname: add0          <br>
   *     if_tname: add
   *
   * - rtc_iname: YourComp0        <br>
   *   port_name: yourcomp_service <br>
   *   interfaces:
   *   - if_polarity: required     <br>
   *     if_iname: echo9           <br>
   *     if_tname: Echo
   *   - if_polarity: provided     <br>
   *     if_iname: add9            <br>
   *     if_tname: add
   *
   * <pre>
   *      MyComp0                                 YourComp0
   *     _______ mycomp_service   yourcomp_service ______
   *            |                                 |
   *          |~~~|---O echo0         echo9 >---|~~~|
   *          |   |---< add0          add9  O---|   |
   *           ~T~                               ~T~
   *            |                                 |
   * </pre>
   *
   * MyComp0  echo0 (ץХ)  YourComp0  echo9 (󥷥塼)
   * MyComp0  add0 (󥷥塼)  YourComp0  echo9 (ץХ)
   * 򤽤줾Фˤ³ΤȲꤹ롣ξ硢
   * ConnectorProfile ϰʲΤ褦ꤹ롣
   * 
   * <pre>
   * ConnectorProfile:
   *   name: ǤդΥͥ̾
   *   connector_id: ʸ
   *   ports[]: mycomp_service λ, yourcomp_service λ
   *   properties:
   *     <add0>: <add9>
   *     <echo9>: <echo0>
   * </pre>
   *
   * 줾
   * 
   * <pre>
   * <add0>  MyComp0.port.mycomp_service.required.add.add0
   * <add9>  YourComp0.port.yourcomp_service.provided.add.add9
   * <echo0>  MyComp0.port.mycomp_service.provided.echo.echo0
   * <echo9>  YourComp0.port.yourcomp_service.required.echo.echo9
   * </pre>
   *
   * Ǥ롣³ץˤơƥݡȤΥץХӥ󥷥塼
   * ޤϡ줾ʲκȤCorbaPort::publishInterfaces(),
   * CorbaPort::PortsubscribeInterfaces() ۴ؿˤƹԤ
   *
   * ץХϡpublishInterfaces() ؿˤơʬΥ󥿡ե
   * һҤ򥭡ȤͤIORʸɽΤ
   * ConnectorProfile::properties ꤹ롣ȤơΥ󥿡
   * եһҤϺԤȤƤ륳ͥˤƤϰդǤ뤿ᡢ
   * Ʊ1Ĥ¸ߤƤϤʤ
   *
   * [ʬεҤ̤εǽ] ʤưŪ󥿡եˤ
   * Ƥϡʲμ³˽뤳ȤȤʤ롣publishInterface()
   * ؿˤơưŪ󥹥̾һ "<type_name>*" ޤϡ
   * 󥯥󥿥󥹥̾һ "<type_name>+" ¸ߤ
   * 뤫ɤ롣ưŪ󥹥̾һ "<type_name>*"
   * ¸ߤ硢ץХΥ󥹥󥹤1Υ󥿡
   * եҤ key ˡIORʸ value ꤹȤȤˡư
   * Ū󥹥̾һ "<type_name>*"  value ˴ޤह٤Ƥ
   * value Υ󥿡եҤ򡢤󥿡ե
   * Ҥ֤롣
   * 
   * 󥯥󥿥󥹥̾һ"<type_name>+" ¸ߤ
   * 硢󥹥̾һҤοץХΥ󥹥󥹤
   * 줾Υ󥿡եҤkey ˡIORʸ value 
   * ꤹȤȤˡ󥯥󥿥󥹥̾һ
   * "<type_name>+"  value ޤह٤Ƥ value Υ󥿡ե
   * ҤФƽˡ줾Υ󥿡եҤ
   * ֤롣
   *
   * ץХ subscribeInterfaces() ǤäϹԤʤ
   *
   * 󥷥塼ޤϡpublishInterfaces() ˤƤäԤʤ
   *
   * subscribeInterfaces() ǤϡʬεһҤ key Ȥ
   * key-value ڥ ¸ߤ뤫ɤĴ١⤷¸ߤС value
   * ꤵ줿ץХΥ󥿡եҤǻꤵ뻲Ȥ
   *  ConnectorProfile::properties õ򥳥󥷥塼ޤ
   * ³Ȥꤹ롣ʤտŪ˥󥷥塼ޤ˥ץХλ
   * ꤷʤϡͽʸ "nil" ޤ "null" ꤹ
   * Ȥ롣
   *
   * 󥷥塼ޤϡ⤷ʬεһҤ¸ߤʤ硢ޤϥץХ
   * λȤ Connector::properties ¸ߤʤ硢󥷥塼ޤϡ
   * ʬΥ󥹥̾ӷ̾ƱΥץХõλȤ
   * ʬȤꤹ롣ϡOpenRTM-aist-0.4 Ȥθߴݻ
   * Υ롼Ǥꡢ1.0ʹߤǤϿ侩ʤ
   *
   * ץХХ󥷥塼ޤбϰаǤɬפϤʤץХ
   * 1 Фơ󥷥塼 nޤϥ󥷥塼 1 ФƥץХ
   *  n Υ롣ץХ 1 Фơ󥷥塼 n 
   * ǤϡץХλҤʣΥ󥷥塼ޤФơ
   * 嵭ˡǻꤵ뤳Ȥˤꡢ¸롣󥷥塼
   * 1 ФƥץХ n ΥǤϡ󥷥塼޻Ҥ key 
   * ơʣΥץХλҤ޶ڤ󤵤Ȥʤ
   * ΤȤ롣
   *
   * ʤ󥿡եбطθ̩ꤹ륪ץȤơ
   * ʲΥץꤹ뤳ȤǤ롣
   *
   * port.connection.strictness: strict, best_effort 
   *
   * strict: ٤ƤΥ󥷥塼ޤ˻ꤷȤ¸ߤĥʥ
   *         ˤ󥷥塼ޤŬڤ˥åȤǤˤΤ Port
   *         ֤³Ω롣
   *
   * best_effort: ʥ˼ԤǤ⡢顼֤Ȥ
   *          Port ֤³Ω롣
   *
   * @since 0.4.0
   *
   * @else
   * @class CorbaPort
   * @brief RT Conponent CORBA service/consumer Port
   *
   * CorbaPort is an implementation of the Port of RT-Component's that provides
   * user-defined CORBA Object Service and Consumer.
   * <p>
   * RT-Component can provide user-defined CORBA serivces, which is called
   * RT-Serivce (Provider), through the Ports.
   * RT-Component can also provide place-holder, which is called RT-Serivce
   * Consumer, to use other RT-Component's service.
   * <p>
   * The CorbaPort can manage any number of Providers and Consumers, can
   * associate Consumers with correspondent Providers when establishing
   * connection among Ports.
   * <p>
   * Usually, CorbaPort is used like the following.
   *
   * <pre>
   * RTC::CorbaPort m_port0; // declaration of Port
   *
   * MyService_impl m_mysvc0; // Serivce Provider that is provided by the Port
   * RTC::CorbaConsumer<YourService> m_cons0; // Consumer of the Port
   *
   * // register Service Provider to the Port
   * m_port0.registerProvider("MyService0", "Generic", m_mysvc0);
   * // register Service Consumer to the Port
   * m_port0.registerConsumer("YourService0", "Generic", m_cons0 );
   *
   * // after connect established
   *
   * m_cons0->your_service_function(); // call a YourService's function
   *
   * // in another component that is connected with the Port
   * m_cons1->my_service_function(); // call a MyService's function
   * </pre>
   *
   * Registering Service Provider by registerProvider(), it can be
   * used from other RT-Components.  Registering Service Consumer by
   * registerConsumer(), other RT-Component's services can be used
   * through the consumer object.
   *
   * PortInterfaceProfile is a one of the profile information to store
   * Provider interface and Consumer interface information. Tools or
   * other RTCs should call one of the Port::connect() with an
   * appropriate ConnectorProfile.
   *
   * In addition, the instance name "*" declares a special type of instance.
   *
   * When the name of the PROVIDED type interface that is the provider
   * interface is "*", Provider interface's instance does not exist at
   * the beginning of connection sequence.  The instances will be
   * created dynamically according to the consumer interface
   * requirement at the connection sequence.  Although the instance
   * name does not exist at the beginning of connection sequence, the
   * created providers shall publish its references to the
   * ConnectorProfile with interface descriptor adequately in the
   * interface publisher phase of the connection sequence.
   *
   * If REQUIRED interface name that is Consumer interface name is
   * "*", it shows that one Consumer interface is able to connect with
   * multiple Provider interfaces. (This feature is not implemented.)
   * 
   * The following describes the rules that specify interface
   * connection between ports.
   *
   * The descriptor format of interfaces associated with Ports is
   * declared as follows. Now some of interface properties are assumed
   * as the followings.
   *
   * - RTC instance name:              rtc_iname
   * - Port name:                      port_name
   * - Interface polarity:             if_polarity
   * - Interface type name:            if_tname
   * - INterface instance name:        if_iname
   *
   * The interface descriptors shall be declared as follows.
   *
   * <rtc_iname>.port.<port_name>.<if_polarity>.<if_tname>.<if_iname>
   *
   * When PROVIDED that is Provider interface properties are the followings,
   *
   * - rtc_iname   = MyComp0
   * - port_name   = myservice
   * - if_polarity = provided
   * - if_tname    = echo_interface
   * - if_iname    = echo_interface2
   * the interface descriptor is here.
   *
   * MyComp0.port.myservice.provided.echo_interface.echo_interface2
   *
   * And, when REQUIRED that is Consumer interfaces properties are the
   * followings,
   *
   * - rtc_iname   = YourComp0
   * - port_name   = yourservice
   * - if_polarity = required
   * - if_tname    = hoge_interface
   * - if_iname    = hoge_interface1
   *
   * interface descriptor is as follows. 
   *
   * YourComp0.port.myservice.required.hoge_interface.hoge_inteface1
   *
   * Specific instance name descriptors that are dynamically generated
   * at the connection time are defined here.
   *
   * - <type_name>*: "Dynamically generated" instance descriptor.
   * - <type_name>+: "Incrementally generated" instance descriptor.
   *
   * When the "Dynamically generated" instance descriptor:
   * "<type_name>*" is specified as interface descriptor that is
   * required by consumers, the provider will generate a instance. If
   * n consumers who demand a provider by the "<type_name>" descriptor
   * exist, the following relation which processes the call from these
   * consumers by one provider will be established.
   *
   * <pre>
   * consumer0 ]---<
   * consumer1 ]---<  O----[ provider0
   * consumer2 ]---<
   * </pre>
   *  
   * On the other hand, when incremental generated type instance name
   * descriptor "<type_name>+" is specified as the provider interface
   * descriptor whom consumers demand, provider's instances are
   * dynamically generated for the number of the descriptors
   * "<type_name>+". When n consumers who demand a provider by the
   * descriptor "<type_name>+" exist the following relations in which
   * n providers process each call from the consumers will be
   * established.
   *
   * <pre>
   * consumer0 ]---<  O----[ provider0
   * consumer1 ]---<  O----[ provider1
   * consumer2 ]---<  O----[ provider2
   * </pre>
   *
   *
   * Describing the appropriate interface mapping specification in the
   * ConnectorProfile::properties, selective connections between
   * providers/consumers interface can be established at the time of
   * connection. However, when different RTC instances of the same
   * instance name exist in a connection, since an interface
   * descriptor uniqueness cannot be guaranteed, this connection
   * mapping rules cannot be used.
   *
   * Here, assume that an interface descriptor is given as <if_desc0>,
   * <if_desc1>, .... And assume that the key and the value of NVList
   * in ConnectorProfile::properties are given as "key: value".
   *
   * Now the case where the service ports of two components are
   * connected is considered. When the service port of each component
   * is the following,
   * 
   * - rtc_iname: MyComp0          <br>
   *   port_name: mycomp_service   <br>
   *   interfaces:
   *   - if_polarity: provided     <br>
   *     if_iname: echo0           <br>
   *     if_tname: Echo
   *   - if_polarity: required     <br>
   *     if_iname: add0            <br>
   *     if_tname: add
   *
   * - rtc_iname: YourComp0        <br>
   *   port_name: yourcomp_service <br>
   *   interfaces:
   *   - if_polarity: required     <br>
   *     if_iname: echo9           <br>
   *     if_tname: Echo
   *   - if_polarity: provided     <br>
   *     if_iname: add9            <br>
   *     if_tname: add
   *
   *
   * <pre>
   *      MyComp0                                 YourComp0
   *     _______ mycomp_service   yourcomp_service ______
   *            |                                 |
   *          |~~~|---O echo0         echo9 >---|~~~|
   *          |   |---< add0          add9  O---|   |
   *           ~T~                               ~T~
   *            |                                 |
   * </pre>
   * 
   *
   *
   * Assume that connection between echo0 (provider) of MyComp0
   * component and echo9 (consumer) of YourComp0 component, and add0
   * (consumer) of MyComp0 and add0 (provider) of YourComp0 is
   * established.  In this case, ConnectorProfile is set up as
   * follows.
   * 
   * <pre>
   * ConnectorProfile:
   *   name: any connector name
   *   connector_id: empty string
   *   ports[]: mycomp_service's reference, yourcomp_service's reference
   *   properties:
   *     <add0>: <add9>
   *     <echo9>: <echo0>
   * </pre>
   *
   * Please note that <add0>, <add9>, <echo0> and <echo9> are the following.
   * 
   * <pre>
   * <add0> is MyComp0.port.mycomp_service.required.add.add0
   * <add9> is YourComp0.port.yourcomp_service.provided.add.add9
   * <echo0> is MyComp0.port.mycomp_service.provided.echo.echo0
   * <echo9> is YourComp0.port.yourcomp_service.required.echo.echo9
   * </pre>
   *
   * In the connection process, the provider and the consumer of each
   * port carries out the following process respectively in the
   * virtual functions such as CorbaPort::publishInterfaces() and
   * CorbaPort::subscribeInerfaces().
   * 
   * A provider sets its IOR string as a value and its interface
   * descriptor as a key in the ConnectorProfile::properties in a
   * publishInterfaces() function. Since this interface descriptor's
   * uniqueness is guaranteed in the current connector, the key of
   * NameValue in the ConnectorProfile::properties is unique.
   *
   *
   * [This functionalities are not implemented] The dynamically
   * generated provider is processed according to the following
   * procedure. The publishInterface() function scans dynamic instance
   * descriptors such as "<type_name>*" and "<type_name>+" in the
   * ConnectorProfile::properties. When the dynamic generation
   * instance descriptor "<tupe_name>*" exists, one instance of
   * provider is generated, and its descriptor and its IOR string are
   * set to ConnectorProfile::properties as the key and the value
   * respectively. Simultaneously, in the
   * ConnectorProfile::properties, all the instance descriptor with
   * the dynamic generation instance name "<type_name>*" will be
   * replaced with newly generated instance descriptor.
   *
   * When the incremental dynamic generation instance descriptor
   * exists, providers are generated for the number of the
   * descriptors, and its descriptor and its IOR string are set to
   * ConnectorProfile::properties as the key and the value
   * respectively. Simultaneously, in the
   * ConnectorProfile::properties, all the instance descriptor with
   * the dynamic generation instance name "<type_name>+" will be
   * replaced with newly generated instance descriptor.
   *
   * The providers do not perform particular operation in
   * subscribeInterfaces() function.
   *
   *
   * The consumers do not perform particular operation in
   * publisherInterfaces() function.
   *
   * On the other hand, a consumer searches a key-value pair with the
   * key of consumer interface descriptor, and if the pair exists, it
   * obtains provider's descriptor from the value. The consumer
   * searches again a key-value pair with the key of provider
   * interface descriptor, and it obtains provider's reference and the
   * reference is set as the consumer's service object. In addition,
   * reserved string "nil" or "null" are used not to set specific
   * provider.
   *
   * If consumer's interface descriptors does not exists in the
   * ConnectorProfile::properties, the consumer searches a provider
   * with same type name and instance name, and its reference is set
   * to the consumer. This rule is for only backward compatibility,
   * and it is not recommended from version 1.0.
   *
   * The correspondence of a provider versus a consumer does not need
   * to be one to one, and the case of one provider to n-consumers and
   * the case of m-providers to one consumer are allowed. The one
   * provider to n-consumers case can be realized by the above
   * mentioned methods. The one consumer to m-provider case can be
   * specified to set the consumer descriptor and comma-separated
   * provider descriptors into the key and the value respectively.
   *
   * The following option is available to specify the strictness of
   * interfaces connection.
   *
   * port.connection.strictness: strict, best_effort
   *
   * strict: The connection is established, if only all the specified
   *         consumers are set appropriate references and narrowed
   *         successfully.  
   *
   * best_effort: The connection is established without any errors,
   *         even if appropriate reference does not exist or reference
   *         narrowing fails.
   *
   * @since 0.4.0
   *
   * @endif
   */
  class CorbaPort
    : public PortBase
  {
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * @param name Port ̾
     *
     * @else
     *
     * @brief Constructor
     *
     * @param name The name of Port 
     *
     * @endif
     */
    CorbaPort(const char* name);
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * @endif
     */
    virtual ~CorbaPort(void);
    
    /*!
     * @if jp
     * @brief ץѥƥν
     *
     * OutPortΥץѥƥ
     *
     * @else
     *
     * @brief Initializing properties
     *
     * This operation initializes outport's properties
     *
     * @endif
     */
    void init(coil::Properties& prop);

    /*!
     * @if jp
     *
     * @brief Provider Ͽ
     *
     *  Port ˤ󶡤ХȤ򤳤 Port ФϿ롣
     * ХȤϡͿ instance_name, type_name 
     * ХȼȤΥ󥹥̾ӥ̾ȤơХȤ
     * Ϣդ롣
     *
     * @param instance_name ХȤΥ󥹥̾
     * @param type_name ХȤΥ̾
     * @param provider CORBA Х
     *
     * @return Ʊ̾ instance_name ϿƤ false ֤
     *
     * @else
     *
     * @brief Register the provider
     *
     * This operation registers a servant, which is provided in this Port,
     * to the Port. The servant is associated with "instance_name" and
     * "type_name" as the instance name of the servant and as the type name
     * of the servant.
     *
     * @param instance_name Instance name of servant
     * @param type_name Type name of the servant
     * @param provider CORBA servant
     *
     * @return Return false if the same name of instance_name is already 
     *         registered.
     *
     * @endif
     */
    bool registerProvider(const char* instance_name, const char* type_name,
                          PortableServer::RefCountServantBase& provider);
    
    /*!
     * @if jp
     *
     * @brief Consumer Ͽ
     *
     *  Port ׵᤹륵ӥΥץ졼ۥǤ륳󥷥塼
     * (Consumer) Ͽ롣Consumer Ϣդ륵ӥΥ
     * ̾ӥ̾Ȥơ instance_name, type_name 
     *  Consumer ȤͿ뤳ȤˤꡢǤ餬Ϣդ
     * 롣Port ֤³ (connect)  ˤϡsubscribeInterfaces() ǽ
     * ٤Ƥ롼˴ŤProvider Interface λȤưŪ
     * Consumer ˥åȤ롣
     *
     * @param instance_name Consumer ׵᤹륵ӥΥ󥹥̾
     * @param type_name Consumer ׵᤹륵ӥΥ̾
     * @param consumer CORBA ӥ󥷥塼
     *
     * @return Ʊ̾ instance_name ϿƤ false ֤
     *
     * @else
     *
     * @brief Register the consumer
     *
     * This operation registers a consumer, which is a service
     * placeholder this port requires. These are associated internally
     * with specified instance_name, type_name and Consumer itself to
     * the argument as service's instance name and its type name
     * associated with Consumer.  The service Provider interface'
     * references will be set automatically to the Consumer Interface
     * object when connections are established, according to the rules
     * that are described at the subscribeInterface() function's
     * documentation.
     *
     * @param instance_name Instance name of the service Consumer requires
     * @param type_name Type name of the service Consumer requires
     * @param consumer CORBA service consumer
     *
     * @return False would be returned if the same instance_name was registered
     *
     * @endif
     */
    bool registerConsumer(const char* instance_name, const char* type_name,
                          CorbaConsumerBase& consumer);
    
  protected:
    /*!
     * @if jp
     *
     * @brief Provider Interface 
     *
     *  Port ͭ Provider 󥿡ե˴ؤ
     * ConnectorProfile::properties ¾ Port ФƸ롣
     * RTCΥ󥹥̾ξ󤬰ʲ̤ǤȤơ
     *
     * - RTC󥹥̾:              rtc_iname
     * - ݡ̾:                       port_name
     * - 󥿡ե:           if_polarity
     * - 󥿡ե̾:           if_tname
     * - 󥿡ե󥹥̾: if_iname
     *
     * NameValue  ConnectorProfile::properties  name  value Ȥ
     * ʲΤΤǼ롣
     *
     * - name
     *   <rtc_iname>.port.<port_name>.provided.<if_tname>.<if_iname>
     * - value
     *   Provider 󥿡ե IOR ʸ 
     * 
     * ʤСȤθߴΤʲɽ NameValue Ʊ
     * ˳Ǽ뤬ΥСǤϺǽ롣
     * 
     * - name
     *   port.<if_tname>.<if_iname>
     * - value
     *   Provider 󥿡ե IOR ʸ
     *
     * ͤ ConnectorProfile::properties ˳Ǽ졢¾ΥݡȤФ
     * ã롣¾ Port ǤΥ󥿡եѤ Consumer 
     * ¸ߤСConnectorProfile 餳Υ饪֥ȥե󥹤
     * 餫ηǻѤ롣
     *
     * @param connector_profile ͥץե
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Publish information about interfaces
     *
     * This operation publishes Provider interfaces information, which
     * is owned by this port, to the other Ports via
     * ConnectorProfile::properties.
     * Now it is assumed RTC instance name and other information is as follows,
     *
     * - RTC instance name:              rtc_iname
     * - Port name:                      port_name
     * - Interface polarity:             if_polarity
     * - Interface type name:            if_tname
     * - Interface instance name:        if_iname
     *
     * the following values are stored as the "name" and the "value"
     * of the NameValue typee element in ConnectorProfile::properties.
     *
     * - name
     *   <rtc_iname>.port.<port_name>.provided.<if_tname>.<if_iname>
     * - value
     *   IOR string value of interface reference
     * 
     * In addition, although the following NameValue values are also
     * stored for the backward compatibility, this will be deleted in
     * the future version.
     *
     * - name
     *   port.<if_tname>.<if_iname>
     * - value
     *   IOR string value of interface reference
     *
     * These values are stored in the ConnectorProfile::properties and
     * are propagated to the other Ports. If the Consumer interface
     * exists that requires this Provider interface, it will retrieve
     * reference from the ConnectorProfile and utilize it.
     *
     * @param connector_profile Connector profile
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t
    publishInterfaces(ConnectorProfile& connector_profile);
    
    /*!
     * @if jp
     *
     * @brief Provider Interface 
     *
     *  Portͭ Consumer Interface Ŭ礹 Provider
     * Interface ˴ؤConnectorProfile::properties Ф
     * Consumer Interface ˥֥ȻȤ򥻥åȤ롣
     *
     * RTC Υ󥹥̾ Consumer Interface ξ󤬰ʲΤ
     * ǤȲꤹȡ
     *
     * - RTC󥹥̾:              rtc_iname
     * - ݡ̾:                       port_name
     * - 󥿡ե:           if_polarity
     * - 󥿡ե̾:           if_tname
     * - 󥿡ե󥹥̾: if_iname
     *
     *  Consumer Interface ɽ󥿡եҤϰʲΤ褦
     * ɽ롣
     *
     * <rtc_iname>.port.<port_name>.required.<if_tname>.<if_iname>
     *
     * δؿϡޤ ConnectorProfile::properties ˾嵭󥿡
     * եҤ򥭡ȤƳǼƤ Provider Interface 
     * ҤõФˡ Provider Interface Ҥ򥭡Ȥ
     * ǼƤ Provider Interface λȤɽ IOR ʸ
     * Consumer Interface ˥åȤ롣
     *
     * ˡProvider  prov(n), λȤIOR(n)  Consumer
     * cons(n) Τ褦˵Ҥ餹٤ƤΥ󥿡եηƱ
     * ǤꡢConnectorProfile ˰ʲͤꤵƤȤ롣
     *
     * <pre>
     * ConnectorProfile::properties =
     * {
     *   prov0: IOR0,
     *   prov1: IOR1,
     *   prov2: IOR2,
     *   cons0: prov2,
     *   cons1: prov1,
     *   cons2: prov0
     * }
     * </pre>
     *
     * ΤȤcons(0..2) ˤϤ줾졢ȤʲΤ褦˥åȤ롣
     *
     * <pre>
     *   cons0 = IOR2
     *   cons1 = IOR1
     *   cons2 = IOR0
     * </pre>
     *
     * ʤСȤθߴΤᡢ
     * ConnectorProfile::properties  Consumer Interface 򥭡Ȥ
     * ͤåȤƤʤǤ⡢Υ롼뤬ŬѤ롣
     *
     *  Consumer Interface 
     *
     * <pre>
     *  PortInterfaceProfile
     *  {
     *    instance_name = "PA10_0";
     *    type_name     = "Manipulator";
     *    polarity      = REQUIRED;
     *  }
     * </pre>
     *
     * ȤϿƤС¾ Port 
     *
     * <pre>
     *  PortInterfaceProfile
     *  {
     *    instance_name = "PA10_0";
     *    type_name     = "Manipulator";
     *    polarity      = PROVIDED;
     *  }
     * </pre> 
     *
     * ȤϿƤ Serivce Provider Υ֥ȻȤõ
     * Consumer ˥åȤ롣ºݤˤϡConnectorProfile::properties 
     *
     * <pre>
     * NameValue = { "port.Manipulator.PA10_0": <Object reference> }
     * </pre>
     *
     * ȤϿƤ NameValue õΥ֥ȻȤ
     * Consumer ˥åȤ롣
     *
     * @param connector_profile ͥץե
     * @return ReturnCode_t Υ꥿󥳡
     *
     * @else
     *
     * @brief Subscribe to interface
     *
     * Retrieve information associated with Provider matches Consumer
     * owned by this port and set the object reference to Consumer.
     *
     * Now, Consumer is registered as the following:
     * <pre>
     *  PortInterfaceProfile
     *  {
     *    instance_name = "PA10_0";
     *    type_name     = "Manipulator";
     *    polarity      = REQUIRED;
     *  }
     * </pre>
     * Find the object reference of Serivce Provider that is registered as
     * the following of other ports:
     * <pre>
     *  PortInterfaceProfile
     *  {
     *    instance_name = "PA10_0";
     *    type_name     = "Manipulator";
     *    polarity      = PROVIDED;
     *  }
     * </pre> 
     * and set to Consumer.
     * In fact, find NameValue that is registered as the following to 
     * ConnectorProfile::properties:
     * <pre>
     * NameValue = { "port.Manipulator.PA10_0": <Object reference> }
     * </pre>
     * and set the object reference to Consumer.
     *
     * @param connector_profile Connector profile
     *
     * @return The return code of ReturnCode_t type
     *
     * @endif
     */
    virtual ReturnCode_t
    subscribeInterfaces(const ConnectorProfile& connector_profile);
    
    /*!
     * @if jp
     *
     * @brief Interface ؤ³
     *
     * Ϳ줿 ConnectorProfile ˴Ϣ Consumer ˥åȤ줿
     * ٤Ƥ Object ³롣
     *
     * @param connector_profile ͥץե
     *
     * @else
     *
     * @brief Unsubscribe interfaces
     *
     * Release all Objects that was set in Consumer associated with the given 
     * ConnectorProfile.
     *
     * @param connector_profile Connector profile
     *
     * @endif
     */
    virtual void
    unsubscribeInterfaces(const ConnectorProfile& connector_profile);

    //============================================================
    // Local operations
    //============================================================
    /*!
     * @if jp
     *
     * @brief Port ƤΥ󥿡ե activates 
     *
     * Port ϿƤƤΥ󥿡ե activate 롣
     *
     * @else
     *
     * @brief Activate all Port interfaces
     *
     * This operation activate all interfaces that is registered in the
     * ports.
     *
     * @endif
     */
    virtual void activateInterfaces();

    /*!
     * @if jp
     *
     * @brief Ƥ Port Υ󥿡ե deactivates 
     *
     * Port ϿƤƤΥ󥿡ե deactivate 롣
     *
     * @else
     *
     * @brief Deactivate all Port interfaces
     *
     * This operation deactivate all interfaces that is registered in the
     * ports.
     *
     * @endif
     */
    virtual void deactivateInterfaces();

  protected:
    /*!
     * @if jp
     * @brief ץѥƥ
     * @else
     * @brief Properties
     * @endif
     */
    coil::Properties m_properties;
    
  private:
    class CorbaConsumerHolder;
    virtual bool findProvider(const NVList& nv, CorbaConsumerHolder& cons);
    virtual bool findProviderOld(const NVList&nv, CorbaConsumerHolder& cons);

  private:
    /*!
     * @if jp
     * @class CorbaProviderHolder
     * @brief Provider ξǼ빽¤
     *
     * CORBA Provider Υۥ饹
     *
     * @else
     * @class CorbaProviderHolder
     * @brief The structure to be stored Provider information.
     *
     * CORBA Provider holder class
     *
     * @endif
     */
    class CorbaProviderHolder
    {
    public:
      CorbaProviderHolder(const char* type_name,
                          const char* instance_name,
                          PortableServer::RefCountServantBase* servant)
        : m_typeName(type_name),
          m_instanceName(instance_name),
          m_servant(servant),
          m_ior()
      {  
        m_oid = Manager::instance().getPOA()->servant_to_id(m_servant);
        try
          {
            Manager::instance().
              getPOA()->activate_object_with_id(m_oid, m_servant);
          }
        catch(const ::PortableServer::POA::ServantAlreadyActive &)
          {
            ;
          }
        catch(const ::PortableServer::POA::ObjectAlreadyActive &)
          {
            ;
          }
        CORBA::Object_var obj;
        obj = Manager::instance().getPOA()->id_to_reference(m_oid);
        CORBA::ORB_ptr orb = Manager::instance().getORB();
        CORBA::String_var ior_var = orb->object_to_string(obj);
        m_ior = ior_var;
#ifndef _EM1D512_
        deactivate();
#endif
      }

      std::string instanceName() { return m_instanceName; }
      std::string typeName() { return m_typeName; }
      std::string ior() { return m_ior; }
      std::string descriptor() { return m_typeName + "." + m_instanceName; }

      void activate()
      {
        try
          {
            Manager::instance().
              getPOA()->activate_object_with_id(m_oid, m_servant);
          }
        catch(const ::PortableServer::POA::ServantAlreadyActive &)
          {
            ; // do nothing
          }
        catch(const ::PortableServer::POA::ObjectAlreadyActive &)
          {
            ; // do nothing
          }
      }
      void deactivate()
      {
        try
          {
            Manager::instance().getPOA()->deactivate_object(m_oid);
          }
        catch(const ::PortableServer::POA::ObjectNotActive&)
          {
            ; // do nothing
          }
      }
    private:
      std::string m_typeName;
      std::string m_instanceName;
      PortableServer::RefCountServantBase* m_servant;
      PortableServer::ObjectId_var m_oid;
      std::string m_ior;
    };

    /*!
     * @if jp
     * @brief Provider ξǼ vector
     * @else
     * @brief vector to stored Providers' information
     * @endif
     */
    typedef std::vector<CorbaProviderHolder> CorbaProviderList;
    CorbaProviderList m_providers;

    /*!
     * @if jp
     * @brief Consumer ξǼ빽¤
     * @else
     * @brief The structure to be stored Consumer information.
     * @endif
     */
    class CorbaConsumerHolder
    {
    public:
      CorbaConsumerHolder(const char* type_name,
                          const char* instance_name,
                          CorbaConsumerBase* consumer)
        : m_typeName(type_name),
          m_instanceName(instance_name),
          m_consumer(consumer)
      {
      }
      std::string instanceName() { return m_instanceName; }
      std::string typeName() { return m_typeName; }
      std::string descriptor() { return m_typeName + "." + m_instanceName; }

      bool setObject(const char* ior)
      {
        CORBA::ORB_ptr orb = ::RTC::Manager::instance().getORB();
        CORBA::Object_var obj = orb->string_to_object(ior);
        if (CORBA::is_nil(obj))
          {
            return false;
          }

        return m_consumer->setObject(obj.in());
      }
      void releaseObject()
      {
        m_consumer->releaseObject();
      }

    private:
      std::string m_typeName;
      std::string m_instanceName;
      CorbaConsumerBase* m_consumer;
    };
    typedef std::vector<CorbaConsumerHolder> CorbaConsumerList;
    CorbaConsumerList m_consumers;
    
    // functors
    /*!
     * @if jp
     * @brief Consumer Υ֥Ȥ뤿 Functor
     * @else
     * @brief Functor to release Consumer's object
     * @endif
     */
    struct unsubscribe
    {
      unsubscribe(CorbaConsumerList& consumers)
        : m_consumers(consumers)
      {
      }
      
      void operator()(const SDOPackage::NameValue& nv)
      {
        for (CorbaConsumerList::iterator it(m_consumers.begin());
             it != m_consumers.end(); ++it)
          {
            std::string name(nv.name);
            if (it->descriptor() == (const char*)nv.name)
              {
                it->releaseObject();
              }
          }
      }
      CorbaConsumerList& m_consumers;
    };
  };
};
#endif // RTC_CORBAPORT_H
