' Using EMS in QuickBasic: Part 3 of 3
' Source Code
'
' By Jon Petrosky (Plasma)
' www.phatcode.net
'
' [Remember to start QB with the /L switch to enable interrupts]

DEFINT A-Z
'$DYNAMIC
'$INCLUDE: 'QB.BI'

DIM SHARED Regs AS RegTypeX
DIM SHARED EMS.Error            'Holds the error code of the last operation

DECLARE FUNCTION EMS.ErrorMsg$ ()
DECLARE FUNCTION EMS.Init ()
DECLARE FUNCTION EMS.Version$ ()
DECLARE FUNCTION EMS.PageFrame ()
DECLARE FUNCTION EMS.FreeHandles ()
DECLARE FUNCTION EMS.FreePages ()
DECLARE FUNCTION EMS.TotalPages ()
DECLARE FUNCTION EMS.AllocPages (NumPages)
DECLARE SUB EMS.DeallocPages (Handle)
DECLARE SUB EMS.MapPage (Physical, Logical, Handle)
DECLARE SUB EMS.MapXPages (PhysicalStart, LogicalStart, NumPages, Handle)
DECLARE SUB EMS.CopyMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)
DECLARE SUB EMS.ExchMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

TYPE ArrayEMS             'EMS array information
  Handle AS INTEGER       'Handle EMS pages for the array are allocated to
  ElementLen AS INTEGER   'Length of each element in bytes (2=int, 2=long, etc.)
  NumElements AS LONG     'Number of elements in the array (numbered starting at 1)
END TYPE

DECLARE FUNCTION EMS.Array.Dim (Array AS ArrayEMS)
DECLARE SUB EMS.Array.Set (Array AS ArrayEMS, Element&, SrcSegment, SrcOffset)
DECLARE SUB EMS.Array.Get (Array AS ArrayEMS, Element&, DstSegment, DstOffset)
DECLARE SUB EMS.Array.Swap (Array AS ArrayEMS, Element1&, Element2&)
DECLARE SUB EMS.Array.Erase (Array AS ArrayEMS)

DIM SHARED Gfx.Handle      'EMS handle allocated to pages for the graphics buffers
DIM SHARED Gfx.NumBuffers  'Total number of graphics buffers allocated
DIM SHARED Gfx.PageFrame   'EMS page frame segment (for quick access)
DIM SHARED Gfx.LastMapped  'First page of the four pages that were last
                           '  mapped to the page frame
DIM SHARED Gfx.TextSeg     'Segment of 8x8 ROM font
DIM SHARED Gfx.TextOff     'Offset of 8x8 ROM font

DECLARE FUNCTION Gfx.Alloc (NumBuffers)
DECLARE SUB Gfx.Cls (Buffer)
DECLARE SUB Gfx.Pset (Buffer, x, y, Colr)
DECLARE FUNCTION Gfx.Point (Buffer, x, y)
DECLARE SUB Gfx.Print (Buffer, x, y, Text$, Colr)
DECLARE SUB Gfx.Pcopy (FromBuffer, ToBuffer)
DECLARE SUB Gfx.Swap (Buffer1, Buffer2)
DECLARE SUB Gfx.Dealloc ()

DECLARE SUB Gfx.Get (Buffer, x1, y1, x2, y2, SprSegment, SprOffset)
DECLARE SUB Gfx.Put (Buffer, x, y, Mask, SprSegment, SprOffset)
DECLARE SUB Gfx.Line (Buffer, x1, y1, x2, y2, Colr)

RANDOMIZE TIMER

CLS

IF NOT EMS.Init THEN
  PRINT "No EMM detected."
  END
END IF

COLOR 14, 1
PRINT SPACE$(22); "Using EMS in QuickBasic: Part 3 of 3"; SPACE$(22)
COLOR 15, 0
PRINT STRING$(31, 196); " EMS Information "; STRING$(32, 196)
COLOR 7
PRINT "EMM Version: "; EMS.Version$

IF EMS.Version$ < "4.0" THEN
  PRINT
  PRINT "EMM 4.0 or later must be present to use some of the EMS functions."
  END
END IF

PRINT "Page frame at: "; HEX$(EMS.PageFrame); "h"
PRINT "Free handles:"; EMS.FreeHandles

IF EMS.FreeHandles = 0 THEN
  PRINT
  PRINT "You need at least one free handle to run this demo."
  END
END IF

PRINT "Total EMS:"; EMS.TotalPages; "pages /"; EMS.TotalPages * 16&; "KB /"; EMS.TotalPages \ 64; "MB"
PRINT "Free EMS:"; EMS.FreePages; "pages /"; EMS.FreePages * 16&; "KB /"; EMS.FreePages \ 64; "MB"

IF EMS.FreePages < 128 THEN
  PRINT
  PRINT "You need at least 128 pages (2 MB) free EMS to run this demo."
  END
END IF

PRINT
COLOR 15
PRINT STRING$(32, 196); " EMS Array Test "; STRING$(32, 196)
COLOR 7
PRINT "Creating an integer EMS array with 1,000,000 elements...";

DIM TestArray AS ArrayEMS
TestArray.ElementLen = 2
TestArray.NumElements = 1000000
IF NOT EMS.Array.Dim(TestArray) THEN
  PRINT
  PRINT "Error allocating EMS pages for the array."
  END
END IF

PRINT "ok!"

FOR Test = 1 TO 6

  Element& = INT(RND(1) * 1000000) + 1
  Value = INT(RND(1) * 32768)

  PRINT "Setting element #"; Element&; "to"; Value; "... ";
  EMS.Array.Set TestArray, Element&, VARSEG(Value), VARPTR(Value)
  PRINT "ok!"

  Value = 0
  PRINT "Getting element #"; Element&; "... ";
  EMS.Array.Get TestArray, Element&, VARSEG(Value), VARPTR(Value)
  PRINT "returned:"; Value

NEXT

PRINT "Erasing EMS array...";
EMS.Array.Erase TestArray
PRINT "ok!"

LOCATE 25, 19
COLOR 31
PRINT "Press any key to test EMS graphics functions";

KeyPress$ = INPUT$(1)

IF NOT Gfx.Alloc(3) THEN
  CLS
  PRINT "Error allocating graphics buffers."
  END
END IF

DIM BallPos(1000, 3)
FOR i = 1 TO 1000
  BallPos(i, 1) = INT(RND(1) * 320)
  BallPos(i, 2) = INT(RND(1) * 200)
  BallPos(i, 3) = INT(RND(1) * 4)
NEXT

SCREEN 13

FOR y = 0 TO 15
  FOR x = 0 TO 15
    READ Pixel
    Gfx.Pset 1, x, y, Pixel
  NEXT
NEXT

DIM BallImage(129)
Gfx.Get 1, 0, 0, 15, 15, VARSEG(BallImage(0)), VARPTR(BallImage(0))

FOR x = 0 TO 319
  Gfx.Line 2, x, 0, x, 199, x MOD 255
NEXT

FOR x = 0 TO 319
  Gfx.Line 3, x, 0, x, 199, (255 - x) MOD 255
NEXT

FOR Buffer = 2 TO 3

  FOR ShiftY = -1 TO 1
    FOR ShiftX = -1 TO 1
      Gfx.Print Buffer, 5 + ShiftX, 169 + ShiftY, "Press + to add balls, - to remove balls", 0
      Gfx.Print Buffer, 5 + ShiftX, 179 + ShiftY, "Press ENTER to toggle backgrounds", 0
      Gfx.Print Buffer, 5 + ShiftX, 189 + ShiftY, "Press SPACE to toggle Vsync, ESC quits", 0
    NEXT
  NEXT
  Gfx.Print Buffer, 5, 169, "Press + to add balls, - to remove balls", 15
  Gfx.Print Buffer, 5, 179, "Press ENTER to toggle backgrounds", 15
  Gfx.Print Buffer, 5, 189, "Press SPACE to toggle Vsync, ESC quits", 15

NEXT

NumBalls = 10
Background = 2
Vsync = 0

DO

  KeyPress$ = INKEY$
  IF KeyPress$ <> "" THEN
    SELECT CASE KeyPress$
      CASE CHR$(27)
        EXIT DO
      CASE CHR$(13)
        IF Background = 2 THEN
          Background = 3
        ELSE
          Background = 2
        END IF
      CASE " "
        IF Vsync THEN
          Vsync = 0
        ELSE
          Vsync = 1
        END IF
      CASE "+"
        IF NumBalls < 1000 THEN
          NumBalls = NumBalls + 1
        END IF
      CASE "-"
        IF NumBalls > 1 THEN
          NumBalls = NumBalls - 1
        END IF
    END SELECT
  END IF

  Gfx.Pcopy Background, 1

  FOR i = 1 TO NumBalls
    PosX = BallPos(i, 1)
    PosY = BallPos(i, 2)
    Direction = BallPos(i, 3)
    Gfx.Put 1, PosX, PosY, 0, VARSEG(BallImage(0)), VARPTR(BallImage(0))

    NewDir = Direction
    SELECT CASE Direction
      CASE 0                            ' up and to the left
        PosX = PosX - 1
        IF PosX < -16 THEN NewDir = 1
        PosY = PosY - 1
        IF PosY < -16 THEN NewDir = 3
      CASE 1                            ' up and to the right
        PosX = PosX + 1
        IF PosX > 320 THEN NewDir = 0
        PosY = PosY - 1
        IF PosY < -16 THEN NewDir = 2
      CASE 2                            ' down and to the right
        PosX = PosX + 1
        IF PosX > 320 THEN NewDir = 3
        PosY = PosY + 1
        IF PosY > 200 THEN NewDir = 1
      CASE 3                            ' down and to the left
        PosX = PosX - 1
        IF PosX < -16 THEN NewDir = 2
        PosY = PosY + 1
        IF PosY > 200 THEN NewDir = 0
    END SELECT

    BallPos(i, 1) = PosX
    BallPos(i, 2) = PosY
    IF NewDir <> Direction THEN
      BallPos(i, 3) = NewDir
    END IF

  NEXT

  IF Vsync THEN
    WAIT &H3DA, 8, 8
    WAIT &H3DA, 8
  END IF
  Gfx.Pcopy 1, 0

LOOP

Gfx.Dealloc

SCREEN 0
WIDTH 80, 25

END

' Sprite image for the ball
DATA 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
DATA 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
DATA 0, 0, 0, 0, 0, 0, 4, 4, 4, 4, 4, 0, 0, 0, 0, 0
DATA 0, 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0, 0
DATA 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0
DATA 0, 0, 4, 4, 4, 4, 4, 4, 4,40,40,40, 4, 4, 4, 0
DATA 0, 4, 4, 4, 4, 4, 4, 4,40,40,40,40,40, 4, 4, 4
DATA 0, 4, 4, 4, 4, 4, 4, 4,40,40,40,40,40, 4, 4, 4
DATA 0, 4, 4, 4, 4, 4, 4, 4,40,40,40,40,40, 4, 4, 4
DATA 0, 4, 4, 4, 4, 4, 4, 4, 4,40,40,40, 4, 4, 4, 4
DATA 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
DATA 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0
DATA 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0
DATA 0, 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0, 0
DATA 0, 0, 0, 0, 0, 0, 4, 4, 4, 4, 4, 0, 0, 0, 0, 0
DATA 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

FUNCTION EMS.AllocPages (NumPages)

  'Allocates the number of pages in [NumPages] and
  'returns the EMS handle the memory is allocated to.

  Regs.ax = &H4300                           'Allocate [NumPages] pages of EMS
  Regs.bx = NumPages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

  EMS.AllocPages = Regs.dx                   'Return the handle

END FUNCTION

FUNCTION EMS.Array.Dim (Array AS ArrayEMS)

  'Allocates the required number of EMS pages for an EMS array. The
  'array must be of ArrayEMS type, and the number of elements and
  'element length must be set before this function is called.
  '
  'Array = EMS array to allocate EMS pages for
  '
  'Returns -1 if successful, or 0 if there was an error.

  IF EMS.FreeHandles = 0 THEN
    EMS.Array.Dim = 0                    'No free handles left!
    EXIT FUNCTION
  END IF

  'Calculate the number of 16K pages needed based on the
  'element length and the number of elements.
  BytesNeeded& = Array.NumElements * Array.ElementLen
  PagesNeeded = BytesNeeded& / 16384 + .5

  IF EMS.FreePages < PagesNeeded THEN
    EMS.Array.Dim = 0                    'Not enough free pages!
    EXIT FUNCTION
  END IF

  Array.Handle = EMS.AllocPages(PagesNeeded)   'Allocate pages for the array
                                               '  and save the handle

  EMS.Array.Dim = -1                           'Return the success code

END FUNCTION

SUB EMS.Array.Erase (Array AS ArrayEMS)

  'Deallocates EMS pages used by the specified EMS array.
  '
  'Array = EMS array to use

  EMS.DeallocPages Array.Handle

END SUB

SUB EMS.Array.Get (Array AS ArrayEMS, Element&, DstSegment, DstOffset)

  'Gets an element stored in an EMS array. The element number must be in
  'range from 1 to the total number of elements in the array. The element
  'value will be placed into base memory at DstSegment:DstOffset.
  '
  'Array      = EMS array to use
  'Element&   = Element number
  'DstSegment = Segment of buffer in base memory to hold returned element value
  'DstOffset  = Offset of buffer in base memory to hold returned element value

  IF Element& < 1 OR Element& > Array.NumElements THEN
    'Element out of range
    EXIT SUB
  END IF

  'Find the absolute offset of the element
  Offset& = (Element& - 1) * Array.ElementLen

  'Find which EMS page the element lies on
  Page = Offset& \ 16384

  'Find the page offset of the element
  PageOffset = Offset& - (Page * 16384&)

  'Copy the element from EMS to base memory
  EMS.CopyMem CLNG(Array.ElementLen), Array.Handle, Page, PageOffset, 0, DstSegment, DstOffset

END SUB

SUB EMS.Array.Set (Array AS ArrayEMS, Element&, SrcSegment, SrcOffset)

  'Sets an element in an EMS array. The element number must be in range
  'from 1 to the total number of elements in the array. The element value
  'will be read from base memory at SrcSegment:SrcOffset.
  '
  'Array      = EMS array to use
  'Element&   = Element number
  'SrcSegment = Segment of buffer in base memory containing element value
  'SrcOffset  = Offset of buffer in base memory containing element value

  IF Element& < 1 OR Element& > Array.NumElements THEN
    'Element out of range
    EXIT SUB
  END IF

  'Find the absolute offset of the element
  Offset& = (Element& - 1) * Array.ElementLen

  'Find which EMS page the element lies on
  Page = Offset& \ 16384

  'Find the page offset of the element
  PageOffset = Offset& - (Page * 16384&)

  'Copy the element from base memory to EMS
  EMS.CopyMem CLNG(Array.ElementLen), 0, SrcSegment, SrcOffset, Array.Handle, Page, PageOffset

END SUB

SUB EMS.Array.Swap (Array AS ArrayEMS, Element1&, Element2&)

  'Swaps two elements in an EMS array. The element number must be in range
  'from 1 to the total number of elements in the array. The elements must
  'be in the same array.
  '
  'Array      = EMS array to use
  'Element1&  = First element number
  'Element2&  = Second element number

  IF Element1& < 1 OR Element2& < 1 OR Element1& > Array.NumElements OR Element2& > Array.NumElements THEN
    'Element out of range
    EXIT SUB
  END IF

  'Find the absolute offset of element #1
  Offset1& = (Element1& - 1) * Array.ElementLen

  'Find which EMS page element #1 lies on
  Page1 = Offset1& \ 16384

  'Find the page offset of element #1
  Page1Offset = Offset1& - (Page1 * 16384&)

  'Find the absolute offset of element #2
  Offset2& = (Element2& - 1) * Array.ElementLen

  'Find which EMS page element #2 lies on
  Page2 = Offset2& \ 16384

  'Find the page offset of element #2
  Page2Offset = Offset2& - (Page2 * 16384&)

  'Swap the elements
  EMS.ExchMem CLNG(Array.ElementLen), Array.Handle, Page1, Page1Offset, Array.Handle, Page2, Page2Offset

END SUB

SUB EMS.CopyMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

  'Copies memory from EMS or base memory to EMS or base memory, where:
  '
  'Length&    = Length of memory to copy in bytes
  'SrcHandle  = EMS handle of source memory (use 0 if source is base memory)
  'SrcSegment = Segment of source memory (or page number if source is EMS)
  'SrcOffset  = Offset of source memory
  'DstHandle  = EMS handle of destination memory (use 0 if destination is base memory)
  'DstSegment = Segment of destination memory (or page number if destination is EMS)
  'DstOffset  = Offset of destination memory

  'Determine the source and destination memory types by checking the handles
  IF SrcHandle = 0 THEN SrcType$ = CHR$(0) ELSE SrcType$ = CHR$(1)
  IF DstHandle = 0 THEN DstType$ = CHR$(0) ELSE DstType$ = CHR$(1)

  'Create a buffer containing the copy information
  CopyInfo$ = MKL$(Length&) + SrcType$ + MKI$(SrcHandle) + MKI$(SrcOffset) + MKI$(SrcSegment) + DstType$ + MKI$(DstHandle) + MKI$(DstOffset) + MKI$(DstSegment)

  Regs.ax = &H5700                           'Copy the memory region
  Regs.ds = VARSEG(CopyInfo$)                'described in the buffer
  Regs.si = SADD(CopyInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

SUB EMS.DeallocPages (Handle)

  'Deallocates the EMS pages allocated the EMS handle [Handle].
  'You MUST remember to call the sub before your program ends
  'if you allocate any memory!

  Regs.ax = &H4500                           'Release the pages allocated to [Handle]
  Regs.dx = Handle
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.ErrorMsg$

  'Returns a text string describing the error code in EMS.Error.

  SELECT CASE EMS.Error
    CASE &H0: Msg$ = "successful"
    CASE &H80: Msg$ = "internal error"
    CASE &H81: Msg$ = "hardware malfunction"
    CASE &H82: Msg$ = "busy -- retry later"
    CASE &H83: Msg$ = "invalid handle"
    CASE &H84: Msg$ = "undefined function requested by application"
    CASE &H85: Msg$ = "no more handles available"
    CASE &H86: Msg$ = "error in save or restore of mapping context"
    CASE &H87: Msg$ = "insufficient memory pages in system"
    CASE &H88: Msg$ = "insufficient memory pages available"
    CASE &H89: Msg$ = "zero pages requested"
    CASE &H8A: Msg$ = "invalid logical page number encountered"
    CASE &H8B: Msg$ = "invalid physical page number encountered"
    CASE &H8C: Msg$ = "page-mapping hardware state save area is full"
    CASE &H8D: Msg$ = "save of mapping context failed"
    CASE &H8E: Msg$ = "restore of mapping context failed"
    CASE &H8F: Msg$ = "undefined subfunction"
    CASE &H90: Msg$ = "undefined attribute type"
    CASE &H91: Msg$ = "feature not supported"
    CASE &H92: Msg$ = "successful, but a portion of the source region has been overwritten"
    CASE &H93: Msg$ = "length of source or destination region exceeds length of region allocated to either source or destination handle"
    CASE &H94: Msg$ = "conventional and expanded memory regions overlap"
    CASE &H95: Msg$ = "offset within logical page exceeds size of logical page"
    CASE &H96: Msg$ = "region length exceeds 1 MB"
    CASE &H97: Msg$ = "source and destination EMS regions have same handle and overlap"
    CASE &H98: Msg$ = "memory source or destination type undefined"
    CASE &H9A: Msg$ = "specified alternate map register or DMA register set not supported"
    CASE &H9B: Msg$ = "all alternate map register or DMA register sets currently allocated"
    CASE &H9C: Msg$ = "alternate map register or DMA register sets not supported"
    CASE &H9D: Msg$ = "undefined or unallocated alternate map register or DMA register set"
    CASE &H9E: Msg$ = "dedicated DMA channels not supported"
    CASE &H9F: Msg$ = "specified dedicated DMA channel not supported"
    CASE &HA0: Msg$ = "no such handle name"
    CASE &HA1: Msg$ = "a handle found had no name, or duplicate handle name"
    CASE &HA2: Msg$ = "attempted to wrap around 1M conventional address space"
    CASE &HA3: Msg$ = "source array corrupted"
    CASE &HA4: Msg$ = "operating system denied access"
    CASE ELSE: Msg$ = "undefined error: " + HEX$(EMS.Error) + "h"
  END SELECT

  EMS.ErrorMsg$ = Msg$

END FUNCTION

SUB EMS.ExchMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

  'Exhanges memory from EMS or base memory to EMS or base memory, where:
  '
  'Length&    = Length of memory to exchange in bytes
  'SrcHandle  = EMS handle of source memory (use 0 if source is base memory)
  'SrcSegment = Segment of source memory (or page number if source is EMS)
  'SrcOffset  = Offset of source memory
  'DstHandle  = EMS handle of destination memory (use 0 if destination is base memory)
  'DstSegment = Segment of destination memory (or page number if destination is EMS)
  'DstOffset  = Offset of destination memory

  'Determine the source and destination memory types by checking the handles
  IF SrcHandle = 0 THEN SrcType$ = CHR$(0) ELSE SrcType$ = CHR$(1)
  IF DstHandle = 0 THEN DstType$ = CHR$(0) ELSE DstType$ = CHR$(1)

  'Create a buffer containing the copy information
  ExchInfo$ = MKL$(Length&) + SrcType$ + MKI$(SrcHandle) + MKI$(SrcOffset) + MKI$(SrcSegment) + DstType$ + MKI$(DstHandle) + MKI$(DstOffset) + MKI$(DstSegment)

  Regs.ax = &H5701                           'Exchange the memory region
  Regs.ds = VARSEG(ExchInfo$)                'described in the buffer
  Regs.si = SADD(ExchInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.FreeHandles

  'Returns the number of free (available) EMS handles.

  Regs.ax = &H4B00                             'Get the # of handles in use
  InterruptX &H67, Regs, Regs
  UsedHandles = Regs.bx

  Regs.ax = &H5402                             'Get the total # of handles
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100    'Store the status code
  TotalHandles = Regs.bx

  EMS.FreeHandles = TotalHandles - UsedHandles 'Subtract to get the # of free handles

END FUNCTION

FUNCTION EMS.FreePages

  'Returns the number of free (available) EMS pages
  '(Multiply by 16 to get the amount free EMS in KB.)

  Regs.ax = &H4200                           'Get the # of free pages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code
  EMS.FreePages = Regs.bx

END FUNCTION

FUNCTION EMS.Init

  'Returns true (-1) if an EMM is installed
  'or false (0) if an EMM is not installed.

  Regs.ax = &H3567                        'Get the interrupt vector for int 67h
  InterruptX &H21, Regs, Regs
  DEF SEG = Regs.es                       'Point to the interrupt segment
  FOR x = 10 TO 17                        'Store the 8 bytes at ES:0A in EMM$
    EMM$ = EMM$ + CHR$(PEEK(x))
  NEXT
  IF EMM$ <> "EMMXXXX0" THEN
    EMS.Init = 0              'EMM not installed
  ELSE
    EMS.Init = -1             'EMM installed
  END IF

END FUNCTION

SUB EMS.MapPage (Physical, Logical, Handle)

  'Maps the logical EMS page [Logical] (allocated to the handle [Handle])
  'to the physical page [Physical] in the EMS page frame.

  Regs.ax = &H4400 + Physical                'Map the logical page [Logical]
  Regs.bx = Logical                          'to the physical page [Physical]
  Regs.dx = Handle
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

SUB EMS.MapXPages (PhysicalStart, LogicalStart, NumPages, Handle)

  'Maps up to 4 logical EMS pages to physical pages in the page frame, where:
  '
  'PhysicalStart = Physical page first logical page is mapped to
  'LogicalStart  = First logical page to map
  'NumPages      = Number of pages to map (1 to 4)
  'Handle        = EMS handle logical pages are allocated to

  'Create a buffer containing the page information
  FOR x = 0 TO NumPages - 1
    MapInfo$ = MapInfo$ + MKI$(LogicalStart + x) + MKI$(PhysicalStart + x)
  NEXT

  Regs.ax = &H5000                           'Map the pages in the buffer
  Regs.cx = NumPages                         'to the pageframe
  Regs.dx = Handle
  Regs.ds = VARSEG(MapInfo$)
  Regs.si = SADD(MapInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.PageFrame

  'Returns the segment of the EMS page frame

  Regs.ax = &H4100                           'Get the segment of the page frame
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Save the status code
  EMS.PageFrame = Regs.bx

END FUNCTION

FUNCTION EMS.TotalPages

  'Returns the total number of EMS pages
  '(Multiply by 16 to get the total amount of EMS in KB.)

  Regs.ax = &H4200                           'Get the # of total pages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code
  EMS.TotalPages = Regs.dx

END FUNCTION

FUNCTION EMS.Version$

  'Returns a string containing the EMM version.
  '(Must be "4.0" or greater to use our routines.)

  Regs.ax = &H4600                           'Get the EMM version
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Save the status code

  Version = Regs.ax AND &HFF                 'Split the version number into
  Major = (Version AND &HF0) \ &H10          'its major and minor counterparts
  Minor = Version AND &HF
  EMS.Version$ = LTRIM$(STR$(Major)) + "." + LTRIM$(STR$(Minor))

END FUNCTION

FUNCTION Gfx.Alloc (NumBuffers)

  'Allocates the required number of EMS pages for the number of graphics
  'buffers specified. Also stores the location of the EMS page frame and the
  '8x8 ROM font for later use.
  '
  'NumBuffers = Number of graphics buffers to allocate EMS pages for
  '
  'Returns -1 if successful, or 0 if there was an error.

  PagesNeeded = NumBuffers * 4    'Each buffer needs 64K, and pages are 16K each

  IF EMS.FreePages < PagesNeeded THEN
    'Not enough pages
    Gfx.Alloc = 0
    EXIT FUNCTION
  END IF

  Gfx.Handle = EMS.AllocPages(PagesNeeded)   'Allocate pages and save the handle

  Gfx.NumBuffers = NumBuffers        'Save the number of buffers to prevent
                                     '  a non-existent buffer from being used

  Gfx.PageFrame = EMS.PageFrame      'Save the page frame segment for
                                     '  quicker access

  Gfx.LastMapped = -1                'Assume no pages have been mapped

  Regs.ax = &H1130                   'Get the segment and offset of the 8x8
  Regs.bx = &H300                    '  ROM font and store it for later use
  InterruptX &H10, Regs, Regs
  Gfx.TextSeg = Regs.es
  Gfx.TextOff = Regs.bp

  FOR Buffer = 1 TO NumBuffers       'Clear the newly allocated buffers to
    Gfx.Cls Buffer                   '  get rid of any "garbage"
  NEXT

  Gfx.Alloc = -1                     'Return success status code

END FUNCTION

SUB Gfx.Cls (Buffer)

  'Clears the specified graphics buffer.
  '
  'Buffer = Graphics buffer to clear (Use 0 for the video buffer)

  IF Buffer = 0 THEN
    CLS                  'That was easy ;)
  ELSE
    Page = (Buffer - 1) * 4   'Find the starting EMS page of the buffer
    DIM Blank(8192)           'Create a null 16K buffer

    'Copy this 16K buffer to each of the 4 pages in the graphics buffer
    EMS.CopyMem 16384, 0, VARSEG(Blank(0)), VARPTR(Blank(0)), Gfx.Handle, Page, 0
    EMS.CopyMem 16384, 0, VARSEG(Blank(0)), VARPTR(Blank(0)), Gfx.Handle, Page + 1, 0
    EMS.CopyMem 16384, 0, VARSEG(Blank(0)), VARPTR(Blank(0)), Gfx.Handle, Page + 2, 0
    EMS.CopyMem 16384, 0, VARSEG(Blank(0)), VARPTR(Blank(0)), Gfx.Handle, Page + 3, 0

    ERASE Blank               'Nuke the 16K buffer
  END IF

END SUB

SUB Gfx.Dealloc

  'Deallocates EMS pages used by the graphics buffers.

  EMS.DeallocPages Gfx.Handle

END SUB

SUB Gfx.Get (Buffer, x1, y1, x2, y2, SprSegment, SprOffset)

  'Gets a sprite from an EMS graphics buffer and stores it in a buffer
  'in base memory located at SprSegment:SprOffset. The sprite is stored in
  'the same format used by QB's GET and PUT statements.
  '
  'Buffer     = Graphics buffer to get sprite from (Use 0 for the video buffer)
  'x1         = Starting X coordinate (0-319)
  'y1         = Starting Y coordinate (0-199)
  'x2         = Ending X coordinate (0-319)
  'y2         = Ending Y coordinate (0-199)
  'SprSegment = Segment of buffer in base memory to hold sprite data
  'SprOffset  = Offset of buffer in base memory to hold sprite data
  '
  'Note: x1 < x2 and y1 < y2

  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    BufferSeg = &HA000    'If so, use A000 for the segment
  ELSE
    BufferSeg = Gfx.PageFrame   'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4     'Find the starting page that needs to be
                                '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF

  DEF SEG = SprSegment

  SizeX = x2 - x1 + 1                      'Find the X size of the sprite
  POKE SprOffset, (SizeX * 8) AND 255      '  and store it in the base memory
  POKE SprOffset + 1, (SizeX * 8) \ 256    '  buffer

  SizeY = y2 - y1 + 1                      'Find the Y size of the sprite
  POKE SprOffset + 2, SizeY                '  and store it also

  FOR PlotY = 0 TO SizeY - 1               'Read in the sprite, pixel by
    FOR PlotX = 0 TO SizeX - 1             '  pixel, and store it in the
                                           '  base memory buffer
      DEF SEG = BufferSeg
      Byte = PEEK((y1 + PlotY) * 320& + x1 + PlotX)
      DEF SEG = SprSegment
      POKE SprOffset + 4 + PlotY * SizeX + PlotX, Byte
    NEXT
  NEXT

END SUB

SUB Gfx.Line (Buffer, x1, y1, x2, y2, Colr)

  'Draws a line on an EMS graphics buffer, using Bresenham's algorithm.
  '
  'Buffer     = Graphics buffer to draw line on (Use 0 for the video buffer)
  'x1         = Starting X coordinate (0-319)
  'y1         = Starting Y coordinate (0-199)
  'x2         = Ending X coordinate (0-319)
  'y2         = Ending Y coordinate (0-199)
  'Colr       = Color to use (0-255)
  '
  'Note: Clipping is not supported, so take care not to exceed the bounds
  '      of the screen.


  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    DEF SEG = &HA000      'If so, use A000 for the segment
  ELSE
    DEF SEG = Gfx.PageFrame     'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4     'Find the starting page that needs to be
                                '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF


  UseX1 = x1               'This sub directly modifies the line
  UseY1 = y1               '  coordinates, so it uses a copy of
  UseX2 = x2               '  the parameters to avoid changing the
  UseY2 = y2               '  the programmer's variables.

  IF UseY1 > UseY2 THEN    'Make sure y1 < y2
    SWAP UseY1, UseY2
    SWAP UseX1, UseX2
  END IF

  DeltaX = UseX2 - UseX1     'Draw the line. Ask Mr. Bresenham if you
  DeltaY = UseY2 - UseY1     '  want to know how this works...

  IF DeltaX > 0 THEN

    IF DeltaX > DeltaY THEN
      DeltaYx2 = DeltaY * 2
      DeltaYx2MinusDeltaXx2 = DeltaYx2 - (DeltaX * 2)
      ErrorTerm = DeltaYx2 - DeltaX
      POKE UseY1 * 320& + UseX1, Colr
      DO WHILE DeltaX > 0
        DeltaX = DeltaX - 1
        IF ErrorTerm >= 0 THEN
          UseY1 = UseY1 + 1
          ErrorTerm = ErrorTerm + DeltaYx2MinusDeltaXx2
        ELSE
          ErrorTerm = ErrorTerm + DeltaYx2
        END IF
        UseX1 = UseX1 + 1
        POKE UseY1 * 320& + UseX1, Colr
      LOOP
    ELSE
      DeltaXx2 = DeltaX * 2
      DeltaXx2MinusDeltaYx2 = DeltaXx2 - (DeltaY * 2)
      ErrorTerm = DeltaXx2 - DeltaY
      POKE UseY1 * 320& + UseX1, Colr
      DO WHILE DeltaY > 0
        DeltaY = DeltaY - 1
        IF ErrorTerm >= 0 THEN
          UseX1 = UseX1 + 1
          ErrorTerm = ErrorTerm + DeltaXx2MinusDeltaYx2
        ELSE
          ErrorTerm = ErrorTerm + DeltaXx2
        END IF
        UseY1 = UseY1 + 1
        POKE UseY1 * 320& + UseX1, Colr
      LOOP
    END IF

  ELSE

    DeltaX = -DeltaX
    IF DeltaX > DeltaY THEN
      DeltaYx2 = DeltaY * 2
      DeltaYx2MinusDeltaXx2 = DeltaYx2 - (DeltaX * 2)
      ErrorTerm = DeltaYx2 - DeltaX
      POKE UseY1 * 320& + UseX1, Colr
      DO WHILE DeltaX > 0
        DeltaX = DeltaX - 1
        IF ErrorTerm >= 0 THEN
          UseY1 = UseY1 + 1
          ErrorTerm = ErrorTerm + DeltaYx2MinusDeltaXx2
        ELSE
          ErrorTerm = ErrorTerm + DeltaYx2
        END IF
        UseX1 = UseX1 - 1
        POKE UseY1 * 320& + UseX1, Colr
      LOOP
    ELSE
      DeltaXx2 = DeltaX * 2
      DeltaXx2MinusDeltaYx2 = DeltaXx2 - (DeltaY * 2)
      ErrorTerm = DeltaXx2 - DeltaY
      POKE UseY1 * 320& + UseX1, Colr
      DO WHILE DeltaY > 0
        DeltaY = DeltaY - 1
        IF ErrorTerm >= 0 THEN
          UseX1 = UseX1 - 1
          ErrorTerm = ErrorTerm + DeltaXx2MinusDeltaYx2
        ELSE
          ErrorTerm = ErrorTerm + DeltaXx2
        END IF
        UseY1 = UseY1 + 1
        POKE UseY1 * 320& + UseX1, Colr
      LOOP
    END IF

  END IF

END SUB

SUB Gfx.Pcopy (FromBuffer, ToBuffer)

  'Copies one EMS graphics buffer to another EMS graphics buffer.
  '
  'FromBuffer = Graphics buffer to copy from (Use 0 for the video buffer)
  'ToBuffer   = Graphics buffer to copy to (Use 0 for the video buffer)

  IF FromBuffer = 0 THEN   'Is the source graphics buffer the video buffer?
    SrcHandle = 0          'If so, use 0 for the handle
    SrcSegment = &HA000    '  and A000 for the segment
  ELSE
    SrcHandle = Gfx.Handle               'If not, use the EMS handle
    SrcSegment = (FromBuffer - 1) * 4    '  and the starting page
  END IF

  IF ToBuffer = 0 THEN     'Is the source graphics buffer the video buffer?
    DstHandle = 0          'If so, use 0 for the handle
    DstSegment = &HA000    '  and A000 for the segment
  ELSE
    DstHandle = Gfx.Handle               'If not, use the EMS handle
    DstSegment = (ToBuffer - 1) * 4      '  and the starting page
  END IF

  'Copy 64000 bytes, from the source buffer to the destination buffer
  EMS.CopyMem 64000, SrcHandle, SrcSegment, 0, DstHandle, DstSegment, 0

END SUB

FUNCTION Gfx.Point (Buffer, x, y)

  'Returns the color of the pixel at (x, y) on the specified EMS graphics
  'buffer.
  '
  'Buffer     = Graphics buffer to read pixel from (Use 0 for the video buffer)
  'x          = X coordinate (0-319)
  'y          = Y coordinate (0-199)
  '
  'Returns: Pixel color (0-255)

  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    DEF SEG = &HA000      'If so, use A000 for the segment
  ELSE
    DEF SEG = Gfx.PageFrame    'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4    'Find the starting page that needs to be
                               '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF

  Gfx.Point = PEEK(y * 320& + x)    'Read the pixel at (x, y)

END FUNCTION

SUB Gfx.Print (Buffer, x, y, Text$, Colr)

  '"Prints" text on the EMS graphics buffer, starting at position (x, y) and
  'using the specified color.
  '
  'Buffer     = Graphics buffer to print text on (Use 0 for the video buffer)
  'x          = X coordinate (0-319)
  'y          = Y coordinate (0-199)
  'Text$      = Text string to print
  'Colr       = Text foreground color (0-255)
  '
  'Note: The X and Y coordinates are true pixel coordinates, not "rows" and
  '      "columns" like QB uses. Also, clipping is not supported, so take
  '      care not to exceed the bounds of the screen.

  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    BufferSeg = &HA000    'If so, use A000 for the segment
  ELSE
    BufferSeg = Gfx.PageFrame   'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4     'Find the starting page that needs to be
                                '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF

  StartX = x    'Save the starting X and Y coordinates
  StartY = y    '  because they will be modified

  FOR i = 1 TO LEN(Text$)   'Step through the string, one character at a time

    Char = ASC(MID$(Text$, i, 1))   'Get the ASCII code of the current char

    FOR j = Gfx.TextOff + Char * 8 TO Gfx.TextOff + 7 + Char * 8

      'Get the font character data, one row at a time
      DEF SEG = Gfx.TextSeg
      Bits = PEEK(j)

      'Draw the font character onto the graphics buffer
      DEF SEG = BufferSeg

      'Find the starting offset of this row in the graphics buffer
      Offset& = StartY * 320& + StartX

      'Check to see which bits are set, and only draw pixels whose
      '  corresponding bits are set
      IF Bits AND 128 THEN POKE Offset&, Colr
      IF Bits AND 64 THEN POKE Offset& + 1, Colr
      IF Bits AND 32 THEN POKE Offset& + 2, Colr
      IF Bits AND 16 THEN POKE Offset& + 3, Colr
      IF Bits AND 8 THEN POKE Offset& + 4, Colr
      IF Bits AND 4 THEN POKE Offset& + 5, Colr
      IF Bits AND 2 THEN POKE Offset& + 6, Colr
      IF Bits AND 1 THEN POKE Offset& + 7, Colr

      StartY = StartY + 1      'Move to the next row
    NEXT

    StartX = StartX + 8   'Move to the next character
    StartY = y            'Reset the row
  NEXT

END SUB

SUB Gfx.Pset (Buffer, x, y, Colr)

  'Sets the pixel at (x, y) on the specified EMS graphics buffer to the
  'specified color.
  '
  'Buffer     = Graphics buffer to set pixel on (Use 0 for the video buffer)
  'x          = X coordinate (0-319)
  'y          = Y coordinate (0-199)
  'Colr       = Pixel color (0-255)

  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    DEF SEG = &HA000      'If so, use A000 for the segment
  ELSE
    DEF SEG = Gfx.PageFrame    'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4    'Find the starting page that needs to be
                               '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF

  POKE y * 320& + x, Colr           'Set the pixel at (x, y) to Colr

END SUB

SUB Gfx.Put (Buffer, x, y, Mask, SprSegment, SprOffset)

  'Puts a sprite stored in a buffer in base memory located at
  'SprSegment:SprOffset onto an EMS graphics buffer at position (x, y). The
  'sprite must be stored in the same format used by QB's GET and PUT
  'statements.
  '
  'Buffer     = Graphics buffer to put sprite on (Use 0 for the video buffer)
  'x          = X coordinate (-32768 to 32767)
  'y          = Y coordinate (-32768 to 32767)
  'Mask       = Color to mask (Use -1 to disable masking)
  'SprSegment = Segment of buffer in base memory containing sprite data
  'SprOffset  = Offset of buffer in base memory containing sprite data
  '
  'Note: Sprite clipping is supported, so x and y do not necessarily have to
  '      be within the bounds of the screen.

  IF Buffer = 0 THEN      'Is the graphics buffer the video buffer?
    BufferSeg = &HA000    'If so, use A000 for the segment
  ELSE
    BufferSeg = Gfx.PageFrame   'If not, use the EMS page frame as the segment
    Page = (Buffer - 1) * 4     'Find the starting page that needs to be
                                '  mapped to the pageframe.

    IF Gfx.LastMapped <> Page THEN           'If this page isn't already
      EMS.MapXPages 0, Page, 4, Gfx.Handle   '  mapped, then assume we have
                                             '  map all the pages for the
                                             '  specified buffer.

      Gfx.LastMapped = Page                  'Save the newly mapped page
    END IF
  END IF

  DEF SEG = SprSegment

  'Read the X and Y sizes from the sprite data
  SizeX = (PEEK(SprOffset) + PEEK(SprOffset + 1) * 256) / 8
  SizeY = PEEK(SprOffset + 2)

  FOR PlotY = 0 TO SizeY - 1           'Draw the sprite onto the graphics
    FOR PlotX = 0 TO SizeX - 1         '  buffer, pixel by pixel
      DEF SEG = SprSegment
      Byte = PEEK(SprOffset + 4 + PlotY * SizeX + PlotX)

      IF Byte <> Mask THEN   'Don't draw the pixel if it's masked
                             '  or out of range
        IF y + PlotY >= 0 AND y + PlotY <= 199 AND x + PlotX >= 0 AND x + PlotX <= 319 THEN
          DEF SEG = BufferSeg
          POKE (y + PlotY) * 320& + x + PlotX, Byte
        END IF
     END IF
    NEXT
  NEXT

END SUB

SUB Gfx.Swap (Buffer1, Buffer2)

  'Swaps one EMS graphics buffer with another EMS graphics buffer.
  '
  'Buffer1 = First graphics buffer to swap (Use 0 for the video buffer)
  'Buffer2 = Second graphics buffer to swap (Use 0 for the video buffer)

  IF Buffer1 = 0 THEN      'Is buffer #1 the video buffer?
    SrcHandle = 0          'If so, use 0 for the handle
    SrcSegment = &HA000    '  and A000 for the segment
  ELSE
    SrcHandle = Gfx.Handle               'If not, use the EMS handle
    SrcSegment = (Buffer1 - 1) * 4       '  and the starting page
  END IF

  IF Buffer2 = 0 THEN      'Is buffer #1 the video buffer?
    DstHandle = 0          'If so, use 0 for the handle
    DstSegment = &HA000    '  and A000 for the segment
  ELSE
    DstHandle = Gfx.Handle               'If not, use the EMS handle
    DstSegment = (Buffer2 - 1) * 4       '  and the starting page
  END IF

  'Swap 64000 bytes from buffer #1 with buffer #2
  EMS.ExchMem 64000, SrcHandle, SrcSegment, 0, DstHandle, DstSegment, 0

END SUB

