package org.cocos2d.menu_nodes;

import java.util.ArrayList;
import java.util.Arrays;

import org.cocos2d.cocoa.CCGeometry.CCSize;
import org.cocos2d.support.CCPointExtension;

/**
 * @addtogroup GUI
 * @{
 * @addtogroup menu
 * @{
 */

/** @brief A CCMenuItemToggle
 * A simple container class that "toggles" it's inner items
 * The inner items can be any MenuItem
 */
public class CCMenuItemToggle extends CCMenuItem {

	/** returns the selected item */
	protected int m_uSelectedIndex;

	public int getSelectedIndex() {
		return m_uSelectedIndex;
	}

	public void setSelectedIndex(int index) {
		if((index != m_uSelectedIndex) && (m_pSubItems.size() > 0)) {
			m_uSelectedIndex = index;
			CCMenuItem currentItem = (CCMenuItem)getChildByTag(kCurrentItem);
			if(currentItem != null) {
				currentItem.removeFromParentAndCleanup(false);
			}

			CCMenuItem item = (CCMenuItem)m_pSubItems.get(m_uSelectedIndex);
			this.addChild(item, 0, kCurrentItem);
			CCSize s = item.getContentSize();
			this.setContentSize(s);
			item.setPosition(s.width/2, s.height/2);
		}
	}

	/** CCMutableArray that contains the subitems. You can add/remove items in runtime, and you can replace the array with a new one.
	 * @since v0.7.2
	 */
	protected ArrayList<CCMenuItem> m_pSubItems;

	public ArrayList<CCMenuItem> getSubItems() {
		return m_pSubItems;
	}

	public void setSubItems(ArrayList<CCMenuItem> var) {
		m_pSubItems = var;
	}

	public CCMenuItemToggle() {
		m_uSelectedIndex = 0;
		m_pSubItems = null;
	}

	/** creates a menu item from a list of items with a target/selector */
	public static CCMenuItemToggle createWithTarget(Object target, String selector, CCMenuItem... items) {
		CCMenuItemToggle pRet = new CCMenuItemToggle();
		pRet.initWithTarget(target, selector, items);
		return pRet;
	}

	/** creates a menu item with no target/selector and no items */
	public static CCMenuItemToggle create() {
		CCMenuItemToggle pRet = new CCMenuItemToggle();
		pRet.initWithItem(null);
		return pRet;
	}

	/** initializes a menu item from a list of items with a target selector */
	public boolean initWithTarget(Object target, String selector, CCMenuItem... items) {
		super.initWithTarget(target, selector);
		this.m_pSubItems = new ArrayList<CCMenuItem>();
		m_pSubItems.addAll(Arrays.asList(items));
		m_uSelectedIndex = Integer.MAX_VALUE;;
		this.setSelectedIndex(0);
		return true;
	}

	/** creates a menu item with a item */
	public static CCMenuItemToggle create(CCMenuItem item) {
		CCMenuItemToggle pRet = new CCMenuItemToggle();
		pRet.initWithItem(item);
		return pRet;
	}


	/** initializes a menu item with a item */
	public boolean initWithItem(CCMenuItem item) {
		super.initWithTarget(null, null);
		setSubItems(new ArrayList<CCMenuItem>());

		if(item != null) {
			m_pSubItems.add(item);
		}
		m_uSelectedIndex = Integer.MAX_VALUE;
		this.setSelectedIndex(0);

		setCascadeColorEnabled(true);
		setCascadeOpacityEnabled(true);

		return true;
	}

	/** add more menu item */
	public void addSubItem(CCMenuItem item) {
		m_pSubItems.add(item);
	}

	/** return the selected item */
	public CCMenuItem selectedItem() {
		return m_pSubItems.get(m_uSelectedIndex);
	}

	@Override
	public void activate() {
		// update index
		if(m_bEnabled) {
			int newIndex = (m_uSelectedIndex + 1) % m_pSubItems.size();
			this.setSelectedIndex(newIndex);
		}
		super.activate();
	}

	@Override
	public void selected() {
		super.selected();
		m_pSubItems.get(m_uSelectedIndex).selected();
	}

	@Override
	public void unselected() {
		super.unselected();
		m_pSubItems.get(m_uSelectedIndex).unselected();
	}

	@Override
	public void setEnabled(boolean enabled) {
		if(m_bEnabled != enabled) {
			super.setEnabled(enabled);

			if((m_pSubItems != null) && (m_pSubItems.size() > 0)) {
				for(CCMenuItem pItem : m_pSubItems) {
					pItem.setEnabled(enabled);
				}
			}
		}
	}

	@Override
	public void setOpacityModifyRGB(boolean bValue) {
	}

	@Override
	public boolean isOpacityModifyRGB() {
		return false;
	}

	// TODO legacy -->

	/** conforms with CCRGBAProtocol protocol */
//    private byte opacity_;
    /** conforms with CCRGBAProtocol protocol */
//    ccColor3B color_ = new ccColor3B();
/*
    public void setOpacity(byte opacity) {
        opacity_ = opacity;
        for (CCMenuItem item : m_pSubItems)
            ((CCRGBAProtocol) item).setOpacity(opacity);
    }
    
    public int getOpacity() {
    	return opacity_;
    }

    public void setColor(final ccColor3B color) {
        color_ = color;
        for (CCMenuItem item : m_pSubItems)
            ((CCRGBAProtocol) item).setColor(color);
    }
*/
}

// end of GUI group
/// @}
/// @}
