/****************************************************************************
Copyright (c) 2010 cocos2d-x.org

http://www.cocos2d-x.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
****************************************************************************/
package org.cocos2d.keypad_dispatcher;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @addtogroup input
 * @{
 */

public class CCKeypadDispatcher {

	public enum ccKeypadMSGType {
		// the back key clicked msg
		kTypeBackClicked,
		kTypeMenuClicked,
	}

	public CCKeypadDispatcher() {
		m_bLocked = false;
		m_bToAdd = false;
		m_bToRemove = false;

		m_pDelegates = Collections.synchronizedList(new ArrayList<CCKeypadHandler>());

		m_pHandlersToAdd	= Collections.synchronizedList(new ArrayList<CCKeypadDelegate>());
		m_pHandlersToRemove	= Collections.synchronizedList(new ArrayList<CCKeypadDelegate>());
	}

	/**
	 * @brief add delegate to concern keypad msg
	 */
	public void addDelegate(CCKeypadDelegate pDelegate) {
		if (pDelegate == null) {
			return;
		}

		if (! m_bLocked) {
			forceAddDelegate(pDelegate);
		} else {
			m_pHandlersToAdd.add(pDelegate);
			m_bToAdd = true;
		}
	}

	/**
	 * @brief remove the delegate from the delegates who concern keypad msg
	 */
	public void removeDelegate(CCKeypadDelegate pDelegate) {
		if(pDelegate == null) {
			return;
		}
		if (! m_bLocked) {
			forceRemoveDelegate(pDelegate);
		} else {
			m_pHandlersToRemove.add(pDelegate);
			m_bToRemove = true;
		}
	}

	/**
	 * @brief force add the delegate
	 */
	public void forceAddDelegate(CCKeypadDelegate pDelegate) {
		CCKeypadHandler pHandler = CCKeypadHandler.handlerWithDelegate(pDelegate);

		if (pHandler != null) {
			m_pDelegates.add(pHandler);
		}
	}

	/**
	 * @brief force remove the delegate
	 */
	public void forceRemoveDelegate(CCKeypadDelegate pDelegate) {
		CCKeypadHandler pHandler;
		for(Object pObj : m_pDelegates) {
			pHandler = (CCKeypadHandler) pObj;
			if (pHandler != null && pHandler.getDelegate() == pDelegate) {
				m_pDelegates.remove(pHandler);
				break;
			}
		}
	}

	/**
	 * @brief dispatch the key pad msg
	 */
	public boolean dispatchKeypadMSG(ccKeypadMSGType nMsgType) {

		m_bLocked = true;

		if (m_pDelegates.size() > 0) {
			for(CCKeypadHandler pHandler : m_pDelegates) {
				if(pHandler == null) break;

				CCKeypadDelegate pDelegate;
				pDelegate = pHandler.getDelegate();

				switch (nMsgType) {
				case kTypeBackClicked:
					pDelegate.keyBackClicked();
					break;
				case kTypeMenuClicked:
					pDelegate.keyMenuClicked();
					break;
				default:
					break;
				}
			}
		}

		m_bLocked = false;

		if (m_bToRemove) {
			m_bToRemove = false;
			for (CCKeypadDelegate pDelegate : m_pHandlersToRemove) {
				forceRemoveDelegate(pDelegate);
			}
			m_pHandlersToRemove.clear();
		}

		if (m_bToAdd) {
			m_bToAdd = false;
			for (CCKeypadDelegate pDelegate : m_pHandlersToAdd) {
				forceAddDelegate(pDelegate);
			}
			m_pHandlersToAdd.clear();
		}

		return true;
	}

	protected List<CCKeypadHandler> m_pDelegates;
	protected volatile boolean m_bLocked;
	protected volatile boolean m_bToAdd;
	protected volatile boolean m_bToRemove;

	protected List<CCKeypadDelegate> m_pHandlersToAdd;
	protected List<CCKeypadDelegate> m_pHandlersToRemove;
}

// end of input group
/// @}
