/****************************************************************************
 Copyright (c) 2010-2012 cocos2d-x.org
 Copyright (c) 2008-2010 Ricardo Quesada
 Copyright (c) 2009      Valentin Milea
 Copyright (c) 2011      Zynga Inc.

 http://www.cocos2d-x.org

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
package org.cocos2d.base_nodes;

import org.cocos2d.include.CCProtocols.CCRGBAProtocol;
import org.cocos2d.include.ccTypes.ccColor3B;

/**
 * @addtogroup base_nodes
 * @{
 */

/** CCNodeRGBA is a subclass of CCNode that implements the CCRGBAProtocol protocol.
 *
 * All features from CCNode are valid, plus the following new features:
 * - opacity
 * - RGB colors
 * 
 * Opacity/Color propagates into children that conform to the CCRGBAProtocol if cascadeOpacity/cascadeColor is enabled.
 * @since v2.1
 */
public class CCNodeRGBA extends CCNode implements CCRGBAProtocol {

	public CCNodeRGBA() {
		_displayedOpacity = 255;
		_realOpacity = 255;
		_displayedColor.set(ccColor3B.ccWHITE);
		_realColor.set(ccColor3B.ccWHITE);
		_cascadeColorEnabled = false;
		_cascadeOpacityEnabled = false;
	}

	public boolean init() {
		if(super.init()) {
			_displayedOpacity = _realOpacity = 255;
			_displayedColor.set(ccColor3B.ccWHITE);
			_realColor.set(ccColor3B.ccWHITE);
			_cascadeOpacityEnabled = _cascadeColorEnabled = false;
			return true;
		}
		return false;
	}

	public int getOpacity() {
		return _realOpacity;
	}

	public int getDisplayedOpacity() {
		return _displayedOpacity;
	}

	public void setOpacity(int opacity) {
		_displayedOpacity = _realOpacity = opacity;

		if(_cascadeOpacityEnabled) {
			int parentOpacity = 255;
			CCRGBAProtocol pParent = (CCRGBAProtocol)m_pParent;
			if((pParent != null) && pParent.isCascadeOpacityEnabled()) {
				parentOpacity = pParent.getDisplayedOpacity();
			}
			this.updateDisplayedOpacity(parentOpacity);
		}
	}

	public void updateDisplayedOpacity(int parentOpacity) {
		_displayedOpacity = (int)(_realOpacity * parentOpacity / 255.0);

		if(_cascadeOpacityEnabled) {
			for(CCNode pObj : m_pChildren) {
				CCRGBAProtocol item = (CCRGBAProtocol)pObj;
				if(item != null) {
					item.updateDisplayedOpacity(_displayedOpacity);
				}
			}
		}
	}

	public boolean isCascadeOpacityEnabled() {
		return _cascadeOpacityEnabled;
	}

	public void setCascadeOpacityEnabled(boolean cascadeOpacityEnabled) {
		_cascadeOpacityEnabled = cascadeOpacityEnabled;
	}

	public ccColor3B getColor() {
		return _realColor;
	}

	public ccColor3B getDisplayedColor() {
		return _displayedColor;
	}

	public void setColor(final ccColor3B color) {
		_displayedColor.set(color);
		_realColor.set(color);

		if(_cascadeColorEnabled) {
			ccColor3B parentColor = ccColor3B.ccWHITE;
			CCRGBAProtocol parent = (CCRGBAProtocol)m_pParent;
			if((parent != null) && parent.isCascadeColorEnabled()) {
				parentColor = parent.getDisplayedColor(); 
			}

			updateDisplayedColor(parentColor);
		}
	}

	public void updateDisplayedColor(final ccColor3B parentColor) {
		_displayedColor.r = (int)(_realColor.r * parentColor.r/255.0);
		_displayedColor.g = (int)(_realColor.g * parentColor.g/255.0);
		_displayedColor.b = (int)(_realColor.b * parentColor.b/255.0);

		if(_cascadeColorEnabled) {
			for(CCNode obj : m_pChildren) {
				CCRGBAProtocol item = (CCRGBAProtocol)obj;
				if(item != null) {
					item.updateDisplayedColor(_displayedColor);
				}
			}
		}
	}

	public boolean isCascadeColorEnabled() {
		return _cascadeColorEnabled;
	}

	public void setCascadeColorEnabled(boolean cascadeColorEnabled) {
		_cascadeColorEnabled = cascadeColorEnabled;
	}

	public void setOpacityModifyRGB(boolean bValue) {
	}

	public boolean isOpacityModifyRGB() {
		return false;
	}

	protected int _displayedOpacity;
	protected int _realOpacity;
	protected ccColor3B _displayedColor = new ccColor3B();
	protected ccColor3B _realColor = new ccColor3B();
	protected boolean _cascadeColorEnabled;
	protected boolean _cascadeOpacityEnabled;
}

// end of base_node group
/// @}
