/****************************************************************************
Copyright (c) 2010-2012 cocos2d-x.org
Copyright (c) 2010      Ricardo Quesada

http://www.cocos2d-x.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
****************************************************************************/
package org.cocos2d;

import java.nio.IntBuffer;

import org.cocos2d.include.ccConfig;
import org.cocos2d.include.ccMacros;
import org.cocos2d.platform.CCPlatformMacros;

import android.opengl.GLES20;

/**
 * @addtogroup global
 * @{
 */

/**
 @brief CCConfiguration contains some openGL variables
 @since v0.99.0
 */
public class CCConfiguration {

	/** returns a shared instance of CCConfiguration */
	public static CCConfiguration sharedConfiguration() {
		synchronized(CCConfiguration.class) {
			if(s_gSharedConfiguration == null) {
				s_gSharedConfiguration = new CCConfiguration();
				s_gSharedConfiguration.init();
			}
		}

		return s_gSharedConfiguration;
	}

	/** purge the shared instance of CCConfiguration */
	static void purgeConfiguration() {
		synchronized(CCConfiguration.class) {
			s_gSharedConfiguration = null;
		}
	}

	/** OpenGL Max texture size. */
	public int getMaxTextureSize() {
		return m_nMaxTextureSize;
	}

	/** OpenGL Max Modelview Stack Depth. */
	public int getMaxModelviewStackDepth() {
		return m_nMaxModelviewStackDepth;
	}

	/** returns the maximum texture units
	 * @since v2.0.0
	 */
	public int getMaxTextureUnits() {
		return m_nMaxTextureUnits;
	}

	/** Whether or not the GPU supports NPOT (Non Power Of Two) textures.
	 * OpenGL ES 2.0 already supports NPOT (iOS).
	 * 
	 * @since v0.99.2
	 */
	public boolean supportsNPOT() {
		return m_bSupportsNPOT;
	}

	/** Whether or not PVR Texture Compressed is supported */
	public boolean supportsPVRTC() {
		return m_bSupportsPVRTC;
	}

	/** Whether or not BGRA8888 textures are supported.
	 * @since v0.99.2
	 */
	public boolean supportsBGRA8888() {
		return m_bSupportsBGRA8888;
	}

	/** Whether or not glDiscardFramebufferEXT is supported
	 * @since v0.99.2
	 */
	public boolean supportsDiscardFramebuffer() {
		return m_bSupportsDiscardFramebuffer;
	}

	/** Whether or not shareable VAOs are supported.
	 * @since v2.0.0
	 */
	public boolean supportsShareableVAO() {
		return m_bSupportsShareableVAO;
	}

	/** returns whether or not an OpenGL is supported */
	public boolean checkForGLExtension(final String searchName) {
		boolean bRet = false;
		final String kSearchName = searchName;

		if (m_pGlExtensions != null && 
				m_pGlExtensions.indexOf(kSearchName) > 0) {
			bRet = true;
		}

		return bRet;
	}

	public boolean init() {
		CCPlatformMacros.CCLOG("cocos2d: GL_VENDOR:     %s", GLES20.glGetString(GLES20.GL_VENDOR));
		CCPlatformMacros.CCLOG("cocos2d: GL_RENDERER:   %s", GLES20.glGetString(GLES20.GL_RENDERER));
		CCPlatformMacros.CCLOG("cocos2d: GL_VERSION:    %s", GLES20.glGetString(GLES20.GL_VERSION));

		m_pGlExtensions = GLES20.glGetString(GLES20.GL_EXTENSIONS);

		IntBuffer value = IntBuffer.allocate(1);
		GLES20.glGetIntegerv(GLES20.GL_MAX_TEXTURE_SIZE, value);
		m_nMaxTextureSize = value.get(0);
		GLES20.glGetIntegerv(GLES20.GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS, value);
		m_nMaxTextureUnits = value.get(0);

		m_bSupportsPVRTC = checkForGLExtension("GL_IMG_texture_compression_pvrtc");
		m_bSupportsNPOT = true;
		m_bSupportsBGRA8888 = checkForGLExtension("GL_IMG_texture_format_BGRA888");
		m_bSupportsDiscardFramebuffer = checkForGLExtension("GL_EXT_discard_framebuffer");

		m_bSupportsShareableVAO = checkForGLExtension("vertex_array_object");

		CCPlatformMacros.CCLOG("cocos2d: GL_MAX_TEXTURE_SIZE: %d", m_nMaxTextureSize);
		CCPlatformMacros.CCLOG("cocos2d: GL_MAX_TEXTURE_UNITS: %d",m_nMaxTextureUnits);
		CCPlatformMacros.CCLOG("cocos2d: GL supports PVRTC: %s", (m_bSupportsPVRTC ? "YES" : "NO"));
		CCPlatformMacros.CCLOG("cocos2d: GL supports BGRA8888 textures: %s", (m_bSupportsBGRA8888 ? "YES" : "NO"));
		CCPlatformMacros.CCLOG("cocos2d: GL supports NPOT textures: %s", (m_bSupportsNPOT ? "YES" : "NO"));
		CCPlatformMacros.CCLOG("cocos2d: GL supports discard_framebuffer: %s", (m_bSupportsDiscardFramebuffer ? "YES" : "NO"));
		CCPlatformMacros.CCLOG("cocos2d: GL supports shareable VAO: %s", (m_bSupportsShareableVAO ? "YES" : "NO") );

		boolean bEnableProfilers = false;

		if(ccConfig.CC_ENABLE_PROFILERS != 0) {
			bEnableProfilers = true;
		} else {
			bEnableProfilers = false;
		}

		CCPlatformMacros.CCLOG("cocos2d: compiled with Profiling Support: %s",
				bEnableProfilers ? "YES - *** Disable it when you finish profiling ***" : "NO");

		if(ccConfig.CC_ENABLE_GL_STATE_CACHE == 0) {
			CCPlatformMacros.CCLOG("cocos2d: **** WARNING **** CC_ENABLE_GL_STATE_CACHE is disabled. To improve performance, enable it by editing ccConfig.h");
		}

		ccMacros.CHECK_GL_ERROR_DEBUG();

		return true;
	}

	private CCConfiguration() {
		m_nMaxTextureSize = 0; 
		m_nMaxModelviewStackDepth = 0;
		m_bSupportsPVRTC = false;
		m_bSupportsNPOT = false;
		m_bSupportsBGRA8888 = false;
		m_bSupportsDiscardFramebuffer = false;
		m_bSupportsShareableVAO = false;
		m_nMaxSamplesAllowed = 0;
		m_nMaxTextureUnits = 0;
		m_pGlExtensions = null;
	}

	private static CCConfiguration s_gSharedConfiguration;

	protected int			m_nMaxTextureSize;
	protected int			m_nMaxModelviewStackDepth;
	protected boolean		m_bSupportsPVRTC;
	protected boolean		m_bSupportsNPOT;
	protected boolean		m_bSupportsBGRA8888;
	protected boolean		m_bSupportsDiscardFramebuffer;
	protected boolean		m_bSupportsShareableVAO;
	protected int			m_nMaxSamplesAllowed;
	protected int			m_nMaxTextureUnits;
	protected String		m_pGlExtensions;
}

// end of global group
/// @}
