/****************************************************************************
Copyright (c) 2010-2012 cocos2d-x.org
Copyright (c) 2008-2010 Ricardo Quesada
Copyright (c) 2011      Zynga Inc.

http://www.cocos2d-x.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
****************************************************************************/
package org.cocos2d.opengl;

import org.cocos2d.include.ccMacros;
import org.cocos2d.kazmath.kmMat4;
import org.cocos2d.kazmath.kmVec3;
import org.cocos2d.kazmath.GL.Matrix;
import org.cocos2d.utils.CCFormatter;

/**
 * @addtogroup base_nodes
 * @{
 */

/**
 * A CCCamera is used in every CCNode.
 * Useful to look at the object from different views.
 * The OpenGL gluLookAt() function is used to locate the
 * camera.
 * 
 * If the object is transformed by any of the scale, rotation or
 * position attributes, then they will override the camera.
 * 
 * IMPORTANT: Either your use the camera or the rotation/scale/position properties. You can't use both.
 * World coordinates won't work if you use the camera.
 * 
 * Limitations:
 * 
 * - Some nodes, like CCParallaxNode, CCParticle uses world node coordinates, and they won't work properly if you move them (or any of their ancestors)
 * using the camera.
 * 
 * - It doesn't work on batched nodes like CCSprite objects when they are parented to a CCSpriteBatchNode object.
 * 
 * - It is recommended to use it ONLY if you are going to create 3D effects. For 2D effects, use the action CCFollow or position/scale/rotate.
 * 
 */
public class CCCamera {

	protected float m_fEyeX;
	protected float m_fEyeY;
	protected float m_fEyeZ;

	protected float m_fCenterX;
	protected float m_fCenterY;
	protected float m_fCenterZ;

	protected float m_fUpX;
	protected float m_fUpY;
	protected float m_fUpZ;

	protected boolean m_bDirty;
	protected kmMat4 m_lookupMatrix;

	public CCCamera() {
		init();
	}

	public void init() {
		restore();
	}

	public String description() {
		return CCFormatter.format("<CCCamera | center = (%.2f,%.2f,%.2f)>", m_fCenterX, m_fCenterY, m_fCenterZ);
	}

	/** sets the dirty value */
	public void setDirty(boolean bValue) {
		m_bDirty = bValue;
	}

	/** get the dirty value */
	public boolean isDirty() {
		return m_bDirty;
	}

	/** sets the camera in the default position */
	public void restore() {
		m_fEyeX = m_fEyeY = 0.0f;
		m_fEyeZ = getZEye();

		m_fCenterX = m_fCenterY = m_fCenterZ = 0.0f;

		m_fUpX = 0.0f;
		m_fUpY = 1.0f;
		m_fUpZ = 0.0f;

		kmMat4.kmMat4Identity(m_lookupMatrix);

		m_bDirty = false;
	}

	/** Sets the camera using gluLookAt using its eye, center and up_vector */
	public void locate() {
		if (m_bDirty) {
			kmVec3 eye = new kmVec3();
			kmVec3 center = new kmVec3();
			kmVec3 up = new kmVec3();

			kmVec3.kmVec3Fill(eye, m_fEyeX, m_fEyeY, m_fEyeZ);
			kmVec3.kmVec3Fill(center, m_fCenterX, m_fCenterY, m_fCenterZ);

			kmVec3.kmVec3Fill(up, m_fUpX, m_fUpY, m_fUpZ);
			kmMat4.kmMat4LookAt(m_lookupMatrix, eye, center, up);

			m_bDirty = false;

			/* TODO legacy -->
			GLU.gluLookAt(gl, m_fEyeX, m_fEyeY, m_fEyeZ,
					m_fCenterX, m_fCenterY, m_fCenterZ,
					m_fUpX, m_fUpY, m_fUpZ);
			// TODO legacy <-- */
		}
		Matrix.kmGLMultMatrix(m_lookupMatrix);
	}

	/** sets the eye values in points */
	public void setEyeXYZ(float fEyeX, float fEyeY, float fEyeZ) {
		m_fEyeX = fEyeX;
		m_fEyeY = fEyeY;
		m_fEyeZ = fEyeZ;

		m_bDirty = true;
	}

	/** sets the center values in points */
	public void setCenterXYZ(float fCenterX, float fCenterY, float fCenterZ) {
		m_fCenterX = fCenterX;
		m_fCenterY = fCenterY;
		m_fCenterZ = fCenterZ;

		m_bDirty = true;
	}

	/** sets the up values */
	public void setUpXYZ(float fUpX, float fUpY, float fUpZ) {
		m_fUpX = fUpX;
		m_fUpY = fUpY;
		m_fUpZ = fUpZ;

		m_bDirty = true;
	}

	/** get the eye vector values in points */
	public void getEyeXYZ(float pEyeX[], float pEyeY[], float pEyeZ[]) {
		pEyeX[0] = m_fEyeX;
		pEyeY[0] = m_fEyeY;
		pEyeZ[0] = m_fEyeZ;
	}

	/** get the center vector values int points */
	public void getCenterXYZ(float pCenterX[], float pCenterY[], float pCenterZ[]) {
		pCenterX[0] = m_fCenterX;
		pCenterY[0] = m_fCenterY;
		pCenterZ[0] = m_fCenterZ;
	}

	/** get the up vector values */
	public void getUpXYZ(float pUpX[], float pUpY[], float pUpZ[]) {
		pUpX[0] = m_fUpX;
		pUpY[0] = m_fUpY;
		pUpZ[0] = m_fUpZ;
	}

	/** returns the Z eye */
	public static float getZEye() {
		return ccMacros.FLT_EPSILON;
	}

	// TODO legacy -->
	public String toString() {
		return CCFormatter.format("<%s = %08X | center = (%.2f,%.2f,%.2f)>", this.getClass(), this, m_fCenterX, m_fCenterY, m_fCenterZ);
	}
}

// end of base_node group
/// @}
