package org.cocos2d.textures;

import static javax.microedition.khronos.opengles.GL10.GL_CLAMP_TO_EDGE;
import static javax.microedition.khronos.opengles.GL10.GL_FLOAT;
import static javax.microedition.khronos.opengles.GL10.GL_LINEAR;
import static javax.microedition.khronos.opengles.GL10.GL_NEAREST;
import static javax.microedition.khronos.opengles.GL10.GL_REPEAT;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_2D;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_COORD_ARRAY;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_MAG_FILTER;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_MIN_FILTER;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_WRAP_S;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_WRAP_T;
import static javax.microedition.khronos.opengles.GL10.GL_TRIANGLE_STRIP;
import static javax.microedition.khronos.opengles.GL10.GL_VERTEX_ARRAY;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.HashMap;

import javax.microedition.khronos.opengles.GL;
import javax.microedition.khronos.opengles.GL10;
import javax.microedition.khronos.opengles.GL11ExtensionPack;

import org.cocos2d.CCDirector;
import org.cocos2d.cocoa.CCGeometry.CCPoint;
import org.cocos2d.cocoa.CCGeometry.CCRect;
import org.cocos2d.cocoa.CCGeometry.CCSize;
import org.cocos2d.include.ccMacros;
import org.cocos2d.include.ccTypes.CCTextAlignment;
import org.cocos2d.include.ccTypes.CCVerticalTextAlignment;
import org.cocos2d.nodes.CCLabel;
import org.cocos2d.opengl.GLResourceHelper;
import org.cocos2d.opengl.GLResourceHelper.GLResorceTask;
import org.cocos2d.opengl.GLResourceHelper.GLResourceLoader;
import org.cocos2d.opengl.GLResourceHelper.Resource;
import org.cocos2d.types.CCTexParams;
import org.cocos2d.utils.CCFormatter;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.opengl.GLUtils;

/**
 * @addtogroup textures
 * @{
 */

/** @brief CCTexture2D class.
 * This class allows to easily create OpenGL 2D textures from images, text or raw data.
 * The created CCTexture2D object will always have power-of-two dimensions. 
 * Depending on how you create the CCTexture2D object, the actual image area of the texture might be smaller than the texture dimensions i.e. "contentSize" != (pixelsWide, pixelsHigh) and (maxS, maxT) != (1.0, 1.0).
 * Be aware that the content of the generated textures will be upside-down!
 */
public class CCTexture2D implements Resource {
	private static final String TAG = CCTexture2D.class.getSimpleName();

	//CONSTANTS:

	/** @typedef CCTexture2DPixelFormat
	 * Possible texture pixel formats
	 */
	public enum CCTexture2DPixelFormat {
		//! 32-bit texture: RGBA8888
		kCCTexture2DPixelFormat_RGBA8888,
		//! 24-bit texture: RGBA888
		kCCTexture2DPixelFormat_RGB888,
		//! 16-bit texture without Alpha channel
		kCCTexture2DPixelFormat_RGB565,
		//! 8-bit textures used as masks
		kCCTexture2DPixelFormat_A8,
		//! 8-bit intensity texture
		kCCTexture2DPixelFormat_I8,
		//! 16-bit textures used as masks
		kCCTexture2DPixelFormat_AI88,
		//! 16-bit textures: RGBA4444
		kCCTexture2DPixelFormat_RGBA4444,
		//! 16-bit textures: RGB5A1
		kCCTexture2DPixelFormat_RGB5A1,
		//! 4-bit PVRTC-compressed texture: PVRTC4
		kCCTexture2DPixelFormat_PVRTC4,
		//! 2-bit PVRTC-compressed texture: PVRTC2
		kCCTexture2DPixelFormat_PVRTC2,
	}

	//! Default texture format: RGBA8888
	public static final CCTexture2DPixelFormat kCCTexture2DPixelFormat_Default = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGBA8888;

	// backward compatibility stuff
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_RGBA8888 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGBA8888;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_RGB888 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB888;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_RGB565 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB565;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_A8 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_A8;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_RGBA4444 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGBA4444;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_RGB5A1 = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB5A1;
	public static final CCTexture2DPixelFormat kTexture2DPixelFormat_Default = kCCTexture2DPixelFormat_Default;

	public CCTexture2D() {
		m_bPVRHaveAlphaPremultiplied = true;
		m_uPixelsWide = 0;
		m_uPixelsHigh = 0;
		m_uName = 0;
		m_fMaxS = 0.0f;
		m_fMaxT = 0.0f;
		m_bHasPremultipliedAlpha = false;
		m_bHasMipmaps = false;
// TODO		m_pShaderProgram = null;

		// TODO legacy -->
		_texParams = new CCTexParams(GL_LINEAR, GL_LINEAR, GL_CLAMP_TO_EDGE, GL_CLAMP_TO_EDGE);
		m_ePixelFormat = kCCTexture2DPixelFormat_Default;
	}

	public String description() {
		return CCFormatter.format("<CCTexture2D | Name = %u | Dimensions = %u x %u | Coordinates = (%.2f, %.2f)>", m_uName, m_uPixelsWide, m_uPixelsHigh, m_fMaxS, m_fMaxT);
	}

	/** These functions are needed to create mutable textures */
/* TODO
	public void releaseData(void *data) {
		;
	}

	public void* keepData(void *data, unsigned int length) {
		;
	}
*/
	/** Initializes with a texture2d with data */
/*	public boolean initWithData(const void* data, CCTexture2DPixelFormat pixelFormat, int pixelsWide, int pixelsHigh, final CGSize contentSize) {
		;
	}
*/
	/**
	 * Drawing extensions to make it easy to draw basic quads using a CCTexture2D object.
	 * These functions require GL_TEXTURE_2D and both GL_VERTEX_ARRAY and GL_TEXTURE_COORD_ARRAY client states to be enabled.
	 */
	/** draws a texture at a given point */
	public void drawAtPoint(GL10 gl, final CCPoint point) {
		gl.glEnable(GL_TEXTURE_2D);

		loadTexture(gl);

		float width = (float) m_uPixelsWide * m_fMaxS;
		float height = (float) m_uPixelsHigh * m_fMaxT;

		float vertices[] = {
				point.x, point.y, 0.0f,
				width + point.x, point.y, 0.0f,
				point.x, height + point.y, 0.0f,
				width + point.x, height + point.y, 0.0f
		};

		mVertices.put(vertices);
		mVertices.position(0);

		float coordinates[] = {
				0.0f, m_fMaxT,
				m_fMaxS, m_fMaxT,
				0.0f, 0.0f,
				m_fMaxS, 0.0f
		};

		mCoordinates.put(coordinates);
		mCoordinates.position(0);

		gl.glEnableClientState(GL_VERTEX_ARRAY);
		gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);

		gl.glBindTexture(GL_TEXTURE_2D, m_uName);

		gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		gl.glVertexPointer(3, GL_FLOAT, 0, mVertices);
		gl.glTexCoordPointer(2, GL_FLOAT, 0, mCoordinates);
		gl.glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

		// Clear the vertex and color arrays
		gl.glDisableClientState(GL_VERTEX_ARRAY);
		gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);

		gl.glDisable(GL_TEXTURE_2D);
	}

	/** draws a texture inside a rect */
	public void drawInRect(GL10 gl, CCRect rect) {
		gl.glEnable(GL_TEXTURE_2D);

		loadTexture(gl);

		// float width = (float) mWidth * _maxS;
		// float height = (float) mHeight * _maxT;

		float vertices[] = {
				rect.origin.x, rect.origin.y, /*0.0f,*/
				rect.origin.x + rect.size.width,	rect.origin.y,	/*0.0f,*/
				rect.origin.x, rect.origin.y + rect.size.height, /*0.0f,*/
				rect.origin.x + rect.size.width, rect.origin.y + rect.size.height, /*0.0f*/
		};

		mVertices.put(vertices);
		mVertices.position(0);

		float coordinates[] = {0.0f, m_fMaxT,
				m_fMaxS, m_fMaxT,
				0.0f, 0.0f,
				m_fMaxS, 0.0f};

		mCoordinates.put(coordinates);
		mCoordinates.position(0);

		gl.glEnableClientState(GL_VERTEX_ARRAY);
		gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);

		gl.glBindTexture(GL_TEXTURE_2D, m_uName);

		gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		gl.glVertexPointer(2, GL_FLOAT, 0, mVertices);
		gl.glTexCoordPointer(2, GL_FLOAT, 0, mCoordinates);
		gl.glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

		// Clear the vertex and color arrays
		gl.glDisableClientState(GL_VERTEX_ARRAY);
		gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);

		gl.glDisable(GL_TEXTURE_2D);
	}

	/**
	 * Extensions to make it easy to create a CCTexture2D object from an image file.
	 * Note that RGBA type textures will have their alpha premultiplied - use the blending mode (GL_ONE, GL_ONE_MINUS_SRC_ALPHA).
	 */

	/** Initializes a texture from a UIImage object */
	public boolean initWithImage(Bitmap uiImage) {
/*
		if(uiImage == null) {
			ccMacros.CCLOG(TAG, "cocos2d: CCTexture2D. Can't create Texture. UIImage is nil");
			return false;
		}

		int imageWidth = uiImage.getWidth();
		int imageHeight = uiImage.getHeight();

		CCConfiguration conf = CCConfiguration.sharedConfiguration();

		unsigned maxTextureSize = conf.getMaxTextureSize();
		if(imageWidth > maxTextureSize || imageHeight > maxTextureSize) {
			ccMacros.CCLOG(TAG, "cocos2d: WARNING: Image (%u x %u) is bigger than the supported %u x %u", imageWidth, imageHeight, maxTextureSize, maxTextureSize);
			return false;
		}

		// always load premultiplied images
		return initPremultipliedATextureWithImage(uiImage, imageWidth, imageHeight);
*/

		// TODO legacy -->

		CCSize imageSize = CCSize.Make(uiImage.getWidth(), uiImage.getHeight());
		CCSize realSize = CCSize.Make(uiImage.getWidth(), uiImage.getHeight());
//      CGAffineTransform transform = CGAffineTransform.identity();

		int width = toPow2((int) imageSize.width);
		int height = toPow2((int) imageSize.height);

		boolean needDownScale = false;
		float factor = 1;
		while (width > kMaxTextureSize || height > kMaxTextureSize) {
			width /= 2;
			height /= 2;
//          transform = transform.getTransformScale(0.5f, 0.5f);
			imageSize.width *= 0.5f;
			imageSize.height *= 0.5f;

			factor *= 2;

			needDownScale = true;
		}

		if(needDownScale) {
			Bitmap bitmap = Bitmap.createScaledBitmap(uiImage, (int)imageSize.width, (int)imageSize.height, false);
			uiImage.recycle();
			uiImage = bitmap;
		}

		if (imageSize.width != width || imageSize.height != height) {
			Bitmap bitmap = Bitmap.createBitmap(width, height,
					uiImage.hasAlpha() ? uiImage.getConfig() : Bitmap.Config.RGB_565); //Bitmap.Config.ARGB_8888
			Canvas canvas = new Canvas(bitmap);
			canvas.drawBitmap(uiImage, 0, 0, null);
			uiImage.recycle();
			uiImage = bitmap;
		}

		init(uiImage, realSize, realSize);//imageSize, imageSize);
		m_uPixelsWide = (int) (uiImage.getWidth() * factor);
		m_uPixelsHigh = (int) (uiImage.getHeight() * factor);

		return true;
	}

	/** Initializes a texture from a string with dimensions, alignment, font name and font size */
	public boolean initWithString(final String text, final String fontName, float fontSize, final CCSize dimensions, CCTextAlignment hAlignment, CCVerticalTextAlignment vAlignment) {
/* TODO
		if(ccPlatformMacros.CC_ENABLE_CACHE_TEXTURE_DATA) {
			// cache the texture data
			VolatileTexture::addStringTexture(this, text, dimensions, hAlignment, vAlignment, fontName, fontSize);
		}

		boolean bRet = false;
		CCImage::ETextAlign eAlign;

		if(CCVerticalTextAlignment.kCCVerticalTextAlignmentTop == vAlignment) {
			eAlign = (kCCTextAlignmentCenter == hAlignment) ? CCImage::kAlignTop
					: (kCCTextAlignmentLeft == hAlignment) ? CCImage::kAlignTopLeft : CCImage::kAlignTopRight;
		} else if (CCVerticalTextAlignment.kCCVerticalTextAlignmentCenter == vAlignment) {
			eAlign = (kCCTextAlignmentCenter == hAlignment) ? CCImage::kAlignCenter
					: (kCCTextAlignmentLeft == hAlignment) ? CCImage::kAlignLeft : CCImage::kAlignRight;
		} else if (CCVerticalTextAlignment.kCCVerticalTextAlignmentBottom == vAlignment) {
			eAlign = (kCCTextAlignmentCenter == hAlignment) ? CCImage::kAlignBottom
					: (kCCTextAlignmentLeft == hAlignment) ? CCImage::kAlignBottomLeft : CCImage::kAlignBottomRight;
		} else {
			assert false : "Not supported alignment format!";
			return false;
		}

		CCImage* pImage = new CCImage();
		do {
			CC_BREAK_IF(NULL == pImage);
			bRet = pImage->initWithString(text, (int)dimensions.width, (int)dimensions.height, eAlign, fontName, (int)fontSize);
			CC_BREAK_IF(!bRet);
			bRet = initWithImage(pImage);
		} while (0);
		pImage = null;

		return bRet;
*/
		return false;
	}

	/** Initializes a texture from a string with font name and font size */
	public boolean initWithString(final String text, final String fontName, float fontSize) {
		return initWithString(text, fontName, fontSize, CCSize.Make(0,0), CCTextAlignment.kCCTextAlignmentCenter, CCVerticalTextAlignment.kCCVerticalTextAlignmentTop);
	}

	/**
	 * Extensions to make it easy to create a CCTexture2D object from a PVRTC file
	 *Note that the generated textures don't have their alpha premultiplied - use the blending mode (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA).
	 */
	/** Initializes a texture from a PVRTC buffer */
/* TODO
	public boolean initWithPVRTCData(const void *data, int level, int bpp, bool hasAlpha, int length, CCTexture2DPixelFormat pixelFormat) {
		ifiCC_SUPPORT_PVRTC
		#endif // CC_SUPPORT_PVRTC
	}
*/
	/** Initializes a texture from a PVR file */
	public boolean initWithPVRFile(final String file) {
		return false;
	}

	/** sets the min filter, mag filter, wrap s and wrap t texture parameters.
	 * If the texture size is NPOT (non power of 2), then in can only use GL_CLAMP_TO_EDGE in GL_TEXTURE_WRAP_{S,T}.
	 * 
	 * @warning Calling this method could allocate additional texture memory.
	 * 
	 * @since v0.8
	 */
	public void setTexParameters(CCTexParams texParams) {
		_texParams.set(texParams);
	}

	/** sets antialias texture parameters:
	 * - GL_TEXTURE_MIN_FILTER = GL_LINEAR
	 * - GL_TEXTURE_MAG_FILTER = GL_LINEAR
	 * 
	 * @warning Calling this method could allocate additional texture memory.
	 * 
	 * @since v0.8
	 */
	public void setAntiAliasTexParameters() {
		setTexParameters(GL_LINEAR, GL_LINEAR, GL_CLAMP_TO_EDGE, GL_CLAMP_TO_EDGE);
	}

	/** sets alias texture parameters:
	 * - GL_TEXTURE_MIN_FILTER = GL_NEAREST
	 * - GL_TEXTURE_MAG_FILTER = GL_NEAREST
	 * 
	 * @warning Calling this method could allocate additional texture memory.
	 * 
	 * @since v0.8
	 */
	public void setAliasTexParameters() {
		setTexParameters(GL_NEAREST, GL_NEAREST, GL_CLAMP_TO_EDGE, GL_CLAMP_TO_EDGE);
	}

	/** Generates mipmap images for the texture.
	 * It only works if the texture size is POT (power of 2).
	 * @since v0.99.0
	 */
	public void generateMipmap() {
		assert ( m_uPixelsWide == toPow2((int)m_uPixelsWide) && m_uPixelsHigh == toPow2((int)m_uPixelsHigh))
		: "Mimpap texture only works in POT textures";

		GLResourceHelper.sharedHelper().perform(new GLResourceHelper.GLResorceTask() {
			
			@Override
			public void perform(GL10 gl) {
				if(m_uName != 0) {
					gl.glBindTexture( GL_TEXTURE_2D, m_uName);
					((GL11ExtensionPack)gl).glGenerateMipmapOES(GL_TEXTURE_2D);
				}
			}
		});
	}

	/** returns the pixel format.
	 * @since v2.0
	 */
	public String stringForFormat() {
		return null;
	}

	/** returns the bits-per-pixel of the in-memory OpenGL texture
	 * @since v1.0
	 */
	public int bitsPerPixelForFormat() {
		return 0;
	}

	/** Helper functions that returns bits per pixels for a given format.
	 * @since v2.0
	 */
	public int bitsPerPixelForFormat(CCTexture2DPixelFormat format) {
		return 0;
	}

	/** sets the default pixel format for UIImagescontains alpha channel.
	 * If the UIImage contains alpha channel, then the options are:
	 * - generate 32-bit textures: kCCTexture2DPixelFormat_RGBA8888 (default one)
	 * - generate 24-bit textures: kCCTexture2DPixelFormat_RGB888
	 * - generate 16-bit textures: kCCTexture2DPixelFormat_RGBA4444
	 * - generate 16-bit textures: kCCTexture2DPixelFormat_RGB5A1
	 * - generate 16-bit textures: kCCTexture2DPixelFormat_RGB565
	 * - generate 8-bit textures: kCCTexture2DPixelFormat_A8 (only use it if you use just 1 color)
	 * 
	 * How does it work ?
	 * - If the image is an RGBA (with Alpha) then the default pixel format will be used (it can be a 8-bit, 16-bit or 32-bit texture)
	 * - If the image is an RGB (without Alpha) then: If the default pixel format is RGBA8888 then a RGBA8888 (32-bit) will be used. Otherwise a RGB565 (16-bit texture) will be used.
	 * 
	 * This parameter is not valid for PVR / PVR.CCZ images.
	 * 
	 * @since v0.8
	 */
/* TODO
	public static void setDefaultAlphaPixelFormat(CCTexture2DPixelFormat format) {
		defaultAlphaPixelFormat_ = format;
	}
*/
	/** returns the alpha pixel format
	 * @since v0.8
	 */
/*
	public static CCTexture2DPixelFormat defaultAlphaPixelFormat() {
		return defaultAlphaPixelFormat_;
	}
*/
	/** treats (or not) PVR files as if they have alpha premultiplied.
	 * Since it is impossible to know at runtime if the PVR images have the alpha channel premultiplied, it is
	 * possible load them as if they have (or not) the alpha channel premultiplied.
	 * 
	 * By default it is disabled.
	 * 
	 * @since v0.99.5
	 */
	public static void PVRImagesHavePremultipliedAlpha(boolean haveAlphaPremultiplied) {
		;
	}

	/** content size */
	public CCSize getContentSizeInPixels() {
		return m_tContentSize;
	}

	public boolean hasPremultipliedAlpha() {
		return m_bHasPremultipliedAlpha;
	}

	public boolean hasMipmaps() {
		return false;
	}
/*
	private boolean initPremultipliedATextureWithImage(Bitmap image, int width, int height) {
		unsigned char*            tempData = image.getData();
		unsigned int*             inPixel32 = NULL;
		unsigned char*            inPixel8 = NULL;
		unsigned short*           outPixel16 = NULL;
		boolean hasAlpha = image.hasAlpha();
		CGSize imageSize = CGSize.make((float)(image.getWidth()), (float)(image.getHeight()));
		CCTexture2DPixelFormat pixelFormat;
		size_t                    bpp = image.getBitsPerComponent();

		// compute pixel format
		if(hasAlpha) {
			pixelFormat = g_defaultAlphaPixelFormat;
		} else {
			if(bpp >= 8) {
				pixelFormat = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB888;
			} else {
				pixelFormat = CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB565;
			}
		}

		// Repack the pixel data into the right format
		int length = width * height;

		if(pixelFormat == CCTexture2DPixelFormat.kCCTexture2DPixelFormat_RGB565) {
			if(hasAlpha) {
				// Convert "RRRRRRRRRGGGGGGGGBBBBBBBBAAAAAAAA" to "RRRRRGGGGGGBBBBB"

				tempData = new unsigned char[width * height * 2];
				outPixel16 = (unsigned short*)tempData;
				inPixel32 = (unsigned int*)image.getData();

				for(int i = 0; i < length; ++i, ++inPixel32) {
					*outPixel16++ = 
							((((*inPixel32 >>  0) & 0xFF) >> 3) << 11) |  // R
							((((*inPixel32 >>  8) & 0xFF) >> 2) << 5)  |  // G
							((((*inPixel32 >> 16) & 0xFF) >> 3) << 0);    // B
				}
			} else {
				// Convert "RRRRRRRRRGGGGGGGGBBBBBBBB" to "RRRRRGGGGGGBBBBB"

	            tempData = new unsigned char[width * height * 2];
	            outPixel16 = (unsigned short*)tempData;
	            inPixel8 = (unsigned char*)image->getData();
	            
	            for(unsigned int i = 0; i < length; ++i)
	            {
	                *outPixel16++ = 
	                (((*inPixel8++ & 0xFF) >> 3) << 11) |  // R
	                (((*inPixel8++ & 0xFF) >> 2) << 5)  |  // G
	                (((*inPixel8++ & 0xFF) >> 3) << 0);    // B
	            }
	        }    
	    }
	    else if (pixelFormat == kCCTexture2DPixelFormat_RGBA4444)
	    {
	        // Convert "RRRRRRRRRGGGGGGGGBBBBBBBBAAAAAAAA" to "RRRRGGGGBBBBAAAA"
	        
	        inPixel32 = (unsigned int*)image->getData();  
	        tempData = new unsigned char[width * height * 2];
	        outPixel16 = (unsigned short*)tempData;
	        
	        for(unsigned int i = 0; i < length; ++i, ++inPixel32)
	        {
	            *outPixel16++ = 
	            ((((*inPixel32 >> 0) & 0xFF) >> 4) << 12) | // R
	            ((((*inPixel32 >> 8) & 0xFF) >> 4) <<  8) | // G
	            ((((*inPixel32 >> 16) & 0xFF) >> 4) << 4) | // B
	            ((((*inPixel32 >> 24) & 0xFF) >> 4) << 0);  // A
	        }
	    }
	    else if (pixelFormat == kCCTexture2DPixelFormat_RGB5A1)
	    {
	        // Convert "RRRRRRRRRGGGGGGGGBBBBBBBBAAAAAAAA" to "RRRRRGGGGGBBBBBA"
	        inPixel32 = (unsigned int*)image->getData();   
	        tempData = new unsigned char[width * height * 2];
	        outPixel16 = (unsigned short*)tempData;
	        
	        for(unsigned int i = 0; i < length; ++i, ++inPixel32)
	        {
	            *outPixel16++ = 
	            ((((*inPixel32 >> 0) & 0xFF) >> 3) << 11) | // R
	            ((((*inPixel32 >> 8) & 0xFF) >> 3) <<  6) | // G
	            ((((*inPixel32 >> 16) & 0xFF) >> 3) << 1) | // B
	            ((((*inPixel32 >> 24) & 0xFF) >> 7) << 0);  // A
	        }
	    }
	    else if (pixelFormat == kCCTexture2DPixelFormat_A8)
	    {
	        // Convert "RRRRRRRRRGGGGGGGGBBBBBBBBAAAAAAAA" to "AAAAAAAA"
	        inPixel32 = (unsigned int*)image->getData();
	        tempData = new unsigned char[width * height];
	        unsigned char *outPixel8 = tempData;
	        
	        for(unsigned int i = 0; i < length; ++i, ++inPixel32)
	        {
	            *outPixel8++ = (*inPixel32 >> 24) & 0xFF;  // A
	        }
	    }
	    
	    if (hasAlpha && pixelFormat == kCCTexture2DPixelFormat_RGB888)
	    {
	        // Convert "RRRRRRRRRGGGGGGGGBBBBBBBBAAAAAAAA" to "RRRRRRRRGGGGGGGGBBBBBBBB"
	        inPixel32 = (unsigned int*)image->getData();
	        tempData = new unsigned char[width * height * 3];
	        unsigned char *outPixel8 = tempData;
	        
	        for(unsigned int i = 0; i < length; ++i, ++inPixel32)
	        {
	            *outPixel8++ = (*inPixel32 >> 0) & 0xFF; // R
	            *outPixel8++ = (*inPixel32 >> 8) & 0xFF; // G
	            *outPixel8++ = (*inPixel32 >> 16) & 0xFF; // B
	        }
	    }
	    
	    initWithData(tempData, pixelFormat, width, height, imageSize);
	    
	    if (tempData != image->getData())
	    {
	        delete [] tempData;
	    }

	    m_bHasPremultipliedAlpha = image->isPremultipliedAlpha();
	    return true;

	}
*/
	// By default PVR images are treated as if they don't have the alpha channel premultiplied
	private boolean m_bPVRHaveAlphaPremultiplied;

	/** pixel format of the texture */
	protected CCTexture2DPixelFormat m_ePixelFormat;

	public CCTexture2DPixelFormat getPixelFormat() {
		return m_ePixelFormat;
	}

	/** width in pixels */
	protected int m_uPixelsWide;

	public int getPixelsWide() {
		return m_uPixelsWide;
	}

	/** height in pixels */
	private int m_uPixelsHigh;

	public int getPixelsHigh() {
		return m_uPixelsHigh;
	}

	/** texture name */
	private int m_uName = 0;

	public int getName() {
		return m_uName;
	}

	/** texture max S */
	private float m_fMaxS;

	public float getMaxS() {
		return m_fMaxS;
	}

	public void setMaxS(float maxS) {
		m_fMaxS = maxS;
	}

	/** texture max T */
	private float m_fMaxT;

	public float getMaxT() {
		return m_fMaxT;
	}

	public void setMaxT(float maxT) {
		m_fMaxT = maxT;
	}

	/** content size */
	private CCSize m_tContentSize;

	public final CCSize getContentSize() {
		CCSize ret = new CCSize();
		ret.width = m_tContentSize.width / ccMacros.CC_CONTENT_SCALE_FACTOR();
		ret.height = m_tContentSize.height / ccMacros.CC_CONTENT_SCALE_FACTOR();
		return ret;
	}

	/** whether or not the texture has their Alpha premultiplied */
	private boolean m_bHasPremultipliedAlpha = false;

	private boolean m_bHasMipmaps;

	/** shader program used by drawAtPoint and drawInRect */
/* TODO
	protected CCGLProgram m_pShaderProgram;

	public CCGLProgram getShaderProgram() {
		return m_pShaderProgram;
	}

	public void setShaderProgram(CCGLProgram var) {
		m_pShaderProgram = var;
	}
*/


	// TODO legacy -->

	public static final int kMaxTextureSize = 1024;

	public Bitmap.Config pixelFormat() {
		Bitmap.Config format;

		switch(m_ePixelFormat) {
		case kCCTexture2DPixelFormat_A8:
			format = Bitmap.Config.ALPHA_8;
			break;
		case kCCTexture2DPixelFormat_RGBA4444:
			format = Bitmap.Config.ARGB_4444;
			break;
		case kCCTexture2DPixelFormat_RGBA8888:
			format = Bitmap.Config.ARGB_8888;
			break;
		case kCCTexture2DPixelFormat_RGB565:
			format = Bitmap.Config.RGB_565;
			break;
		default:
			format = Bitmap.Config.RGB_565;
		}

		return format;
	}

	/**
     * width in pixels
     */
    public float getWidth() {
        return m_tContentSize.width;
    }

    /**
     * height in pixels
     */
    public float getHeight() {
        return m_tContentSize.height;
    }

    private FloatBuffer mVertices;
    private FloatBuffer mCoordinates;
//    private ShortBuffer mIndices;

    /** this mBitmap should be created when we call load(),
     * when we create the texture mBitmap is destroyed
     */
    private Bitmap mBitmap;

    private CCTexParams _texParams;
    
    private GL mCreator;
    
//    /** this object is responsible for loading Bitmap for texture */
//    private GLResourceHelper.GLResourceLoader mLoader;

    public void releaseTexture (GL10 gl) {
        if (m_uName != 0) {
            gl.glDeleteTextures(1, new int[]{m_uName}, 0);
            m_uName = 0;
        }
    }
    
    @Override
    protected void finalize() throws Throwable {
    	if (m_uName != 0) {
    		GLResourceHelper.sharedHelper().perform(new GLResourceHelper.GLResorceTask() {
    			
				@Override
				public void perform(GL10 gl) {
					if(mCreator == gl) {
						IntBuffer intBuffer = IntBuffer.allocate(1);
						intBuffer.put(0, m_uName);
						gl.glDeleteTextures(1, intBuffer);
					}
				}
				
			});
    	}

        super.finalize();
    }

//    public void checkName() {
//    	if (mLoader != null && _name == 0)
//    		mLoader.load(this);
//    }
    
    public void setLoader(GLResourceHelper.GLResourceLoader loader) {
    	if(loader != null) {
    		loader.load(this);
    		
        	// we called load and should not add task
    		GLResourceHelper.sharedHelper().addLoader(this, loader, false);
    	}
//    	mLoader = loader;
    }
    
    public void initWithImage(Bitmap image, CCSize imageSize) {
        Bitmap.Config config = Bitmap.Config.ARGB_8888;
        Bitmap bitmap = Bitmap.createBitmap((int) imageSize.width, (int) imageSize.height, config);
        Canvas canvas = new Canvas(bitmap);
        
        canvas.drawBitmap(image, 0, 0, new Paint());
        image.recycle();

        init(bitmap, imageSize, imageSize);
    }
    
    public void initWithImage(Bitmap image, CCSize imageSize, CCSize contentSize) {
        Bitmap.Config config = Bitmap.Config.ARGB_8888;
        Bitmap bitmap = Bitmap.createBitmap((int) imageSize.width, (int) imageSize.height, config);
        Canvas canvas = new Canvas(bitmap);
        
        canvas.drawBitmap(image, 0, 0, new Paint());
        image.recycle();

        init(bitmap, imageSize, contentSize);
    }

    private void init(Bitmap image, CCSize imageSize, CCSize contentSize) {
        mBitmap = image;

        m_uPixelsWide = image.getWidth();
        m_uPixelsHigh = image.getHeight();
        m_tContentSize = contentSize;
        // _format = image.getConfig();
        m_fMaxS = m_tContentSize.width / (float) m_uPixelsWide;
        m_fMaxT = m_tContentSize.height / (float) m_uPixelsHigh;
        ByteBuffer vfb = ByteBuffer.allocateDirect(4 * 3 * 4);
        vfb.order(ByteOrder.nativeOrder());
        mVertices = vfb.asFloatBuffer();

        ByteBuffer tfb = ByteBuffer.allocateDirect(4 * 2 * 4);
        tfb.order(ByteOrder.nativeOrder());
        mCoordinates = tfb.asFloatBuffer();
        
        // GLUtils.texImage2D makes premultiplied alpha
		if(mBitmap.getConfig() == Bitmap.Config.ARGB_8888)
			m_bHasPremultipliedAlpha = true;
		
//        ByteBuffer isb = ByteBuffer.allocateDirect(6 * 2);
//        isb.order(ByteOrder.nativeOrder());
//        mIndices = isb.asShortBuffer();
//		CCTextureCache.sharedTextureCache().addTexture(this);
		
		// for call loadTexture when reinit
		if(m_uName != 0) {
			m_uName = 0;
			loadTexture(CCDirector.gl);
		} else {
    		GLResourceHelper.sharedHelper().perform(new GLResourceHelper.GLResorceTask() {
    			
				@Override
				public void perform(GL10 gl) {
					loadTexture(gl);
				}
			});			
		}
    }

    /**
      Extensions to make it easy to create a CCTexture2D object from a string of text.
      Note that the generated textures are of type A8 - use the blending mode (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA).
    */
    /** Initializes a texture from a string with font name and font size */
    public void initWithText(String text, String fontname, float fontSize) {
    	initWithText(text, calculateTextSize(text, fontname, fontSize),
                CCLabel.TextAlignment.CENTER, fontname, fontSize);
    }
    
    // for styled text
    public void initWithText(String text, String fontname, float fontSize, int fontStyle) {
    	initWithText(text, calculateTextSize(text, fontname, fontSize),
                CCLabel.TextAlignment.CENTER, fontname, fontSize, fontStyle);
    }

    private static CCSize calculateTextSize(String text, String fontname, float fontSize) {
//        Typeface typeface = Typeface.create(fontname, Typeface.NORMAL);
    	Typeface typeface;
    	if(!typefaces.containsKey(fontname)) {
	        try {
	        	CCDirector.theApp.getAssets().open(fontname);
	        	typeface = Typeface.createFromAsset(CCDirector.theApp.getAssets(), fontname);
	        } catch(IOException e) {
	        	typeface = Typeface.create(fontname, Typeface.NORMAL);
	        }
	        typefaces.put(fontname, typeface);
    	} else {
    		typeface = typefaces.get(fontname);
    	}
    	
//        typeface = Typeface.
//    	try{
//    		typeface = Typeface.createFromAsset(CCDirector.theApp.getAssets(), fontname);
//    	} catch (Exception e) {
//    		typeface = Typeface.create(fontname, Typeface.NORMAL);
//		}
//    	typeface = Typeface.DEFAULT;

        Paint textPaint = new Paint();
        textPaint.setTypeface(typeface);
        textPaint.setTextSize(fontSize);
        textPaint.setAntiAlias(true);

        int ascent = (int) Math.ceil(-textPaint.ascent());  // Paint.ascent is negative, so negate it
        int descent = (int) Math.ceil(textPaint.descent());
        int measuredTextWidth = (int) Math.ceil(textPaint.measureText(text));

        return CCSize.Make(measuredTextWidth, ascent + descent);
    }

    private static int toPow2(int v) {
        if ((v != 1) && (v & (v - 1)) != 0) {
            int i = 1;
            while (i < v)
                i *= 2;
            v = i;
        }
//        if (v > CCTexture2D.kMaxTextureSize) {
//        	v = CCTexture2D.kMaxTextureSize;
//        }
        return v;
    }

    private static HashMap<String, Typeface> typefaces = new HashMap<String, Typeface>();
    /** Initializes a texture from a string with dimensions, alignment, font name and font size */
    public void initWithText(String text, CCSize dimensions, CCLabel.TextAlignment alignment, String fontname, float fontSize) {
    	initWithText(text, dimensions, alignment, fontname, fontSize, Typeface.NORMAL);
    }
    
    public void initWithText(String text, CCSize dimensions, CCLabel.TextAlignment alignment, String fontname, float fontSize, int fontStyle) {
    	Typeface typeface;
    	if(!typefaces.containsKey(fontname)) {
	        try {
	        	CCDirector.theApp.getAssets().open(fontname);
	        	typeface = Typeface.createFromAsset(CCDirector.theApp.getAssets(), fontname);
	        } catch(IOException e) {
	        	typeface = Typeface.create(fontname, fontStyle);
	        }
	        typefaces.put(fontname, typeface);
    	} else {
    		typeface = typefaces.get(fontname);
    	}

        Paint textPaint = new Paint();
        textPaint.setTypeface(typeface);
        textPaint.setTextSize(fontSize);
        textPaint.setAntiAlias(true);

        float ascent = -textPaint.ascent();  // Paint.ascent is negative, so negate it
        float descent = textPaint.descent();

        int textHeight = (int)(ascent + descent);
        int spacing = (int) Math.ceil((ascent + descent) * 0.1f);

        int width = toPow2((int)dimensions.width);
        int height = toPow2((int) dimensions.height);

        Bitmap.Config config = Bitmap.Config.ALPHA_8;
        Bitmap bitmap = Bitmap.createBitmap(width, height, config);
        Canvas canvas = new Canvas(bitmap);
        bitmap.eraseColor(Color.TRANSPARENT);

        ArrayList<String> wrapped = WrapText(textPaint, text, dimensions.width);
        
        float blockHeight = (ascent + descent) * wrapped.size();

        for(int i = 0; i < wrapped.size(); ++i)
        {
        	String str = wrapped.get(i);
        	float offset = 0;
        	float vOffset = 0;

	        switch (alignment) {
	            case LEFT:
	                offset = 0;
	                break;
	            case CENTER:
	            	offset = (dimensions.width - textPaint.measureText(str)) * 0.5f;
	            	vOffset = (dimensions.height - blockHeight) * 0.5f;
	                break;
	            case RIGHT:
	            	offset = (dimensions.width - textPaint.measureText(str));
	                break;
	        }

	        canvas.drawText(str,
	                offset,
	                vOffset + ascent + ((textHeight + spacing) * i),
	                textPaint);
        }

        init(bitmap, dimensions, dimensions);
    }

    protected ArrayList<String> WrapText(Paint textPaint, String text, float width)
    {
        float spaceLeft = width;

        String [] words = text.split(" ");
        ArrayList<String> lines = new ArrayList<String>();
        float spaceWidth = textPaint.measureText(" ");
        StringBuilder tempLine = new StringBuilder("");

        for(String word : words)
        {
            float wordWidth = textPaint.measureText(word);

            if (wordWidth > spaceLeft) {
            	if(tempLine.length() > 0) {
                	tempLine.deleteCharAt(tempLine.length() - 1);
                }
            	
                lines.add(tempLine.toString());
                
                tempLine = new StringBuilder("");
                tempLine.append(word);

                spaceLeft = width - (wordWidth + spaceWidth);
            }
            else
            {
                tempLine.append(word);
                spaceLeft -= (wordWidth + spaceWidth);
            }

            tempLine.append(" ");
        }
        
        if(tempLine.length() > 0) {
        	tempLine.deleteCharAt(tempLine.length() - 1);
        }

        lines.add(tempLine.toString());

        return lines;
    }

    public void loadTexture(GL10 gl) {
        if (m_uName == 0) {
        	mCreator = gl;
        	
            int[] textures = new int[1];
            gl.glGenTextures(1, textures, 0);

            m_uName = textures[0];
            
            applyTexParameters(gl);

            // this shouldn't be so never, but if so, needs to be found where
            // texture reloading is in progress 
        	if(mBitmap == null)
        		return;

            GLUtils.texImage2D(GL_TEXTURE_2D, 0, mBitmap, 0);
            mBitmap.recycle();
            mBitmap = null;
        }
    }

    public boolean isLoaded() {
        return mBitmap == null && m_uName != 0;
    }



    //
    // Use to apply MIN/MAG filter
    //

//    private static CCTexParams _gTexParams = new CCTexParams(GL_LINEAR, GL_LINEAR, GL_CLAMP_TO_EDGE, GL_CLAMP_TO_EDGE);
//    private static CCTexParams _texParamsCopy;

    /** sets the min filter, mag filter, wrap s and wrap t texture parameters.
      If the texture size is NPOT (non power of 2),
             then in can only use GL_CLAMP_TO_EDGE in GL_TEXTURE_WRAP_{S,T}.
      @since v0.8
    */
    public void setTexParameters(int min, int mag, int s, int t) {
    	_texParams.set(min, mag, s, t);
    	if(m_uName != 0) {
    		GLResourceHelper.sharedHelper().perform(new GLResourceHelper.GLResorceTask() {
    			
				@Override
				public void perform(GL10 gl) {
					applyTexParameters(gl);
				}
    		});
    	}
    }

//    public static CCTexParams texParameters() {
//        return gTexParams;
//    }

    private void applyTexParameters(GL10 gl) {
        gl.glBindTexture(GL_TEXTURE_2D, m_uName);
        gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, _texParams.minFilter );
        gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, _texParams.magFilter);
        gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, _texParams.wrapS);
        gl.glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, _texParams.wrapT);
    }

//    public static void restoreTexParameters() {
//        _gTexParams = _texParamsCopy;
//    }
//
//    public static void saveTexParameters() {
//        _texParamsCopy = _gTexParams.copy();
//    }



    static Bitmap.Config defaultAlphaPixelFormat_ = Bitmap.Config.ARGB_8888;
    
}

//end of textures group
/// @}
