﻿using System;
using System.Linq;
using System.Collections.Generic;

using Vintagestory.API.Common;
using Vintagestory.API.Config;
using Vintagestory.API.MathTools;
using Vintagestory.API.Server;
using Vintagestory.API.Util;

using Vintagestory.Server;

using ProtoBuf;


namespace AdminToolkit
{
	[ProtoContract]
	internal struct Spawnpoint
	{
		[ProtoMember(1)]
		public PlayerSpawnPos Location;

		[ProtoMember(2)]
		public string Name;

		public Spawnpoint(string name, Vec3i loc)
		{
		Name = name;
		Location = new PlayerSpawnPos {
			x = loc.X,
			y = loc.Y,
			z = loc.Z
		};
		}

	}

	public class VariableSpawnpoints : AdminModCommand
	{
		private const string _spawnsDataKey = @"spawnpoints";
		private Random random = new Random();

		private List<Spawnpoint> Spawnpoints { get; set; }

	
		public VariableSpawnpoints(ICoreServerAPI _serverAPI) : base(_serverAPI)
		{				
		this.Command = "spawnpoints";
		this.Description = "Control add / remove / adjust List of approved player spawn points";
		this.RequiredPrivilege = Privilege.setspawn;
		this.Syntax = @"Add {name} {123 456 -678}|here / Remove {name} / List / Enable / Disable";
		this.handler += HandleCommand;

		ServerAPI.Event.ServerRunPhase(EnumServerRunPhase.RunGame, ReloadSpawnpoints);
		
		ServerAPI.Event.PlayerDeath += PlayerDied;
		ServerAPI.Event.PlayerCreate += (byPlayer) => ChangeDefaultRoleSpawnPoint();
		

		ServerAPI.Event.ServerRunPhase(EnumServerRunPhase.Shutdown, PersistSpawnpoints);
		}

		private void HandleCommand(IServerPlayer player, int groupId, CmdArgs args)
		{
		string name;
		Vec3i position;

		bool console = player is ServerConsolePlayer;

		if (args.Length >= 1) {
		string cmd = args.PopWord(string.Empty).ToLowerInvariant();
		switch (cmd) {
		
		case "add":
			name = args.PopWord("?");

			if (console == false && String.Equals(args.PeekWord( ), "here", StringComparison.OrdinalIgnoreCase)) {
			position = player.Entity.Pos.XYZInt;
			}
			else {
			var pos = args.PopVec3i(ServerAPI.World.DefaultSpawnPosition.AsBlockPos.ToVec3i( ));
			position = RelativeToAbsolute(pos);
			}

			if (AddSpawnpoint(name, position)) {
			player.SendMessage(groupId, "Added Spawnpoint.", EnumChatType.CommandSuccess);
			}
			else {
			player.SendMessage(groupId, "Can't add Spawnpoint!", EnumChatType.CommandError);
			}
			break;

		case "remove":
			name = args.PopWord( );
			if (RemoveSpawnpoint(name)) {
			player.SendMessage(groupId, $"Removed Spawnpoint '{name}'", EnumChatType.CommandSuccess);
			}
			break;

		case "list":
			ListSpawnpoints(groupId, player);
			break;

		case "enable":
			Toggle(true);
			break;
			
		case "disable":
			Toggle(false);
			break;

		case "reset":
			if (console) {
			name = args.PopWord("?");
			#if DEBUG
			ResetPlayer(name);
			player.SendMessage(groupId, $"Reset State / Playerdata for '{name}'", EnumChatType.CommandSuccess);
			#endif
			}
			else {
			player.SendMessage(groupId, "Only CONSOLE may use command!", EnumChatType.CommandError);
			}
			break;

		default:
			player.SendMessage(groupId, "Unrecognised command ", EnumChatType.CommandError);
			break;

		}
		}
		else {
		player.SendMessage(groupId, "Supply a command verb", EnumChatType.CommandError);
		}

		}

		private bool AddSpawnpoint(string name, Vec3i position)
		{
		if (IsItSafe(position.AsBlockPos)) {
		//Appears OK...
		Logger.VerboseDebug($"Adding variable spawnpoint '{name}' @ {position}");
		var newSpawnpoint = new Spawnpoint(name, position);

		this.Spawnpoints.Add(newSpawnpoint);
		return true;
		}
		else {
		Logger.VerboseDebug($"Apparently '{name}' spawnpoint is Unsafe!  @ABS:{position}");	
		}

		return false;
		}

		private bool IsItSafe(BlockPos position)
		{
		//Check position is 'safe'; 2~ blocks airish, and solid block BELOW...
		var surfaceBlock = ServerAPI.World.BlockAccessor.GetBlock(position);
		var aboveBlock = ServerAPI.World.BlockAccessor.GetBlock(position.AddCopy(BlockFacing.UP));
		var belowBlock = ServerAPI.World.BlockAccessor.GetBlock(position.AddCopy(BlockFacing.DOWN));

		if (belowBlock.MatterState == EnumMatterState.Solid &&
			aboveBlock.BlockMaterial == EnumBlockMaterial.Air &&
			surfaceBlock.BlockMaterial == EnumBlockMaterial.Air 
			|| surfaceBlock.BlockMaterial == EnumBlockMaterial.Snow
			|| surfaceBlock.BlockMaterial == EnumBlockMaterial.Liquid
			|| surfaceBlock.BlockMaterial == EnumBlockMaterial.Leaves
		) 
		{
		return true;
		}

		return false;
		}

		private bool RemoveSpawnpoint(string name)
		{

		if (this.Spawnpoints.Count > 0) {
		var count = this.Spawnpoints.RemoveAll(sp => String.Equals(sp.Name, name, StringComparison.InvariantCultureIgnoreCase));
		return count > 0;
		}

		return false;
		}

		private void ListSpawnpoints( int chatGroupId, IServerPlayer player )
		{
		if (this.Spawnpoints.Count > 0) {
		foreach (var sp in this.Spawnpoints) {
		player.SendMessage(chatGroupId, $"Spawn: '{sp.Name}' ({PrettyFormat(sp.Location)}) ABS:({sp.Location})", EnumChatType.CommandSuccess);
		}
		}
		}

		private void Toggle(bool state)
		{
		this.CachedConfiguration.VariableSpawnpoints = state;
		}

		private void ResetPlayer(string name)
		{
		var byPlayer = ServerAPI.Server.Players.SingleOrDefault(plr => plr.PlayerName == name);
		if (byPlayer != null && this.Spawnpoints != null && this.Spawnpoints.Count > 0) 
			{
			byPlayer.ClearSpawnPosition( );

			var serverMain = ServerAPI.World as ServerMain;
			var servPlayer = byPlayer as ServerPlayer;
			var pdm = ServerAPI.PlayerData as PlayerDataManager;
			var swpd = servPlayer.WorldData as ServerWorldPlayerData;
			//swpd.Init(serverMain);
			pdm.PlayerDataByUid.Remove(byPlayer.PlayerUID);
			pdm.WorldDataByUID.Remove(byPlayer.PlayerUID);

			//pdm.GetOrCreateServerPlayerData(byPlayer.PlayerUID);
			}
		}

		#region Events

		private void ReloadSpawnpoints( )
		{
		var spawnsBytes = ServerAPI.WorldManager.SaveGame.GetData(_spawnsDataKey);

		if (spawnsBytes != null) {
		this.Spawnpoints = SerializerUtil.Deserialize<List<Spawnpoint>>(spawnsBytes);
		Logger.Notification("Loaded Variable-Spawns # ({0}) STATE: {1}", Spawnpoints.Count, (this.CachedConfiguration.VariableSpawnpoints ? "Enabled" : "Disabled"));
		}
		else {
		this.Spawnpoints = new List<Spawnpoint>( );

		var emptySpawnsList = SerializerUtil.Serialize(this.Spawnpoints);

		ServerAPI.WorldManager.SaveGame.StoreData(_spawnsDataKey, emptySpawnsList);

		Logger.Notification("Created (empty) Spawnpoints Data");
		}

		ChangeDefaultRoleSpawnPoint( );
		}

		private void PersistSpawnpoints( )
		{
		if (this.Spawnpoints != null && this.Spawnpoints.Count > 0) {
		var spawnDatas = SerializerUtil.Serialize(this.Spawnpoints);
		ServerAPI.WorldManager.SaveGame.StoreData(_spawnsDataKey, spawnDatas);
		Logger.Notification("Persisted Spawnpoints Data");
		}

		}

		private void RotateSpawnPoint( IServerPlayer byPlayer)
		{			
		if (this.CachedConfiguration.VariableSpawnpoints && this.Spawnpoints.Count > 0) {
				
			var swpd = byPlayer.WorldData as ServerWorldPlayerData;
			if (swpd != null && (swpd.SpawnPosition == null || KnownSpawn(swpd.SpawnPosition))) 
			{
			RandomizeSpawnpoint(byPlayer);
			}
		}
		}

		private void PlayerDied(IServerPlayer byPlayer, DamageSource damageSource)
		{
		if (this.CachedConfiguration.VariableSpawnpoints && this.Spawnpoints.Count > 0) {

			var swpd = byPlayer.WorldData as ServerWorldPlayerData;
			if (swpd != null && (swpd.SpawnPosition == null || KnownSpawn(swpd.SpawnPosition))) {
			RandomizeSpawnpoint(byPlayer);
			}
		}
		}

		private void ChangeDefaultRoleSpawnPoint( )
		{
		if (this.CachedConfiguration.VariableSpawnpoints && this.Spawnpoints.Count > 0) {

		var randSp = this.Spawnpoints[random.Next(0, Spawnpoints.Count)];

		string roleCode = CachedConfiguration.PlayerRoleNormal; //ServerAPI.Server.Config.DefaultRoleCode;
		var serverPlayerRole = ServerAPI.Server.Config.Roles.Single(pr => pr.Code == roleCode) as PlayerRole;

		#if DEBUG
		Logger.VerboseDebug($"Changing ({serverPlayerRole.Name}) to spawn '{randSp.Name}' @ {randSp.Location}");
		#endif
		serverPlayerRole.DefaultSpawn = randSp.Location;		
		}

		}

		#endregion


		internal void RandomizeSpawnpoint(IServerPlayer byPlayer)
		{
		var randSp = this.Spawnpoints[random.Next(0, Spawnpoints.Count)];
		byPlayer.SetSpawnPosition(randSp.Location);
		#if DEBUG
		Logger.VerboseDebug($"Changed spawn Pos: '{randSp.Name}' @ {randSp.Location} for {byPlayer.PlayerName}");
		#endif
		}

		internal bool KnownSpawn(PlayerSpawnPos aSpawn)
		{
		if (this.Spawnpoints.Count > 0) {
		return this.Spawnpoints.Any(vsps => vsps.Location.x == aSpawn.x 
				                    && vsps.Location.y == aSpawn.y 
				                    && vsps.Location.z == aSpawn.z);
		}

		return false;
		}

		internal Vec3i RelativeToAbsolute(Vec3i location)
		{
		var start = ServerAPI.World.DefaultSpawnPosition.AsBlockPos;
		return new Vec3i( location.X + start.X, location.Y, location.Z + start.Z);
		}

		private Vec3i AbsoluteToRelative(Vec3i location)
		{
		var start = ServerAPI.World.DefaultSpawnPosition.AsBlockPos;
		return new Vec3i(start.X - location.X, location.Y, start.Z - location.Z);
		}

		internal string PrettyFormat(PlayerSpawnPos spawnPos)
		{
		var start = ServerAPI.World.DefaultSpawnPosition.AsBlockPos;

		return String.Format
        (
		@"{0:D}, {1:D}, {2:D}",
    	spawnPos.x - start.X,
        spawnPos.y - start.Y,
        spawnPos.z - start.Z
		);
				
		}
	}
}


