#include <node.h>
#include <v8.h>
#include "SDL.h"
#include "SDL_mixer.h"

using namespace v8;

#define MAX_PATH 512

// SDLの初期化を行う関数
Handle<Value> open(const Arguments& args) {
  HandleScope scope;

  // SDLの初期化
  SDL_Init(SDL_INIT_AUDIO);
  // オーディオミキサーを開く
  int ret = Mix_OpenAudio(44100, MIX_DEFAULT_FORMAT, 2 ,1024);

  // Mix_OpenAudio関数の戻り値を返す
  return scope.Close(Number::New(ret));
}


// WAVEファイルを再生する関数
Handle<Value> play(const Arguments& args) {
  HandleScope scope;

  // 引数の数をチェックする
  if(args.Length() < 3) {
    // TypeError例外を発生させる
    ThrowException(Exception::TypeError(String::New("Wrong number of arguments")));
    // undefinedを返す
    return scope.Close(Undefined());
  };

  // 引数の型をチェックする
  // 1つめの引数は文字列、2つめと3つめは数値である必要がある
  if(!args[0]->IsString()
     && !args[1]->IsNumber()
     && !args[2]->IsNumber()) {
    // TypeError例外を発生させる
    ThrowException(Exception::TypeError(String::New("Wrong arguments")));
    // undefinedを返す
    return scope.Close(Undefined());
  }

  // 変数をJavaScriptオブジェクト形式からC/C++のネイティブ形式に変換する
  char filename[MAX_PATH];
  // 1つめの引数をUTF-8形式の文字列に変換し、filename変数に書き込む
  args[0]->ToString()->WriteUtf8(filename, MAX_PATH - 1);

  // 2つめと3つめの引数をint型に変換する
  int loops = args[1]->NumberValue();
  int ticks = args[2]->NumberValue();

  // WAVEファイルをロードする
  Mix_Chunk *chunk;
  chunk = Mix_LoadWAV(filename);

  // ロードに成功したらWAVEファイルを再生する
  if(chunk) {
    Mix_PlayChannelTimed(-1, chunk, loops, ticks);
    // ticks引数で指定された時間待機する
    SDL_Delay(ticks);
  }
  // ロードしたWAVEファイルデータを解放する
  Mix_FreeChunk(chunk);

  // undefinedを返す
  return scope.Close(Undefined());
}


// SDLの後処理をする関数
Handle<Value> close(const Arguments& args) {
  HandleScope scope;

  // 音声再生を停止する
  Mix_HaltMusic();
  // オーディオミキサーをクローズする
  Mix_CloseAudio();
  // SDLの後処理を実行する
  SDL_Quit();

  // undefinedを返す
  return scope.Close(Undefined());
}

// C++関数を外部からJavaScriptの関数として呼び出せるようにする
void init(Handle<Object> target) {
  // open関数を外部から呼び出せるようにする
  target->Set(String::NewSymbol("open"),
      FunctionTemplate::New(open)->GetFunction());
  // play関数を外部から呼び出せるようにする
  target->Set(String::NewSymbol("play"),
      FunctionTemplate::New(play)->GetFunction());
  // close関数を外部から呼び出せるようにする
  target->Set(String::NewSymbol("close"),
      FunctionTemplate::New(close)->GetFunction());
}

// アドオンモジュールの名前とロード時に実行する関数を指定するマクロ
NODE_MODULE(waveplayer, init)

