/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include "sl_s_parse.h"
#include "yyy_test.h"

/*****************************************************************************/

#define UTF8_INVALID_CODEPOINT (char)0xEF, (char)0xBF, (char)0xBD

/*****************************************************************************/

static const char src_atom_1[] = " test ";
static const char exp_atom_1[] = "test";
static int TestAtom1(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_1,
        &index,
        sizeof(src_atom_1) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_1) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_1);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_1,
        &index,
        sizeof(src_atom_1) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_atom_2[] = " \"Quoted Test\" ";
static const char exp_atom_2[] = "Quoted Test";
static int TestAtom2(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_2,
        &index,
        sizeof(src_atom_2) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_2) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_2);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_2,
        &index,
        sizeof(src_atom_2) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_atom_3[] = " \"Escaped\\\"Test\" ";
static const char exp_atom_3[] = "Escaped\"Test";
static int TestAtom3(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_3,
        &index,
        sizeof(src_atom_3) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_3) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_3);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_3,
        &index,
        sizeof(src_atom_3) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_atom_4[] = " Aleph Zeta NULL ";
static const char exp_atom_4_1[] = "Aleph";
static const char exp_atom_4_2[] = "Zeta";
static const char exp_atom_4_3[] = "NULL";
static int TestAtom4(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_4,
        &index,
        sizeof(src_atom_4) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_4_1) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_4_1);
    SL_S_FREE_ATOM(atom);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_4,
        &index,
        sizeof(src_atom_4) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_4_2) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_4_2);
    SL_S_FREE_ATOM(atom);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_4,
        &index,
        sizeof(src_atom_4) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_4_3) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_4_3);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_4,
        &index,
        sizeof(src_atom_4) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

#define ATOM_UTF8_1 \
    (char)0x41, \
    (char)0xE2, \
    (char)0x89, \
    (char)0xA2, \
    (char)0xCE, \
    (char)0x91, \
    '.', \
    'x'
static const char src_atom_utf8_1[] = {
    ' ', ' ', ATOM_UTF8_1, ' ', 'z', 'i', 'o', ' ', 0
};
static const char exp_atom_utf8_1_1[] = { ATOM_UTF8_1, 0 };
static const char exp_atom_utf8_1_2[] = "zio";

static int TestAtomUTF8_1(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_1,
        &index,
        sizeof(src_atom_utf8_1) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_utf8_1_1) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_utf8_1_1);
    SL_S_FREE_ATOM(atom);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_1,
        &index,
        sizeof(src_atom_utf8_1) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_utf8_1_2) - 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_utf8_1_2);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_utf8_1,
        &index,
        sizeof(src_atom_utf8_1) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

/* ^Zarya (tm) */
static const char src_atom_utf8_2[] = {
    '\t', (char)0x8E, 'a', 'r', 'y', 'a', (char)0x99, ' ', 0
};
static const char exp_atom_utf8_2[2][16] = {
    /* No-UTF8 encoding. */
    { (char)0x8E, 'a', 'r', 'y', 'a', (char)0x99, 0 },
    { UTF8_INVALID_CODEPOINT, 'a', 'r', 'y', 'a', UTF8_INVALID_CODEPOINT, 0 }
}; 
static int TestAtomUTF8_2(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    const char *exp;
    unsigned exp_len;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_TRUE(sl_s_unicode_enabled == 0 || sl_s_unicode_enabled == 1);
    exp = exp_atom_utf8_2[sl_s_unicode_enabled];
    exp_len = strlen(exp);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_2,
        &index,
        sizeof(src_atom_utf8_2) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, exp_len);
    if(SUCCESS_INDICATOR)
        YYY_EXPECT_STR_EQ_N(atom->text, exp, atom->len);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_utf8_2,
        &index,
        sizeof(src_atom_utf8_2) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

/* X(tm)N Zed ;LOL\nYIPP */
static const char src_atom_utf8_3[] = {
    ' ', 'X', (char)0x99, 'N', ' ', ' ', 'Z', 'e', 'd', ';', 'L', 'O', 'L', '\n', 'Y', 'I', 'P', 'P', 0
};
static const char exp_atom_utf8_3_1[2][16] = {
    /* No-UTF8 encoding. */
    { 'X', (char)0x99, 'N', 0 },
    { 'X', UTF8_INVALID_CODEPOINT, 'N', 0 }
};
static const char exp_atom_utf8_3_2[] = "Zed";
static const char exp_atom_utf8_3_3[] = "YIPP";

static int TestAtomUTF8_3(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    const char *exp;
    unsigned exp_len;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_TRUE(sl_s_unicode_enabled == 0 || sl_s_unicode_enabled == 1);
    exp = exp_atom_utf8_3_1[sl_s_unicode_enabled];
    exp_len = strlen(exp);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_3,
        &index,
        sizeof(src_atom_utf8_3) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, exp_len);
    if(SUCCESS_INDICATOR)
        YYY_EXPECT_STR_EQ_N(atom->text, exp, atom->len);
    SL_S_FREE_ATOM(atom);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_3,
        &index,
        sizeof(src_atom_utf8_3) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_utf8_3_2)-1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_utf8_3_2);
    SL_S_FREE_ATOM(atom);
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_utf8_3,
        &index,
        sizeof(src_atom_utf8_3) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(atom->len, sizeof(exp_atom_utf8_3_3)-1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, exp_atom_utf8_3_3);
    SL_S_FREE_ATOM(atom);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_atom_utf8_3,
        &index,
        sizeof(src_atom_utf8_3) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_empty_list1[] = "()";

static int TestAtomEmptyList1(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_EXPECT_INT_EQ(SL_S_ParseValue(src_empty_list1,
        &index,
        sizeof(src_empty_list1) - 1,
        &value), 0);
    YYY_EXPECT_TRUE(SL_S_IS_NIL(value));
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_empty_list1,
        &index,
        sizeof(src_empty_list1) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_empty_list2[] = "( )";

static int TestAtomEmptyList2(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_EXPECT_INT_EQ(SL_S_ParseValue(src_empty_list2,
        &index,
        sizeof(src_empty_list2) - 1,
        &value), 0);
    YYY_EXPECT_TRUE(SL_S_IS_NIL(value));
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_empty_list2,
        &index,
        sizeof(src_empty_list2) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_empty_list3[] = "(   )";

static int TestAtomEmptyList3(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_EXPECT_INT_EQ(SL_S_ParseValue(src_empty_list3,
        &index,
        sizeof(src_empty_list3) - 1,
        &value), 0);
    YYY_EXPECT_TRUE(SL_S_IS_NIL(value));
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_empty_list3,
        &index,
        sizeof(src_empty_list3) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_list1[] = "(a)";

static int TestAtomList1(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_List *list;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_list1,
        &index,
        sizeof(src_list1) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_LIST(value));
    list = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_TRUE(SL_S_IS_NIL(list->tail));
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(list->head));
    
    atom = SL_S_PTR_FROM_TAG(list->head);
    
    YYY_EXPECT_INT_EQ(atom->len, 1);
    YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, "a");
    SL_S_FREE_LIST(list);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_list1,
        &index,
        sizeof(src_list1) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_list2[] = "(abc d99 \n0xEF g-g)";
#define TEST_LIST2_LEN 4
static const char exp_list2[TEST_LIST2_LEN][8] = {
    "abc",
    "d99",
    "0xEF",
    "g-g"
};

static int TestAtomList2(void){
    int SUCCESS_INDICATOR;
    unsigned index, i;
    void *value;
    struct SL_S_List *list, *iter;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_list2,
        &index,
        sizeof(src_list2) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_LIST(value));
    iter = list = SL_S_PTR_FROM_TAG(value);
    
    for(i = 0; i < TEST_LIST2_LEN; i++){
        YYY_EXPECT_FALSE(SL_S_IS_NIL(iter));
        if(SL_S_IS_NIL(iter))
            break;
        
        YYY_ASSERT_TRUE(SL_S_IS_ATOM(iter->head));
        
        atom = SL_S_PTR_FROM_TAG(iter->head);
        
        YYY_EXPECT_INT_EQ(atom->len, strlen(exp_list2[i]));
        if(atom->len == strlen(exp_list2[i]))
            YYY_EXPECT_STR_EQ_N(atom->text, exp_list2[i], atom->len);
        iter = iter->tail;
    }
    
    YYY_EXPECT_TRUE(SL_S_IS_NIL(iter));
    SL_S_FREE_LIST(list);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_list2,
        &index,
        sizeof(src_list2) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_list3[] = "(abc(1 2 99 5)0xEF () ( x ) g-g)";
#define TEST_LIST3_1_LEN 6
static const char *const exp_list3_1[TEST_LIST3_1_LEN] = {
    exp_list2[0],
    NULL,
    exp_list2[2],
    NULL,
    NULL,
    exp_list2[3]
};

#define TEST_LIST3_2_LEN 4
static const char exp_list3_2[TEST_LIST3_2_LEN][4] = {
    "1",
    "2",
    "99",
    "5"
};

static int TestAtomList3(void){
    int SUCCESS_INDICATOR;
    unsigned index, i, e;
    void *value;
    struct SL_S_List *list, *iter1, *iter2;
    struct SL_S_Atom *atom;
    
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_list3,
        &index,
        sizeof(src_list3) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_LIST(value));
    iter1 = list = SL_S_PTR_FROM_TAG(value);
    
    e = 0; /* e is persistent */
    for(i = 0; i < TEST_LIST3_1_LEN; i++){
        YYY_EXPECT_FALSE(SL_S_IS_NIL(iter1));
        if(SL_S_IS_NIL(iter1))
            break;
        
        if(exp_list3_1[i] == NULL){
            if(e == TEST_LIST3_2_LEN){
                /* Expect an empty list */
                YYY_EXPECT_TRUE(SL_S_IS_NIL(iter1->head));
                e++;
            }
            else if(e > TEST_LIST3_2_LEN){
                YYY_ASSERT_TRUE(SL_S_IS_LIST(iter1->head));
                iter2 = SL_S_PTR_FROM_TAG(iter1->head);
                YYY_EXPECT_INT_EQ(e, TEST_LIST3_2_LEN + 1);
                YYY_EXPECT_TRUE(SL_S_IS_NIL(iter2->tail));
                
                /* Expect a list of only the atom 'x' */
                YYY_ASSERT_TRUE(SL_S_IS_ATOM(iter2->head));
                atom = SL_S_PTR_FROM_TAG(iter2->head);    
                YYY_EXPECT_INT_EQ(atom->len, 1);
                YYY_EXPECT_STR_EQ_LITERAL_N(atom->text, atom->len, "x");
            }
            else{
                YYY_ASSERT_TRUE(SL_S_IS_LIST(iter1->head));
                iter2 = SL_S_PTR_FROM_TAG(iter1->head);
                for(e = 0; e < TEST_LIST3_2_LEN; e++){
                    YYY_ASSERT_FALSE(SL_S_IS_NIL(iter2));
                    YYY_ASSERT_TRUE(SL_S_IS_ATOM(iter2->head));
                    atom = SL_S_PTR_FROM_TAG(iter2->head);    
                    YYY_EXPECT_INT_EQ(atom->len, strlen(exp_list3_2[e]));
                    if(atom->len == strlen(exp_list3_2[e]))
                        YYY_EXPECT_STR_EQ_N(atom->text, exp_list3_2[e], atom->len);
                    iter2 = iter2->tail;
                }
                YYY_EXPECT_TRUE(SL_S_IS_NIL(iter2));
            }
        }
        else{
            YYY_ASSERT_TRUE(SL_S_IS_ATOM(iter1->head));
            atom = SL_S_PTR_FROM_TAG(iter1->head);
            
            YYY_EXPECT_INT_EQ(atom->len, strlen(exp_list3_1[i]));
            if(atom->len == strlen(exp_list3_1[i]))
                YYY_EXPECT_STR_EQ_N(atom->text, exp_list3_1[i], atom->len);
        }
        iter1 = iter1->tail;
    }
    
    YYY_EXPECT_TRUE(SL_S_IS_NIL(iter1));
    SL_S_FREE_LIST(list);
    
    YYY_EXPECT_INT_NOT_EQ(SL_S_ParseValue(src_list3,
        &index,
        sizeof(src_list3) - 1,
        &value), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestLineNumbers1(void){
    int SUCCESS_INDICATOR;
    unsigned index;
    void *value;
    struct SL_S_Atom *atom;
    
    if(sl_s_parse_info_enabled == 0){
        puts("Skipping disabled line number test");
        return 1;
    }
    SUCCESS_INDICATOR = 1;
    index = 0;
    
    YYY_ASSERT_INT_EQ(SL_S_ParseValue(src_atom_1,
        &index,
        sizeof(src_atom_1) - 1,
        &value), 0);
    YYY_ASSERT_TRUE(SL_S_IS_ATOM(value));
    atom = SL_S_PTR_FROM_TAG(value);
    
    YYY_EXPECT_INT_EQ(SL_S_LINE(atom), 1);
    SL_S_FREE_ATOM(atom);
    
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const char src_line_number_test2[] = 
"; \n"
"a b\n"
"c (x\n"
" y z\n"
" (1 2 3)) d;\n"
";  \n"
"e\n";

static const unsigned char exp_line_number_test2[] = {
    2, 2, 3, 3, 3, 4, 4, 5, 5, 5, 5, 5, 7, 0
};
struct TestLineNumbersData{
    const unsigned char *exp;
    int success_indicator;
};

static void test_line_numbers_func(const void *value, void *arg){
    struct TestLineNumbersData *data = arg;
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_NOT_EQ(data->exp[0], 0);
    if(data->exp[0] != 0){
        YYY_EXPECT_INT_EQ(SL_S_LINE(value), *(data->exp));
        data->exp++;
    }
    if(SUCCESS_INDICATOR == 0)
        data->success_indicator = 0;
    if(SL_S_IS_LIST(value))
        SL_S_ForEach(test_line_numbers_func, SL_S_PTR_FROM_TAG(value), arg);
}

static int TestLineNumbers2(void){
    void *value;
    unsigned index;
    struct TestLineNumbersData data;

    if(sl_s_parse_info_enabled == 0){
        puts("Skipping disabled line number test");
        return 1;
    }
    data.exp = exp_line_number_test2;
    data.success_indicator = 1;
    

    while(SL_S_ParseValue(src_line_number_test2,
        &index,
        sizeof(src_line_number_test2) - 1,
        &value) == 0){
        
        test_line_numbers_func(value, &data);
    }
    YYY_ASSERT_INT_EQ(data.exp[0], 0);
    return data.success_indicator;
}

/*****************************************************************************/

static struct YYY_Test sl_s_parse_tests[] = {
    YYY_TEST(TestAtom1),
    YYY_TEST(TestAtom2),
    YYY_TEST(TestAtom3),
    YYY_TEST(TestAtom4),
    YYY_TEST(TestAtomUTF8_1),
    YYY_TEST(TestAtomUTF8_2),
    YYY_TEST(TestAtomUTF8_3),
    YYY_TEST(TestAtomEmptyList1),
    YYY_TEST(TestAtomEmptyList2),
    YYY_TEST(TestAtomEmptyList3),
    YYY_TEST(TestAtomList1),
    YYY_TEST(TestAtomList2),
    YYY_TEST(TestAtomList3),
    YYY_TEST(TestLineNumbers1),
    YYY_TEST(TestLineNumbers2)
};

YYY_TEST_FUNCTION(ParseTest, sl_s_parse_tests, "ParseTest")

int main(int argc, char *argv[]){
    int ok, i;
    struct YYY_Test *target;
    ok = 0;
    if(argc > 1){
        for(i = 1; i < argc; i++){
            YYY_FIND_TEST_FUNC(target, argv[i], sl_s_parse_tests);
            if(target == NULL){
                fputs("Could not find test ", stderr);
                fputs(argv[i], stderr);
                fputc('\n', stderr);
            }
            else{
                YYY_RUN_TEST(*target, ok, "ParseTest");
            }
        }
    }
    else{
        YYY_RUN_TEST_SUITE(ParseTest, ok, "ParseTest");
    }
    return ok;
}
