/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include "sl_s_parse.h"
#include "yyy_test.h"

/* Get offsetof */
#include <stddef.h>

/* Just for nice housekeeping, keep the other license code in another file. */
#include "test_memcpy.c"

/*****************************************************************************/

#define TEST_SL_COMPARE(GREAT, LESS) do{ \
    YYY_EXPECT_INT_LT(SL_S_Compare(LESS, GREAT), 0); \
    YYY_EXPECT_INT_GT(SL_S_Compare(GREAT, LESS), 0); \
}while(0)

/*****************************************************************************/

#ifndef __VBCC__

static int TestLayout(void){
    int SUCCESS_INDICATOR = 1;
    /* Not really a unit test, this just checks that the layout of the structs
     * meets our requirements. */
    
    YYY_EXPECT_INT_EQ(
        offsetof(struct SL_S_Ref, ref),
        offsetof(struct SL_S_List, ref));
    YYY_EXPECT_INT_EQ(
        offsetof(struct SL_S_Ref, ref),
        offsetof(struct SL_S_Atom, ref));
#ifndef SL_S_NO_PARSE_INFO
    YYY_EXPECT_INT_EQ(
        offsetof(struct SL_S_Line, line),
        offsetof(struct SL_S_List, line));
    YYY_EXPECT_INT_EQ(
        offsetof(struct SL_S_Line, line),
        offsetof(struct SL_S_Atom, line));
#endif
    return SUCCESS_INDICATOR;
}

#endif

/*****************************************************************************/

static int TestLengthEmpty(void){
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_EQ(SL_S_Length(SL_S_NIL), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestLengthOne(void){
    struct SL_S_List base;
    int SUCCESS_INDICATOR = 1;
    base.head = SL_S_NIL;
    base.tail = SL_S_NIL;
    YYY_EXPECT_INT_EQ(SL_S_Length(&base), 1);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestLengthMany(void){
    struct SL_S_List a, b, c, d, e;
    int SUCCESS_INDICATOR = 1;
    a.head = SL_S_NIL;
    a.tail = &b;
    b.head = SL_S_NIL;
    b.tail = &c;
    c.head = SL_S_NIL;
    c.tail = &d;
    d.head = SL_S_NIL;
    d.tail = &e;
    e.head = SL_S_NIL;
    e.tail = SL_S_NIL;
    YYY_EXPECT_INT_EQ(SL_S_Length(&a), 5);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestMemcpy1(void){
    return TestMemCpy(0);
}


/*****************************************************************************/

static int TestMemcpy2(void){
    return TestMemCpy(292312);
}


/*****************************************************************************/

static int TestMemcpy3(void){
    return TestMemCpy(5876309);
}

/*****************************************************************************/

static int TestCompareAtomSame(void){
    static const struct SL_S_Atom a = SL_S_STATIC_ATOM("a");
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_EQ(SL_S_Compare(SL_S_MK_ATOM(&a), SL_S_MK_ATOM(&a)), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareAtomEquivalent(void){
    static const struct SL_S_Atom a = SL_S_STATIC_ATOM("Zenith");
    static const struct SL_S_Atom b = SL_S_STATIC_ATOM("Zenith");
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_EQ(SL_S_Compare(SL_S_MK_ATOM(&a), SL_S_MK_ATOM(&b)), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareAtomDifferent(void){
    static const struct SL_S_Atom a = SL_S_STATIC_ATOM("Zenith");
    static const struct SL_S_Atom b = SL_S_STATIC_ATOM("Matrix");
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_NOT_EQ(SL_S_Compare(SL_S_MK_ATOM(&a), SL_S_MK_ATOM(&b)), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareNil(void){
    int SUCCESS_INDICATOR = 1;
    YYY_EXPECT_INT_EQ(SL_S_Compare(SL_S_NIL, SL_S_NIL), 0);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareNilAtom(void){
    int SUCCESS_INDICATOR = 1;
    static const struct SL_S_Atom a = SL_S_STATIC_ATOM("a");
    TEST_SL_COMPARE(SL_S_MK_ATOM(&a), SL_S_NIL);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareNilList(void){
    int SUCCESS_INDICATOR = 1;
    struct SL_S_List list;
    list.head = SL_S_NIL;
    list.tail = SL_S_NIL;
    TEST_SL_COMPARE(SL_S_MK_LIST(&list), SL_S_NIL);
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareAtomList(void){
    int SUCCESS_INDICATOR = 1;
    static const struct SL_S_Atom a = SL_S_STATIC_ATOM("a");
    struct SL_S_List list;
    list.head = SL_S_NIL;
    list.tail = SL_S_NIL;
    TEST_SL_COMPARE(SL_S_MK_ATOM(&a), SL_S_MK_LIST(&list));
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static int TestCompareListSame(void){
    int SUCCESS_INDICATOR = 1;
    struct SL_S_List list;
    list.head = SL_S_NIL;
    list.tail = SL_S_NIL;
    YYY_EXPECT_INT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&list), SL_S_MK_LIST(&list)), 0);
    return SUCCESS_INDICATOR;
}


/*****************************************************************************/

static int TestCompareLists1(void){
    int SUCCESS_INDICATOR = 1;
    struct SL_S_List a, a_tail, b, b_tail;
    
    static const struct SL_S_Atom a1 = SL_S_STATIC_ATOM("Zenith");
    static const struct SL_S_Atom a2 = SL_S_STATIC_ATOM("Matrix");
    
    static const struct SL_S_Atom b1 = SL_S_STATIC_ATOM("Zenith");
    static const struct SL_S_Atom b2 = SL_S_STATIC_ATOM("Matrix");
    
    a.head = SL_S_MK_ATOM(&a1);
    a.tail = &a_tail;
    a_tail.head = SL_S_MK_ATOM(&a2);
    a_tail.tail = SL_S_NIL;
    
    b.head = SL_S_MK_ATOM(&b1);
    b.tail = &b_tail;
    b_tail.head = SL_S_MK_ATOM(&b2);
    b_tail.tail = SL_S_NIL;
    
    YYY_EXPECT_INT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&a), SL_S_MK_LIST(&b)), 0);
    
    YYY_EXPECT_INT_NOT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&a_tail), SL_S_MK_LIST(&a)), 0);
        
    YYY_EXPECT_INT_NOT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&a_tail), SL_S_MK_LIST(&b)), 0);
        
    YYY_EXPECT_INT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&a_tail), SL_S_MK_LIST(&b_tail)), 0);
    
    YYY_EXPECT_INT_NOT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&b_tail), SL_S_MK_LIST(&a)), 0);
        
    YYY_EXPECT_INT_NOT_EQ(
        SL_S_Compare(SL_S_MK_LIST(&b_tail), SL_S_MK_LIST(&b)), 0);
    
    return SUCCESS_INDICATOR;
}

/*****************************************************************************/

static const struct YYY_Test sl_s_parse_tests[] = {
#ifndef __VBCC__
    YYY_TEST(TestLayout),
#endif
    YYY_TEST(TestLengthEmpty),
    YYY_TEST(TestLengthOne),
    YYY_TEST(TestLengthMany),
    YYY_TEST(TestMemcpy1),
    YYY_TEST(TestMemcpy2),
    YYY_TEST(TestMemcpy3),
    YYY_TEST(TestCompareAtomSame),
    YYY_TEST(TestCompareAtomEquivalent),
    YYY_TEST(TestCompareAtomDifferent),
    YYY_TEST(TestCompareNil),
    YYY_TEST(TestCompareNilAtom),
    YYY_TEST(TestCompareNilList),
    YYY_TEST(TestCompareAtomList),
    YYY_TEST(TestCompareListSame),
    YYY_TEST(TestCompareLists1)
};

YYY_TEST_FUNCTION(BaseTest, sl_s_parse_tests, "BaseTest")

int main(void){
    int ok;
    ok = 0;
    YYY_RUN_TEST_SUITE(BaseTest, ok, "BaseTest");
    return ok;
}
