/*
 * Copyright (c) 2011 ARM Ltd
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the company may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ARM LTD ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL ARM LTD BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Modified by AlaskanEmily for use in Sapphire Lisp */

#ifndef BUFF_SIZE
#define BUFF_SIZE 1024
#endif

#ifndef START_COPY
#define START_COPY 256
#endif

#ifndef MAX_BLOCK_SIZE
#define MAX_BLOCK_SIZE 128
#endif

#ifndef MAX_OFFSET
#define MAX_OFFSET 3
#endif

#if (START_COPY + MAX_OFFSET + MAX_BLOCK_SIZE >= BUFF_SIZE)
#error "Buffer overrun: START_COPY + MAX_OFFSET + MAX_BLOCK_SIZE >= BUFF_SIZE."
#endif

#include "sl_s.h"
#include "yyy_test.h"

static int TestMemCpy(int seed){
    /* Allocate buffers to read and write from.  */
    unsigned char src[BUFF_SIZE], dest[BUFF_SIZE], backup_src[BUFF_SIZE];
    int SUCCESS_INDICATOR = 1;
    unsigned i, j;
    unsigned xor_state1, xor_state2, xor_s, xor_t, sa, da, n;
    xor_state1 = seed;
    xor_state2 = ~seed;
    
    for (i = 0; i < BUFF_SIZE; i+=8){
        xor_t = xor_state1;
        xor_s = xor_state2;
        xor_t ^= (xor_t << 23);
        xor_t ^= (xor_t >> 17);
        xor_t ^= xor_s;
        xor_t ^= (xor_s >> 26);
        for(j = 0; j < 8 && i + j < BUFF_SIZE; j++){
            src[i + j] = ((xor_t + xor_s) >> (j + j + j) & 0xFF);
            backup_src[i + j] = src[i + j];
        }
        xor_state1 = xor_state2;
        xor_state2 = xor_t;
    }

    /* Make calls to memcpy with block sizes ranging between 1 and
    MAX_BLOCK_SIZE bytes, aligned and misaligned source and destination.  */
    for (sa = 0; sa <= MAX_OFFSET; sa++){
        for (da = 0; da <= MAX_OFFSET; da++){
            for (n = 1; n <= MAX_BLOCK_SIZE; n++){
                /* Zero dest so we can check it properly after the copying.  */
                for (j = 0; j < BUFF_SIZE; j++)
                    dest[j] = 0;
                
                SL_S_MemCopy(dest + START_COPY + da, src + sa, n);
                /* Check that content of the destination buffer
                is the same as the source buffer, and
                memory outside destination buffer is not modified.  */
                for (j = 0; j < BUFF_SIZE; j++){
                    if (j < START_COPY + da){
                        /*
                        print_error ("\nFailed: after memcpy of %u bytes "
                           "with src_align %u and dst_align %u, "
                           "byte %u before the start of dest is not 0.\n",
                           n, sa, da, START_COPY - j);
                        */
                        YYY_EXPECT_INT_EQ(dest[j], 0);
                    }
                    else if (j < START_COPY + da + n){
                        i = j - START_COPY - da;
                        /*
                        print_error ("\nFailed: after memcpy of %u bytes "
                            "with src_align %u and dst_align %u, "
                            "byte %u in dest and src are not the same.\n",
                            n, sa, da, i);
                        */
                        YYY_EXPECT_INT_EQ(dest[j], (src + sa)[i]);
                    }
                    else{
                        /*
                        print_error ("\nFailed: after memcpy of %u bytes "
                            "with src_align %u and dst_align %u, "
                            "byte %u after the end of dest is not 0.\n",
                            n, sa, da, j - START_COPY - da - n);
                        */
                        YYY_EXPECT_INT_EQ(dest[j], 0);
                    }
                }
                /* Check src is not modified.  */
                for (j = 0; j < BUFF_SIZE; j++){
                    YYY_EXPECT_INT_EQ(src[j], backup_src[j]);
                    /*
                    print_error("\nFailed: after memcpy of %u bytes "
                        "with src_align %u and dst_align %u, "
                        "byte %u of src is modified.\n",
                        n, sa, da, j);
                    */
                }
            }
        }
    }
    return SUCCESS_INDICATOR;
}
